;;; -*- Mode: LISP; Package: atn; Syntax: Common-lisp; -*-
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   atn.main.cl
;;; Short Desc:  Main ATN network-traversal routines
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Mike Lenz
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================

(in-package :atn)

;;
;; M. Lenz
;; pAIL ATN module
;;
;; Main ATN network-traversal routines
;; 01-8-91
;;

;;
;; CURRENT DEFAULT ACTIONS
;;   (cat 'N 3)      --> (cat 'N 3 (list 'N *))
;;   (push 'NP 5)    --> (push 'NP 5 (list *))
;;   (pop)           --> (pop (cons 'NP *))
;;

;;
;; NETWORK FORMAT:
;;   (<net>*) where <net> is
;;   (network-name
;;     [(regs <register-list>)]
;;     [(init <initial-action>)]
;;     (node-name <arc>*)*
;;   )
;;
;; where <arc> is one of
;;   (tst <target-node> [<test> <action>])
;;   (jump <target-node> [<test> <action>])
;;   (word <word-list> <target-node> [<test> <action>])
;;   (vir <label> <to-node> [<test> <action>])
;;   (cat <category-name> <target-node> [<test> <action>])
;;   (push <network-name> <target-node> [<test> <preaction> <action>])
;;   (pop [<test> <action>])
;;

;;
;; Stack elements:
;;   1. name of network being traversed (e.g. S)
;;   2. node we have just reached       (e.g. O --VP--> O)
;;   3. output string so far            (e.g. (S (NP the man)))
;;   4. action to perform upon return   (e.g. (list *))
;;   5. list of active registers
;;

;;
;; "Temporary" globals used for variable lookups from
;; inside UI module.
;;


;;
;; Sample calling routines.
;; Parameter text should be a string.
;;

(defun autoatn (text)
     (doatn (or *start-subnet* (caar *net*)) text))



(defun natn (network text)
     (setq *net* network)
     (autoatn text))



(defun doatn (start-subnet text)
     (cond ((null *net*)
               (format-display *atn-output-window* "A network has not been specified.~%"))
              ((or (null start-subnet)
                    (not (listp *net*)))
               (format-display *atn-output-window* "** Error: Bad network format.~%"))
              ((not (car *lextree*))
               (format-display *atn-output-window* "** Warning: The lexicon is empty.~%"))
              (t
                 (setq *input* text)
                 (setq *input-length* (length text))
                 (setq *input-done* nil)
                 (setq *nextpos* 0)
                 (setq *print-pretty* t)
                 (setq *show-warnings* t)
                 (setq *num-parses* 0)
                 (atn-recognize start-subnet))))



;;
;; Given a lexicon (list of entries of form (word . <feature-list>),
;; stores them in the global *lextree* via the lex-word function.
;; Converts words to lower case.
;;

(defun store-lexicon (lexicon)
      (if (listp lexicon)
         (progn
              (clear-lexicon)
              (dolist (aword lexicon)
                    (lex-string (string-downcase (format nil "~A" (car aword)))
                     (cdr aword))))
         (format-display *atn-output-window*
              "** Warning: Bad lexicon format.~%")))



;;
;; Initializes the ATN with the given subnetwork as "global"
;; and begins execution.
;;

(defun atn-recognize (subnetname)
     (format-display *atn-output-window* "~%SENTENCE: ~a~%" *input*)
     (let* ((star (read-word 0))
             (lex (lookup-string star))
             (initregisters '((HOLD nil))))
         (setf (cadar initregisters) nil)       ; why do I need this?
         (catch 'abort
             (init-network subnetname
              *nextpos*
              star
              lex
              nil
              initregisters
              nil)))
     (if *show-graphics*
        (move-cursor-to nil nil))
     (format-display *atn-output-window*"~%~@(~R~) parse~:P found.~%" *num-parses*))



;;
;; Sets up local register list and performs push arc's preactions
;; and initial actions for the given subnetwork; then continues
;; execution with the subnet's first node.
;;

(defun init-network (netname inputpos star lex preaction registers stack)
     (let* ((start-index 0)
             (output nil)
             (network (get-network netname))
             (reglist (assoc regs network))
             (initaction (assoc init network)))
         
         (if reglist
            (progn
                (dolist (reg (cdr reglist))
                     (setq registers
                          (cons (list reg nil) registers)))
                (setq start-index (1+ start-index))))
         
         (do-eval preaction star lex registers stack)
         
         (if initaction
            (progn (do-eval (cadr initaction) star lex registers stack)
                (setq start-index (1+ start-index))))
         
         (examine-node netname
          (car (nth start-index network))
          inputpos
          output
          star
          lex
          registers
          stack)))



;;
;; Register-handling functions
;;

;;
;; find-register: searches register list and the stack
;; for an entry (reg value) for given reg; returns entry
;; or nil.
;;

(defun find-register (reg)
     (let ((entry (assoc reg *registers*)))
         (if entry
            entry
            (progn
                (dolist (top-stack *stack*)
                     (setq entry (assoc reg (fifth top-stack)))
                     (if entry
                        (return-from find-register entry)))
                nil))))



;;
;; Macro setr: invoked by a setr call in an action or test
;; statement in a network.  Calls set-reg which searches the
;; current register list and stack of pushed registers for
;; the given register, then sets it to value.
;;

(defmacro setr (reg value)
     `(set-register (quote ,reg) ,value))


(defun set-register (reg value)
     (let ((entry (find-register reg)))
         (if entry
            (setf (second entry) value)
            (format-display *atn-output-window* "** Error: Register ~a not declared.~%" reg))))



;;
;; Macro sendr: Should be called only from a PUSH preaction.
;; Not executed until new subnetwork's local registers have
;; been set up; then just like setr except it does not search
;; outer scopes for named register.
;;

(defmacro sendr (reg value)
     `(let ((entry (assoc (quote ,reg) *registers*)))
          (if entry
             (setf (second entry) ,value)
             (format-display *atn-output-window*
                 "** Error: Register ~a not declared within subnetwork.~%"
                 (quote ,reg)))))



;;
;; Macro getr: Returns the value of the given register, or
;; returns an error if not found.
;;

(defmacro getr (reg)
     `(get-register (quote ,reg)))


(defun get-register (reg)
     (let ((entry (find-register reg)))
         (if entry
            (second entry)
            (format-display *atn-output-window*
                "** Error: Register ~a not defined.~%" reg))))



;;
;; getftr looks at *lex* -- the lexical entry of the last
;; word recognized -- for the feature indicated and returns
;; its value, or t if found with no value, or nil if not found.
;; 
;; OR if 2nd argument provided, it must be a register name;
;; in which case we do a getr on it to get the source lexeme.
;; Note the format is the same (i.e. any lexeme is expected
;; to be a list.)
;;
;; *lex* is a list of lists, of length either >= 1 or exactly 1.
;; The former is the case when a word is originally looked up;
;; the latter is the case after a CAT arc has been successfully
;; traversed on the word.
;; By convention, getftr returns nil for any feature lookup
;; when length of *lex* is > 1 (since the word is thus "ambiguous"
;; between two different readings).
;;

(defmacro getftr (feature &optional reg)
     `(let ((lex (if (quote ,reg) (getr ,reg) *lex*)))
          (if (> (length lex) 1)
             nil
             (dolist (testftr (cdar lex) nil)
                  (if (eql testftr (quote ,feature))
                     (return t)
                     (if (and (listp testftr)
                                 (eql (car testftr) (quote ,feature)))
                        (return (cadr testftr))))))))



;;
;; This is just like "list" except it does a getr
;; on any unquoted symbols it runs across.
;;

(defmacro build (&rest args)
     `(mapcar (quote sym-eval)
            (quote ,args)))


(defun sym-eval (arg)
     (if (symbolp arg)
        (let ((entry (find-register arg)))
            (if entry
               (return-from sym-eval (second entry)))))
     (eval arg))



;;
;; Appends the list ('<label> . <value>) to the front
;; of the global HOLD register.
;;

(defmacro hold (label value)
     `(setr HOLD (cons (cons (quote ,label)
                                       ,value)
                             (getr HOLD))))



;;
;; If the head of the HOLD register is a list of the form
;; ('<label> . <value>), removes the entry from HOLD and
;; returns <value>; otherwise returns nil.
;;

(defmacro unhold (label)
     `(let ((hold (getr HOLD)))
          (if (eql (caar hold) (quote ,label))
             (prog1 (cdar hold)
                 (setr HOLD (cdr hold))))))



;;
;; This is called with a test or action to be evaluated...
;; just sets temporary global variables containing current
;; stack and registers, to be accessed by setr and getr.
;;

(defun do-eval (form star lex registers stack)
     (setq * star)
     (setq *lex* lex)
     (setq *registers* registers)
     (setq *stack* stack)
     (eval form))



;;
;; For each of node's arcs, dispatches to proper routine by arc type
;; to traverse that arc if possible.  Also updates the graphical
;; display of current node, and halts for user input if in single
;; step mode.
;;

(defun examine-node (netname node inputpos output star lex registers stack)
     (if *show-diagnostics*
        (let ((indent (get-indent stack)))
            (format-display *atn-output-window*
                "~V@T*      ~a~%~V@TLEX    ~a~%~V@TOUTPUT ~a~%~V@TREGS   ~a~%~V@TSTACK  ~a~%"
                indent star indent lex indent output indent registers
                indent stack)))
     
     (if *show-graphics*
        (move-cursor-to netname node))
     
     (if *single-step*
             (progn
                 ;; this do-eval just sets up *, *lex*, etc. for us
                 ;; so user can see them in step mode.
                 (do-eval nil star lex registers stack)
                 (format-display *atn-output-window*
                     "-- In network ~a at node ~a, * is ~a~%"
                     netname node (print-word star))
                 (if (not (get-step-command))
                    (throw 'abort t))))
     
     (dolist (arc (arcs-from netname node))
          
          (if *show-status*
             (format-display *atn-output-window*
                 "~V@T~a node ~a: " (get-indent stack) netname node))
          ; (print (arc-type arc))
          (cond 
                   ((eq (arc-type arc) *word*)
                    (try-word-arc arc netname inputpos output star lex registers stack))
                   ((eq (arc-type arc) *jump*)
                    (try-jump-arc arc netname inputpos output star lex registers stack))
                   ((eq (arc-type arc) *tst*)
                    (try-tst-arc arc netname inputpos output star lex registers stack))
                   ((eq (arc-type arc) *vir*)
                    (try-vir-arc arc netname inputpos output star lex registers stack))
                   ((eq (arc-type arc) *cat*)
                    (try-cat-arc
                     arc netname node inputpos output star lex registers stack))
                   ((eq (arc-type arc) *push*)
                    (try-push-arc arc netname inputpos output star lex registers stack))
                   ((eq (arc-type arc) *pop*)
                    (try-pop-arc arc netname inputpos output star lex registers stack))
                   (t
                      (if *show-warnings*
                         (format-display *atn-output-window*
                             "~%** Warning: unknown arc type ~a.~%"
                             (arc-type arc)))))
          ))



;;
;; These routines all call traverse-arc if successful, which then
;; evaluates the arc's test to verify that the arc can be traversed.
;;

;;
;; Succeeds if * is a member of the arc's word list.
;;

(defun try-word-arc (arc netname inputpos output star lex registers stack)
     (let ((label (arc-label arc)))
         
         (if *show-status*
            (format-display *atn-output-window*
                "Trying word arc on ~a" (print-word star)))
         
         (if (or (equal star label)
                  (and (listp label)
                          (member star label :test #'equal)))
            (traverse-arc
             arc netname inputpos t output star star star lex registers stack)
            
            (if *show-status*
               (format-display *atn-output-window*
                   ": failed~%")))))



;;
;; Always succeeds; does not advance input.
;;

(defun try-jump-arc (arc netname inputpos output star lex registers stack)
     
     (if *show-status*
        (format-display *atn-output-window*
            "Trying jump arc on ~a" (print-word star)))
     
     (traverse-arc
      arc netname inputpos nil output nil star star lex registers stack))



;;
;; Succeeds unless the input has been exhausted (* is nil).
;;

(defun try-tst-arc (arc netname inputpos output star lex registers stack)
     
     (if *show-status*
        (format-display *atn-output-window*
            "Trying tst arc on ~a" (print-word star)))
     
     (if star
        (traverse-arc
         arc netname inputpos t output star star lex registers stack)
        (if *show-status*
           (format-display *atn-output-window*
               ": failed (* is null)~%"))))




;;
;; Attempts to pop a list of the form (<label> . <value>) from
;; the HOLD stack; if successful, * gets <value> and the
;; default action becomes (list label value).
;;

(defun try-vir-arc (arc netname inputpos output star lex registers stack)
     
     (let ((label (arc-label arc))
            (hold (getr HOLD)))
         
         (if *show-status*
            (format-display *atn-output-window*
                "Trying vir arc ~a on ~a"
                label (print-word star)))
         
         ;; *We don't call unhold here because it may return nil which
         ;; can mean either <label> was associated with nil, in which
         ;; case we should traverse the arc, or <label> did not head
         ;; the HOLD register, in which case we shouldn't.
         
         (if (eql (caar hold) label)
            (let* ((result (cdar hold))
                    (default (list label result)))
                (setr HOLD (cdr hold))
                (traverse-arc arc netname inputpos nil output default
                 result star lex registers stack))
            
            (if *show-status*
               (format-display *atn-output-window* ": failed~%")))))



;;
;; Succeeds if * is of the proper category.  If so, traverses
;; the arc once for EACH valid lexical entry of *.
;;

(defun try-cat-arc
      (arc netname curnode inputpos output star lex registers stack)
     (let* ((label (arc-label arc))
             (newlex (collect label lex))
             (entrynum 0))
         
         (dolist (onelex newlex)
              
              (if *show-status*
                 (progn (setq entrynum (1+ entrynum))
                     (if (> entrynum 1)
                        (format-display *atn-output-window*
                            "~V@T~a node ~a: "
                            (get-indent stack) netname curnode))
                     (format-display *atn-output-window*
                         "Trying cat arc ~a on ~a" label (print-word star))
                     (if (> (length newlex) 1)
                        (format-display *atn-output-window*
                            " (entry ~a)" entrynum))))
              
              (traverse-arc arc
               netname
               inputpos
               t
               output
               (list label star)
               star
               star
               (list onelex)
               registers
               stack))
         
         (if (and *show-status* (zerop entrynum))
            (progn (format-display *atn-output-window*
                           "Trying cat arc ~a on ~a: failed~%"
                           label (print-word star))))))



;;
;; Continues if the arc's test succeeds (evaluates to non-nil);
;; updates input and output tapes, and recursively calls
;; examine-node.
;;
;; Appends the result of the arc's action to global output tape.
;; If no action specified, defaultresult is appended.
;;
;; Parameter note:
;; star is the current input word, which is what a * in the test
;;   clause refers to.
;; returnstar is the 'output' of the current arc, i.e. what a *
;;   in the action clause refers to.
;;

(defun traverse-arc
      (arc netname inputpos nextwordp output defaultresult
       returnstar star lex registers stack)
     (if (do-eval (arc-test arc) star lex registers stack)
        (let* ((action (arc-action arc))
                (actionresult (if (null action)
                                        defaultresult
                                        (do-eval action returnstar lex registers stack)))
                (newoutput (if (null actionresult)
                                     output
                                     (append output (list actionresult))))
                (newstar (if nextwordp (read-word inputpos) star))
                (newlex (if nextwordp (lookup-string newstar) lex))
                (newinputpos (if nextwordp *nextpos* inputpos)))
            
            (if *show-status*
               (format-display *atn-output-window* ": OK~%"))
            
            ;; *We'd like the above to use lookup-word to prevent
            ;; pointless conversion to char list...
            
            (examine-node netname
             (arc-tgtnode arc)
             newinputpos
             newoutput
             newstar
             newlex
             registers
             stack))
        
(if *show-status*
   (format-display *atn-output-window*
       ": test failed~%"))))




;;
;; If the arc's test succeeds, initialize and begin executing
;; the pushed-to subnetwork.  Creates a new stack frame with
;; current state, including default action of (push *); action
;; will be executed inside try-pop-arc with * set to subnetwork's
;; return value.
;;

(defun try-push-arc (arc netname inputpos output star lex registers stack)
     
     (if *show-status*
        (format-display *atn-output-window*
            "Pushing to network ~a on ~a"
            (arc-label arc) (print-word star)))
     
     (if (do-eval (arc-test arc) star lex registers stack)
        (let ((label (arc-label arc))
               (postaction (arc-action arc)))
            
            (if *show-status*
               (format-display *atn-output-window* ": OK~%"))
            
            (if (null postaction)
               (setq postaction (list 'list '*)))
            
            (init-network label
             inputpos
             star
             lex
             (arc-preaction arc)
             nil
             ;; build up a new stack element
             (cons (list netname
                         (arc-tgtnode arc)
                         output
                         postaction
                         registers)
                 stack)))
        
        (if *show-status*
           (format-display *atn-output-window* ": test failed~%"))))



;;
;; A pop arc fails only if its test fails, OR if it is a pop
;; from the root subnetwork (i.e. stack is null) but there is
;; input left.  Otherwise, default action is (cons netname output);
;; * gets the result of this action when the popped action is
;; executed.
;;

(defun try-pop-arc (arc netname inputpos output star lex registers stack)
     
     (if *show-status*
        (format-display *atn-output-window*
            "Popping from ~a on ~a" netname (print-word star)))
     
     ;; "*" is initially set to the network's output SO FAR--
     ;; i.e. the current output tape, so the pop arc's action
     ;; may then refer to it.  Then * is set to the return
     ;; value of the pop arc, or to the default.
     
     (if (do-eval (arc-test arc) star lex registers stack)
        
        (if (and (null stack)
                    (not (null star)))
           (if *show-status*
              (format-display *atn-output-window*
                  ": failed (not end of input)~%"))
           
           (let* ((nextaction (arc-action arc))
                   (result (if nextaction
                                 (do-eval nextaction output lex registers stack)
                                 (cons netname output))))
               
               (if *show-status*
                  (format-display *atn-output-window*": OK~%"))
               
               (if (null stack)
                  (progn (setq *num-parses* (1+ *num-parses*))
                      (format-display *atn-output-window*
                          "~%PARSE ~a: ~a~%" *num-parses* result)
                      )
                  
                  (let ((prevregs (pop-registers stack))
                         (newstack (cdr stack)))
                      (examine-node
                       (pop-network stack)
                       (pop-node stack)
                       inputpos
                       ;; append result of the pushed action to pushed output
                       (append-list (pop-output stack)
                        (do-eval (pop-action stack)
                         result
                         lex
                         prevregs
                         newstack))
                       star
                       lex
                       prevregs
                       newstack)))))
        
        (if *show-status*
           (format-display *atn-output-window *": test failed~%"))))



;;
;; Utility routines
;;
;; Returns the list composed of members of alist whose
;; first elements are all equal to label.
;;

(defun collect (label alist)
     (remove-if-not #'(lambda (x) (eql (car x) label))
         alist))



;;
;; Tests if two words are the same.
;; This routine could be enhanced, e.g.
;;  -- to ignore the case (upper/lower) of the words;
;;  -- to search lexword's lexical entry for "alternate forms"
;; and also compare inputword to them.
;;

(defun word-equal (inputword lexword)
     (eql inputword lexword))



;;
;; The following routines determine the format
;; of the network specification.
;;

(defun arcs-from (netname node)
     (let ((arcs (cdr (assoc node (get-network netname)))))
         
         (if (and (null arcs) *show-warnings*)
            (format-display *atn-output-window* 
                "** Warning: Node ~a of network ~a does not exist or has no exiting arcs.~%"
                node netname))
         arcs))



;;
;; Access to stack elements
;; 

(defun pop-network (stack)
     (first (car stack)))

(defun pop-node (stack)
     (second (car stack)))

(defun pop-output (stack)
     (third (car stack)))

(defun pop-action (stack)
     (fourth (car stack)))

(defun pop-registers (stack)
     (fifth (car stack)))



;;
;; *Note these presume format:
;; assumes 'test' and 'action' fields of an arc are
;;   the 2nd and 3nd elements for a POP arc
;;   the 4th and 5th element for any other arc.
;;

;; An unspecified test defaults to t.

(defun arc-type (arc)
     (first arc))

(defun arc-label (arc)
     (second arc))

(defun arc-tgtnode (arc)
     (let ((type (arc-type arc)))
         (if (or (eql type *jump*) (eql type *tst*))
            (second arc)
            (third arc))))

(defun arc-test (arc)
     (let ((type (arc-type arc))
            (len (length arc)))
         (if (eql type *pop*)
            (if (< len 2) t (second arc))
            (if (or (eql type *jump*) (eql type *tst*))
               (if (< len 3) t (third arc))
               (if (< len 4) t (fourth arc))))))

(defun arc-preaction (arc)
     (fifth arc))

(defun arc-action (arc)
     (let ((type (arc-type arc)))
         (if (eql type *pop*)
            (third arc)
            (if (or (eql type *jump*) (eql type *tst*))
               (fourth arc)
               (if (eql type *push*)
                  (sixth arc)
                  (fifth arc))))))


(defun get-network (name)
     (let ((network (cdr (assoc name *net*))))
         (if (and (null network) *show-warnings*)
            (format-display *atn-output-window*
                "** Warning: Network ~a does not exist.~%" name))
         network))



;;
;; Like append but it turns its second argument into a list
;; if it's an atom, rather than signaling an error
;;

(defun append-list (list1 item)
     (if (atom item)
        (append list1 (list item))
        (append list1 item)))



(defun list-subnets ()
  (let ((return-value nil))
    (dolist (item *network*)
      (push (car item) return-value))
    (reverse return-value)))

;;

(defun get-indent (stack)
     (+ 2 (* 2 (length stack))))

;;

(defun print-word (word)
     (if word
        (format nil "~C~a~C" #\" word #\")
        (format nil "-nil-")))
