;;; -*- Mode: LISP; Package: atn; Syntax: Common-lisp; -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   atn-interface.cl
;;; Short Desc: general user interface for the  ATN tool
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.9.91 - FB
;;; Author(s):  Mike Lenz, Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Mike Lenz wrote the character oriented version of the interface
;;; Fabio Baj wrote the graphic user interface
;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================

(in-package :atn)


(defun get-new-sentence ()
     (let (new-sent aux button)
         (multiple-value-setq
               (new-sent aux button)
               (ask-user-for-string
                    "Please, enter sentence to parse:"
                    *last-sentence*
                    "OK"
                    "Cancel"))
         (if (and (string/= new-sent "") (string= button "OK"))
            (progn
                (setq *last-sentence* new-sent)
                (pushnew new-sent *sentence-list* :test #'string=)))
         (string= button "OK")))


     
(defun load-network ()
     (let ((nfile 
            (ask-user-for-existing-pathname
                        "Network to load"
                        :stream  *atn-main-window*
                        :allowed-types '(("Network Files" . "*.net")("All Files"  . "*.*"))
                        :host (concatenate 'string *pail-directory* "atn\\networks\\")
                        )))
         (when nfile
              (progn
                  (load nfile)
                  (format-display *atn-output-window*
                      "Network ~a loaded!~%~%" (file-namestring nfile))
                   (setq *start-subnet* nil)
                   (set-menu-after-load-network)
                  (setq *current-net-file* nfile)
                  (setq *net* *network*)
                   (when *show-graphics*
                          (update-graphics)
                          (process-pending-events)
                          (update-graphics))
                   *net*))))



(defun load-lexicon ()
     (let ((lfile 
            (ask-user-for-existing-pathname
                        "Lexicon to load"
                        :stream  *atn-main-window*
                        :allowed-types '(("Lexicon Files" . "*.lex")("All Files"  . "*.*"))
                        :host (concatenate 'string *pail-directory* "atn\\lexicon\\")
                        )))
         (when lfile
                (setq *current-lex-file* lfile)
                (setq *lexicon* (loadlex lfile))
                (create-lex-window *lexicon*)
                (format-display *atn-output-window*
                 "~%Lexicon ~a loaded!~%~%" (file-namestring lfile))
                (set-menu-after-load-lexicon))))

(defun new-lexicon ()
     (let ((lfile (concatenate 'string *pail-directory* "atn\\lexicon\\untitled.lex")))
         (setq *current-lex-file* lfile)
         (setq *lexicon* nil)
         (create-lex-window *lexicon*)
         (set-menu-after-load-lexicon)
         ))



(defun set-menu-after-load-network ()
      (set-menu-item-available-p
           (find-named-object ':edit-current-network *atn-network-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':parser-settings *lisp-menu-bar*) t)
      (when *lexicon*
             (set-menu-item-available-p
                   (find-named-object ':set-starting-subnet *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':parse-sentence *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-registers *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-hold *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-stack *lisp-menu-bar*) t)))
             


(defun set-menu-after-load-lexicon ()
      (set-menu-item-available-p
           (find-named-object ':save-lexicon *atn-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':new-word *atn-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':delete-word *atn-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':find-word *atn-lexicon-menu*) t)
      (when *network*
             (set-menu-item-available-p
                   (find-named-object ':set-starting-subnet *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':parse-sentence *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-registers *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-hold *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':print-stack *lisp-menu-bar*) t)))



(defun set-menu-before-load-lexicon ()
      (set-menu-item-available-p
           (find-named-object ':save-lexicon *atn-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':new-word *atn-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':delete-word *atn-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':find-word *atn-lexicon-menu*) nil)
      (when *network*
             (set-menu-item-available-p
                   (find-named-object ':set-starting-subnet *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':parse-sentence *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':parse-predefined-text *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':repeat-last-parse *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':print-registers *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':print-hold *lisp-menu-bar*) nil)
             (set-menu-item-available-p
                   (find-named-object ':print-stack *lisp-menu-bar*) nil)))



(defun parse-sentence ()
      (when (get-new-sentence)
             (repeat-last-parse)
             (set-menu-item-available-p
                   (find-named-object ':parse-predefined-text *lisp-menu-bar*) t)
             (set-menu-item-available-p
                   (find-named-object ':repeat-last-parse *lisp-menu-bar*) t)))


(defvar *select-sentence-dialog* nil)

(defun parse-predefined-text ()
      (let ((height (box-height (clipping-box *screen*)))
              (width (box-width (clipping-box *screen*))))
          (setq *select-sentence-dialog*
                (open-dialog 
                      (list
                         (make-dialog-item :widget 'static-text
                                :value "Select sentence to parse"
                                :box (make-box 110 10 290 30))
                         (make-dialog-item :widget 'single-item-list
                                :value (car *sentence-list*)
                                :name ':sentences
                                :range *sentence-list*
                                :box (make-box 10 50  390 (+ 50 (* 20 (length *sentence-list*)))))
                         (make-dialog-item :widget 'default-button
                                :title "OK"
                                :box (make-box 240 (+ 70 (* 20 (length *sentence-list*)))
                                                310 (+ 100 (* 20 (length *sentence-list*))))
                                :set-value-fn #'(lambda (item new old) (values t t)))
                         (make-dialog-item :widget 'cancel-button
                                :title "Cancel"
                                :box (make-box 320 (+ 70 (* 20 (length *sentence-list*)))
                                                390 (+ 100 (* 20 (length *sentence-list*))))
                                :set-value-fn '(lambda (item new old) (close *select-sentence-dialog*))))
                      'atn-dialog *atn-main-window* :pop-up-p t
                      :title "Sequence Selection"
                      :window-border :dialog-box
                      :user-closable nil :user-resizable nil :user-scrollable nil
                      :window-interior
                      (make-box
                             (round (- (/ width 2) 200))
                             (round (- (/ height 2) (/ (+ (* 20 (length *sentence-list*)) 110) 2)))
                             (round (+ (/ width 2) 200))
                             (round (+ (/ height 2) (/ (+ (* 20 (length *sentence-list*)) 110) 2))))))
          (pop-up-dialog *select-sentence-dialog*)
          (parse (dialog-item-value
                            (find-named-object
                               ':sentences
                               *select-sentence-dialog*)))))
                                                  
         

(defun parse (sentence)
     (if (not (string= sentence ""))
        (progn
            (setq *last-sentence* sentence)
            (repeat-last-parse))))


(defun repeat-last-parse ()
     (autoatn (string-downcase *last-sentence*)))



(defun begins-with (prefix string)
     (string= prefix string :start1 0 :end1  ( length prefix)
         :start2 0 :end2  ( length prefix)))


(defun terminates-with (substr str)
     (head-equal  (reverse (coerce str 'list))
      (reverse (coerce substr 'list))))

(defun head-equal (list sublist)
     (cond ((null sublist) t)
              ( (null list) nil)
              ((equal (car list) (car sublist)) (head-equal (cdr list)( cdr  sublist)))))



(defun print-registers (regs holdonlyp indent)
      (format-display *atn-output-window* "~%")
      (if (and (null regs) (not holdonlyp))
         (format-display *atn-output-window*
              "~&~VTRegisters: None~%" indent)
         (progn
              (format-display *atn-output-window*
                   "~&~VTRegister    Value~%" indent)
              (format-display *atn-output-window* 
                   "~VT--------    -----~%" indent)
              (if holdonlyp
                 (let ((reg (getr HOLD)))
                     (format-display *atn-output-window* 
                          "~VT~a~VT~a~%"
                          indent (car reg) (+ indent 12) (cadr reg)))
                 (dolist (reg regs)
                       (format-display *atn-output-window*  "~VT~a~VT~a~%"
                            indent (car reg) (+ indent 12) (cadr reg))))))
      (format-display *atn-output-window* "~%"))
      

;;

(defun print-stack ()
     (format-display *atn-output-window*  "~&  Top of stack~%")
     (dolist (entry *stack*)
          (format-display *atn-output-window*  "-----------------~%")
          (format-display *atn-output-window*  "  Network    ~a~%" (pop-network (list entry)))
          (format-display *atn-output-window*  "  Node       ~a~%" (pop-node (list entry)))
          (format-display *atn-output-window*  "  Output     ~a~%" (pop-output (list entry)))
          (format-display *atn-output-window*  "  Action     ~a~%" (pop-action (list entry)))
          (print-registers (pop-registers (list entry)) nil 2))
     (format-display *atn-output-window*  "-----------------~%"))

;;

(defun print-lookup (word)
     (let ((lex (if (stringp word) (lookup-string (string-downcase word))
                      (lookup-string (string-downcase (format nil "~A" word))))))
         (if lex
            (dolist (entry lex)
                 (format-display *atn-output-window*  "~&Word:      ~a~%" word)
                 (format-display *atn-output-window*  "Category:  ~a~%" (car entry))
                 (format-display *atn-output-window*  "Features:  ~a~%" (cdr entry))
                 (format-display *atn-output-window*  "--------~%"))
            (format-display *atn-output-window*  "-- No entries for ~a --~%" word))))

(defun center (but disp)
     (floor (- (width disp) (width but)) 2))



(defun get-step-command ()
     (do () (*got-step-command*)
          (process-pending-events))
     (let ((choosen-action  *got-step-command*))
         (setq *got-step-command* nil)
         (case choosen-action
             ((:next-step) t)
             ((:quit) nil)
             ((:go) (progn (set-stepper) t)))))