;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*- 
;;; 
;;; ************************************************************************ 
;;; 
;;; PORTABLE AI LAB - IDSIA LUGANO 
;;; 
;;; ************************************************************************ 
;;; 
;;; Filename:   folpars.cl 
;;; Short Desc: A PARSER for First Order Languages 
;;; Version:    1.0 
;;; Status:     Review 
;;; Last Mod:   8.9.91 - FB 
;;; Author:     Fabio Baj 
;;; 
;;; Copyright (c) 1992 Istituto D|all|e Molle (IDSIA), University of 
;;; Zurich, Swiss Federal Institute of Technology Lausanne. 
;;; 
;;; Permission is granted to any individual or institution to use, copy, 
;;; modify, and distribute this software, provided that this complete 
;;; copyright and permission notice is maintained, intact, in |all| 
;;; copies and supporting documentation. 
;;; 
;;; IDSIA provides this software "as is" without express or implied 
;;; warranty.   
;;; 
 
;;; 
;;; -------------------------------------------------------------------------- 
;;; Change History: 
;;; 
;;; 
;;; -------------------------------------------------------------------------- 
 


 
;;; ========================================================================== 
;;; PACKAGE DECLARATIONS 
;;; ========================================================================== 
 
 
(in-package :atp) 
 


;;---------------------------------------------------------- 
;; pars         : LIST-OF(TOKEN)  --->  SEXPR 
;; Behavior     : Parses sent and trasforms it into 
;;                a formula in prefix notation 
;; Side effects : None 
;; Example      : (pars '(A & B #\( x #\) ) )  = '(& A ( B x)) 
 


(defun pars (sent) 
      (let ((s (if $prolog-mode$
                      (plg-fmla sent) 
                      (fmla sent)))) 
          (cond ((null s) nil) 
                      ((null (cdr s)) (car s))))) 
     
     
    
 
 
 
;;------------------------------------------------------------------------------------ 
;; fmla         : LIST-OF(TOKEN)  --->  (SEXPR .  LIST-OF(TOKEN) ) 
;; Behavior     : Parses sent and trasforms it into 
;;                a formula in prefix notation, returned togheter with 
;;                a list of residual tokens. It implements the production rules: 
;;                   FMLA --> I-FMLA  => FMLA 
;;                   FMLA --> I-FMLA <=> FMLA 
;;                   FMLA --> I-FMLA  
;;                The computational schema is similar to |all| the following functions  
;;                so only the computed production rules will be shown. 
;; Side effects : None 
;; Example      : (pars '(A & B #\( x #\) f g h ) = ( '(& A ( B x))  f g h) 
 
 

(defun fmla (sent) 
      (let ((p1)
              (f1)) 
          (or (and (setq p1 (i-fmla sent)) 
                          (not (null (cdr p1)))   
                          (member (cadr p1) '(=> <=>) :test #'equal) 
                          (setq f1 (fmla (cddr p1))) 
                          (cons
                                (list (cadr p1) (car p1) (car f1)) 
                                (cdr f1))) 
                p1))) 
         

 
;;---------------------------------- 
;;   I-FMLA  --> PROD +  I-FMLA 
;;   I-FMLA  --> PROD  
 


(defun i-fmla (sent) 
      (let ((p1)
              (f1)) 
          (or (and (setq p1 (prod sent)) 
                          (not (null (cdr p1))) 
                          (is-ORsym (cadr p1)) 
                          (setq f1 (i-fmla (cddr p1))) 
                          (cons
                                (list '+ (car p1) (car f1)) 
                                (cdr f1))) 
                p1))) 
 


;;---------------------------------- 
;;   PROD --> TERM & PROD 
;;   PROD --> TERM  
 
(defun prod (sent) 
      (let ((t1) (p1)) 
          (or (and (setq t1 (term sent)) 
                          (not (null (cdr t1))) 
                          (is-ANDsym (cadr t1)) 
                          (setq p1 (prod (cddr t1))) 
                          (cons
                                (list '& (car t1) (car p1)) 
                                (cdr p1))) 
                t1)))               


   
;;---------------------------------- 
;;   TERM --> ( FMLA ) 
;;   TERM --> ~ TERM 
;;   TERM --> exist VAR TERM 
;;   TERM --> |all|   VAR TERM 
;;   TERM --> ATOMICF 
 
;;; (defun term (sent) 
;;;       (let ((f1) (tr1)) 
;;;           (or (and (equal '#\( (car sent)) (not (null (cdr sent))) 
;;;                           (setq f1 (fmla (cdr sent))) 
;;;                           (equal '#\) (cadr f1)) 
;;;                           (cons
;;;                                 (car f1)
;;;                                 (cddr f1))) 
;;;                 (and (is-NOTsym (car sent)) 
;;;                          (setq  tr1 (term (cdr sent))) 
;;;                          (cons
;;;                                (list '~ (car tr1)) 
;;;                                (cdr tr1))) 
;;;                 (and  
;;;                          (member (car sent) '(|exists| |all|) :test #'equal) 
;;;                          (is-variable (cadr sent)) 
;;;                          (setq tr1 (term (cddr sent))) 
;;;                          (cons
;;;                                (list (car sent) (cadr sent) (car tr1)) 
;;;                                (cdr tr1))) 
;;;                 (atomic-f sent)))) 
 


;;---------------------------------- 
;;   ATOMICF --> ( TM <postfix functor>) 
;;   ATOMICF --> ( TM <infix functor> TM) 
;;   ATOMICF --> <predicate symbol> ( TM-LIST ) 
;;   ATOMICF --> <an atom> 
 
(defun atomic-f (sent) 
      (let (ps1 tl1 tm1 tm2) 
          (or  
                (and (equal (car sent) '#\() 
                         (setq tm1 (tm-x (cdr sent))) 
                         (is-postfix-op (cadr tm1))  
                         (equal '#\) (caddr tm1)) 
                         (cons
                               (list (cadr tm1) (car tm1)) 
                               (cdddr tm1))) 
                (and (equal (car sent) '#\() 
                         (setq tm1 (tm-x (cdr sent))) 
                         (is-infix-op (cadr tm1)) 
                         (setq tm2 (tm-x (cddr tm1))) 
                         (equal '#\) (cadr tm2)) 
                         (cons
                               (list (mk-sym (cadr tm1))(car tm1)(car tm2)) 
                               (cddr tm2))) 
                (and $answering$  
                         (equal (car sent) '?) 
                         (not (null (cdr sent)))   
                         (equal '#\( (cadr sent)) 
                         (setq tl1 (t-list (cddr sent))) 
                         (equal '#\) (cadr tl1)) 
                         (cons
                               (list '~ (cons '$ans (car tl1))) 
                               (cddr tl1))) 
                (and (setq ps1 (p-sym sent)) 
                         (not (null (cdr ps1)))   
                         (equal '#\( (cadr ps1)) 
                         (setq tl1 (t-list (cddr ps1))) 
                         (equal '#\) (cadr tl1)) 
                         (cons
                               (cons (car ps1) (car tl1)) 
                               (cddr tl1))) 
                sent))) 
 


(defun is-infix-op (op-var) 
      (or
            (member op-var $infix-functors$ :test #'equal) 
            (and  
                     (not (characterp op-var)) 
                     (member (right-case op-var) $infix-functors$ :test #'equal)) 
            (member (mk-sym op-var) $infix-functors$ :test #'equal))) 



(defun is-postfix-op (op-var) 
      (or
            (member op-var $postfix-functors$ :test #'equal) 
            (and  
                     (not (characterp op-var)) 
                     (member (right-case op-var) $postfix-functors$ :test #'equal)) 
            (member (mk-sym op-var) $postfix-functors$ :test #'equal))) 


 
(defun mk-sym (char) 
      (if (not (member char '(#\( #\) ) :test #'equal)) 
         (intern (string char) :atp))) 
 
 

;;----------------------------- 
;;   TM-LIST --> TM , TM-LIST 
;;   TM-LIST --> TM 
 
(defun t-list (sent) 
      (let ((f1) (tl1)) 
          (or (and (setq f1 (tm-x sent)) 
                          (not (null (cdr f1))) 
                          (equal #\, (cadr f1)) 
                          (setq tl1 (t-list (cddr f1))) 
                          (cons
                                (cons (car f1) (car tl1)) 
                                (cdr tl1))) 
                (cons (list (car f1)) (cdr f1))))) 
 


;;------------------------------ 
;;   TM   -->  <prolog list> 
;;   TM   -->  number 
;;   TM   -->  ATOMICF (with function symbols in place od predicate symbols) 
 
(defun tm-x (sent) 
      (let (ps1 tl1 tm1 tm2) 
          (or 
                (plg-list sent)			 
                (and (equal (car sent) '#\() 
                         (setq tm1 (tm-x (cdr sent))) 
                         (is-postfix-op (cadr tm1))  
                         (equal '#\) (caddr tm1)) 
                         (cons
                               (list (cadr tm1) (car tm1)) 
                               (cdddr tm1))) 
                (and (equal (car sent) '#\() 
                         (setq tm1 (tm-x (cdr sent))) 
                         (is-infix-op (cadr tm1)) 
                         (setq tm2 (tm-x (cddr tm1))) 
                         (equal '#\) (cadr tm2)) 
                         (cons
                               (list (mk-sym (cadr tm1)) (car tm1) (car tm2)) 
                               (cddr tm2))) 
                (and (setq ps1 (f-sym sent)) 
                         (not (null (cdr ps1))) 
                         (equal '#\( (cadr ps1)) 
                         (setq tl1 (t-list (cddr ps1))) 
                         (equal '#\) (cadr tl1)) 
                         (cons
                               (cons (car ps1) (car tl1)) 
                               (cddr tl1))) 
                (and (numberp (car sent)) 
                         (if (or t $semantic-simplification$ $prolog-mode$) 
                            (cons (coerce (car sent) 'float) (cdr sent)) 
                            (cons (format nil "~D" (car sent)) (cdr sent)))) 
                sent))) 
 


;;------------------------------------------------------------------- 
;; Here I should ensure that s is not a variable 
;; but relaxing this condidtion |all|ows to have 
;; second order predicates such as "not" (PROLOG) and "apply" (LISP) 
;; or induction principles 
 


(defun f-sym (s)    
      (cond 
                  ((equal '#\- (car s)) (cons '- (cdr s))) 
                  ((and  
                             (atom (car s)) 
                             (not (numberp (car s))))
                   s))) 
 

(defun p-sym (s)
      (cond ((atom (car s)) s))) 
 

(defun is-atomic-f (a) 
      (let ((a1)) 
          (and (setq a1 (atomic-f (list a))) 
                   (null (cdr a1))))) 
 
 

;;-------------------------------------------------- 
;; actually a user logical variable begins with x, y, z ... 
;; 
;; but any other choice is acceptable 
 

 
(defun is-variable (x) 
      (if $prolog-mode$
         (upper-case-p (char (format nil "~A" x) 0)) 
         (member (char (format nil "~A" x) 0) '(#\x #\u #\y #\t #\z #\w #\v) :test #'equal))) 


  
(defun is-ANDsym (c) 
      (member c '(#\&) :test #'equal)) 



(defun is-ORsym (c) 
      (member c '(#\| #\+) :test #'equal)) 



(defun is-NOTsym (c) 
      (member c '(#\~) :test #'equal)) 
  

