;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp; -*-
;;; ******************************
;;; *  PORTABLE AI LAB -  IDSIA  *
;;; ******************************
;;;
;;; Filename:   LEN-adt
;;; Short Desc: Data definitions for ATP module
;;;             Clause class hierarchy
;;;             Proof-trees
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   May 1991
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;--------------------- CLAUSES -------------------------------
;; A clause is a disjunction of literals. A literal is either
;; an atomic formula or a negated atomic formula.
;; A clause is an object made of two slots:
;; positive-atoms : a list of atoms which are positive in the clause
;; negative-atoms : a list of atoms which are negative in the clause
;; Atoms are represented as lists.
;; Example 
;; The clause P(x) | ~ Q(x) | A(f(x,y)) is represented by the CLOS object:
;;
;; negative-atoms    ((Q 20001))
;; positive-atoms    ((P 20001) (A (f 20001 20002)))
;;
;; notice that variables are internally stored as integers

(in-package :atp)

(defclass clause-class()
  ((positive-atoms :initarg :positive-atoms
			   :accessor positive-atoms
			   :initform nil)
   (negative-atoms :initarg :negative-atoms
		   :accessor negative-atoms
		   :initform nil))
  (:documentation "A clause is a disjunction of positive 
                  or negative atomic formulas (see Chang-Lee, pag xxx)"))

(defclass all-negative-clause (clause-class)())
(defclass positive-atom (clause-class)())
(defclass para-clause (clause-class)())
(defclass equation (positive-atom para-clause)())
(defclass oriented-equation(positive-atom para-clause)())
(defclass horn-clause (clause-class)())

;; CLAUSE CONSTRUCTOR
(defmethod make-clause (pos-list neg-list)
  (make-instance 'clause-class
    :positive-atoms pos-list
    :negative-atoms neg-list))

;; CLAUSE MODIFIER
(defmethod remake-clause (pos-list neg-list (clause clause-class))
  (setf (positive-atoms clause) pos-list)
   (setf (negative-atoms clause) neg-list)
   clause)





(defmethod clause-equal (null-clause null-clause1)
  (declare (ignore null-clause null-clause1))
  t)

(defmethod clause-equal (null-clause (cl1 clause-class))
  (declare (ignore null-clause))
  nil)

(defmethod clause-equal ((cl1 clause-class) null-clause)
  (declare (ignore null-clause))
  nil)  

(defmethod clause-equal ((cl1 clause-class)(cl2 clause-class))
  (and (equal (positive-atoms cl1)
	      (positive-atoms cl2))
       (equal (negative-atoms cl1)
	      (negative-atoms cl2))))

   


;;--------------------Information about Clauses------------------------
;; These are informations attached to each clause
;; Actually a clause identifier has an object of clause class
;; as value, and an object of clause-info class as 'info property
;; The most important slot (for the user) is ancestors, that contatins
;; the history of the clause:
;; EX : ancestors ( c1 resolution c5 simp ( c2 c4))
;;      means that the present clause has been deduced by resolution between
;;      c1 and c5, and simplifying the result with c2 and then with c4.



(defclass clause-info ()
  ((ancestors      :initarg :ancestors
		   :initform nil
		   :accessor ancestors)
   (size           :initarg :size
		   :initform nil
		   :accessor size)
   (sizelist        :initarg  :sizelist
		   :initform nil
		   :accessor sizelist)
   (classlist    :initarg :classlist
                     :initform nil
		     :accessor classlist  ))

  (:documentation "Each clause has some associated information
                  used by the prover in many ways"))



;;--------------------Information about Strategy------------------------
;; A strategy is a CLOS object. The  main algorithm does not
;; know which strategy is actually using because the interface with 
;; strategies is uniform.
;; See file choose.cl for documentation about strategy class.


(defclass strategy ()
  ((init-function  :initarg :init-function 
		   :accessor init-function)
   (insert-function  :initarg :insert-function
		      :accessor insert-function )
   (insert-sets  :initarg :insert-sets
		  :accessor insert-sets )
   (delete-sets  :initarg  :delete-sets
		 :accessor delete-sets)
   (choose-clauses-function :initarg :choose-clauses-function
			     :accessor choose-clauses-function)))
 
    
 
