  

(in-package :bp)


(defclass net-graphics ()
       ((net :accessor net)
        (a-disp :accessor a-disp :initform nil)
        (aa-disp :accessor aa-disp :initform nil)
        (w-disp :accessor w-disp :initform nil)
        (ww-disp :accessor ww-disp :initform nil)
        (net-display :accessor net-display :initform nil)
        (weight-display :accessor weight-display :initform nil)
        (nbr-of-layers :accessor nbr-of-layers)
        (nbr-of-nodes :accessor nbr-of-nodes)
        (matrix-len :accessor matrix-len)
        (w-max-button :accessor w-max-button :initform nil)
        (border :accessor border)
        (w-box-width :accessor w-box-width)
        (w-max :accessor w-max)
        (w-disp-width :accessor w-disp-width)
        (w-disp-height :accessor w-disp-height)
        (w-positions :accessor w-positions)
        (w-tabs :accessor w-tabs)
        (a-box-width :accessor a-box-width)
        (a-max :accessor a-max)
        (a-disp-width :accessor a-disp-width)
        (a-disp-height :accessor a-disp-height)
        (a-positions :accessor a-positions)))



(defmethod close-activation-display ((graphics net-graphics))
      (close (aa-disp graphics)))



(defmethod close-weight-display ((graphics net-graphics))
      (close (ww-disp graphics)))




(defmethod init ((graphics net-graphics) (net net-class)
                                &key
                                (max-activity 1)
                                (max-weight 1)
                                (border 50)
                                (x-gab 53)
                                (y-gab 118)
                                (gab 20)
                                (w-box-width 14)
                                (a-box-width 20))
      (let* (a-box-offset max-nodes-in-layer
                (w-box-offset (- (/ w-box-width 2))))
          (setf (nbr-of-layers graphics)
                    (length (get-node-seq-names net))
                    (nbr-of-nodes graphics)
                    (mapcar 
                           (function (lambda (node-seq-name)
                                                  (length (get-node-seq net node-seq-name))))
                           (reverse (get-node-seq-names net)))
                    max-nodes-in-layer (apply #'max (nbr-of-nodes graphics))
                    a-box-width (min
                                                 (truncate
                                                     (- (a-disp-width graphics) (* 2 border))
                                                     (* max-nodes-in-layer 2))
                                                 (truncate
                                                     (- (a-disp-height graphics) (* 2 border))
                                                     (* (nbr-of-layers graphics) 4)))
                    x-gab (+ a-box-width
                                    (truncate
                                        (- (a-disp-width graphics) (* a-box-width max-nodes-in-layer) border)
                                        (1+ max-nodes-in-layer)))
                    y-gab (+ a-box-width
                                    (truncate
                                        (- (a-disp-height graphics) (* a-box-width (nbr-of-layers graphics)) border)
                                        (nbr-of-layers graphics)))
                    a-box-offset (/ a-box-width 2)
                    (net graphics) net)
          (setf *junk* graphics)
          (setf (matrix-len graphics)
                    (apply #'+ (nbr-of-nodes graphics)))
          (setf (w-disp-width graphics)
                    (+ (* 2 border)
                        (* (+ (matrix-len graphics) 1) gab))
                    (w-disp-height graphics)
                    (+ (* 2 border)
                        (* (- (matrix-len graphics) 1) gab))
                    (a-box-width graphics) a-box-width
                    (w-box-width graphics) w-box-width
                    (a-max graphics) max-activity
                    (w-max graphics) max-weight
                    (border graphics) border
                    (a-disp-height graphics)
                    (+ (* 2 border)
                        (* (- (nbr-of-layers graphics) 1) y-gab)))
          (setf (a-positions graphics)
                    (let ((nbr-of-layer 0)
                            (seq-names (reverse (get-node-seq-names net))))
                        (mapcar
                               #'(lambda (nbr-per-layer)
                                       (incf nbr-of-layer)
                                       (do ((count 1 (1+ count))
                                               (nodes (get-node-seq net (nth (1- nbr-of-layer) seq-names)) (rest nodes))
                                               (pos-list nil)
                                               (layer-offset (/ (* (- (apply #'max (nbr-of-nodes graphics))
                                                                                   nbr-per-layer)
                                                                                x-gab) 2)))
                                              ((eq count (1+ nbr-per-layer)) (reverse pos-list))
                                             (setf pos-list
                                                       (cons
                                                             (list
                                                                (+ (round border 2) (* (1- count) x-gab) a-box-offset layer-offset)
                                                                (+ (round border 2) (* (1- nbr-of-layer) y-gab) a-box-offset)
                                                                (first nodes))
                                                             pos-list))))
                               (nbr-of-nodes graphics))))
          (setf (w-positions graphics)
                    (make-array
                           (list
                              (matrix-len graphics) 
                              (matrix-len graphics))
                           :adjustable nil
                           :initial-element nil))
          (setf (w-tabs graphics)
                    (make-array
                           (list (matrix-len graphics) 3)
                           :adjustable nil
                           :initial-element nil))
          (let ((all-nodes (apply #'append
                                             (mapcar 
                                                    (function (lambda (node-seq-name)
                                                                           (get-node-seq net node-seq-name)))
                                                    (reverse (get-node-seq-names net))))))
              (dotimes (i (matrix-len graphics))
                    (let ((x-node (nth i all-nodes)))
                        (dolist (connection (in-connections x-node))
                              (let ((j (position (from-node connection) all-nodes)))
                                  (setf (aref (w-positions graphics)
                                                 i
                                                 j)
                                            (list
                                               (- (+ (* 2 border) (* j gab) w-box-offset) 20)
                                               (+ 20 (* i gab))
                                               connection)))))))
          (dotimes (i (matrix-len graphics))
                (setf (aref (w-tabs graphics) i 0)
                          (+ border (* i gab) w-box-offset))
                (setf (aref (w-tabs graphics) i 1)
                          (+ border (* i gab) w-box-offset w-box-width))
                (setf (aref (w-tabs graphics) i 2) nil))
          (let ((pos 0))
              (setf (aref (w-tabs graphics) pos 2)
                        (+ border (* pos gab)))
              (dolist (nbr (butlast (nbr-of-nodes graphics)))
                    (incf pos nbr)
                    (setf (aref (w-tabs graphics) pos 2)
                              (+ border (* pos gab)))))))



(defun create-net-topology ()
      (setq *topology*
            (list
               (list 'output-layer (number-of-nodes *target-set*))
               (list 'hidden-layer (if *hidden-neurons*
                                                     *hidden-neurons*
                                                     (number-of-nodes *input-set*)))
               (list 'input-layer (number-of-nodes *input-set*))))
      (unless *hidden-neurons*
            (setf *hidden-neurons* (number-of-nodes *input-set*)))
      (setq *current-net*
            (bp-make-net 'current-net *topology*))
      (setf *total-count* 0)
      (setf *net-graphics* (make-instance 'net-graphics))
      (setf (a-disp-width *net-graphics*)
                (min 280 (round (* 0.75 (box-width (window-interior *bp-main-window*))))))
      (setf (a-disp-height *net-graphics*)
                (min 540 (round (* 0.75 (box-height (window-interior *bp-main-window*))))))
      (init *net-graphics* *current-net* :max-weight *w-max*)
      (when *show-topology*
             (setq *topology-window* (open-activation-display *net-graphics*)))
      (when *show-errors*
             (setq *errors-window* (open-error-display)))
      (when *show-weights*
             (setq *weights-window* (open-weight-display *net-graphics*))))
                             




(defun make-training-data (def)
      (make-instance 'training-data-class
             :input-patterns *input-set*
             :target-patterns *target-set*))




(defun start-learn ()
     (when *target-set-editor*
          (when (table-needs-saving-p *target-set-editor*)
               (setq *target-set* (editor-to-table *target-set-editor*))))
     (when *input-set-editor*
           (when (table-needs-saving-p *input-set-editor*)
                (setq *input-set* (editor-to-table *input-set-editor*))     
              (unless *show-topology*  (create-net-topology)))) ;;OKKIO 15.11.95
      (setf *training-data*
                (make-training-data *topology*))
      (when *show-errors*
             (when *not-stable*
                    (setq *total-count* 0)
                    (setq *not-stable-continue* t)
                    (init-window *errors-window*)))
      (setq *not-stable* nil)
      (setf *stop* nil)
      (when *show-topology*
             (set-dialog-item-available-p
              (find-named-object ':test-net *topology-window*)
              t)
             (set-dialog-item-available-p
              (find-named-object ':continue *topology-window*)
              nil))
      (incf *total-count*
          (learn
              *training-data*
              *current-net*
              :error-tolerance *tolerance*
              :max-count *max-trials*))
      (format-display *bp-output-window*
           "~%~a total steps.~%" *total-count*)
      (when *show-topology*
             (show-activities *net-graphics*))
      (when *show-weights*
             (show-weights *net-graphics*))
      (setf *learn-done* t)
      (unless *stop* (start-stop-learn)))



(defun stop-learn ()
      (setq *stop* t))






(defmethod learn ((training-data training-data-class)
                                    (net bp-net-class)
                                    &key 
                                    (error-tolerance 0.1)
                                    (max-count 500))
      (reset-net net)
      (format-display *bp-output-window*
           "~%Learning ...~%~%")   
      (dotimes
             (counter max-count
                   (progn
                        (setq *not-stable* t)
                        (format-display *bp-output-window*
                             "Not stable at ~a.~%" max-count)
                        max-count))
            (process-pending-events)
            (format-display *bp-output-window*
                 "Cycle ~a~%" (1+ counter))
            (when (or *stop*
                               (prog1
                                    (learn-patterns net (convert (rows (input-patterns training-data))
                                                                           (rows (target-patterns training-data)))
                                        (get-node-seq-names net)
                                        :error-tolerance error-tolerance)
                                    (when *show-errors*
                                           (disp-performance *errors-window*
                                               (make-position
                                                      (round (+ *init-x* (* *step-x* (+ *total-count* (1+ counter)))))
                                                      (round (- *init-y* (* *step-y* *performance*)))))
                                           (disp-error *errors-window*
                                               (make-position
                                                      (round (+ *init-x* (* *step-x* (+ *total-count* (1+ counter)))))
                                                      (round (- *init-y* (* *step-y-err* (/ *error-value* 2)))))))
                                    (when *auto-update* (show-weights *graphics*))))
                   (progn
                        (format-display *bp-output-window*
                              "~%~a learning cycles.~%" (1+ counter))
                        (unless *stop*
                              (if *not-stable-continue*
                                 (setq *not-stable-continue* nil)
                                 (when (< (+ *total-count* (1+ counter)) (/ *max-trials* 2))
                                        (when (yes-or-no-p 
                                                           (format nil
                                                                "The scale is not good. You may choose a good~%scale if max. trials = ~a.~%Should I set it for you?"
                                                                (round (* (+ *total-count* (1+ counter)) 1.5))))
                                               (setq *max-trials* (round (* (+ *total-count* (1+ counter)) 1.5)))))))
                        (return (1+ counter))))))



(defun convert (input target)
      (if (and input target)
         (cons (list (car input) (car target))
               (convert (cdr input) (cdr target)))
         nil))




(defun reinitialize ()
      (reinit-net *current-net*)
      (when (windowp *topology-window*)
             (show-activities *net-graphics*)
             (set-dialog-item-available-p
                  (find-named-object ':test-net *topology-window*)
                   t)
             (set-dialog-item-available-p
                  (find-named-object ':continue *topology-window*)
                   nil))
      (when (windowp *weights-window*)
             (show-weights *net-graphics*))
      (when (windowp *errors-window*)
             (init-window *errors-window*))
       (setf *total-count* 0)
       (format-display *bp-output-window* "~%Network reinitialized.~%"))
      


(defparameter *test-nb* 0)

(defun test-net ()
      (let ((input-pattern (first (rows *input-set*))))
          (set-dialog-item-available-p
                (find-named-object ':test-net *topology-window*)
                nil)
          (compute-net-output *current-net* input-pattern)
          (show-activities *net-graphics*)
          (set-dialog-item-available-p
                (find-named-object ':continue *topology-window*)
                t)          
          (setq *test-nb* 1)))
      

      
(defun test-continue ()
      (let ((input-pattern (nth *test-nb* (rows *input-set*))))
          (compute-net-output *current-net* input-pattern)
          (show-activities *net-graphics*)
          (incf *test-nb*)
          (when (= *test-nb* (length (rows *input-set*)))
                 (set-dialog-item-available-p
                       (find-named-object ':test-net *topology-window*)
                       t)
                 (set-dialog-item-available-p
                       (find-named-object ':continue *topology-window*)
                       nil))))
                 
      
      
