;;; -*- Mode: LISP; Package: BP; Syntax: Common-lisp;      -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   bp.lsp
;;; Short Desc: Kernel of neural network simulations
;;; Version:    1.0
;;; Status:     Experimental (July 1990)
;;; Last Mod:   27.1.92 - TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;;
;;;


(in-package :bp)



(defmacro make-node-seq (net a-node-class node-seq-name 
                                                      number-of-nodes-or-symbol-list &rest args)
      (let ((dummy (gensym))
              (node-sym (gensym)))
          `(if (numberp ,number-of-nodes-or-symbol-list)
               (dotimes (,dummy ,number-of-nodes-or-symbol-list)
                     (make-node ,net ,a-node-class (list ,node-seq-name) ,@args))
               (dolist (,node-sym
                             (reverse ,number-of-nodes-or-symbol-list)
                             (get-node-seq ,net ,node-seq-name))
                     (set ,node-sym
                           (make-node ,net ,a-node-class (list ,node-seq-name) ,@args))))))



(defmacro connect-nodes-to-node (a-connection-class 
                                                                     from-node-seq to-node &rest args)
      (let ((node (gensym)))
          `(dolist (,node ,from-node-seq)
                 (make-connection ,a-connection-class ,node ,to-node ,@args))))



(defmacro connect-node-to-nodes (a-connection-class 
                                                                     from-node to-node-seq &rest args)
      (let ((node (gensym)))
          `(dolist (,node ,to-node-seq)
                 (make-connection ,a-connection-class ,from-node ,node ,@args))))



(defmacro connect-nodes-to-nodes (a-connection-class 
                                                                       from-node-seq to-node-seq &rest args)
      (let ((node (gensym)))
          `(dolist (,node ,to-node-seq)
                 (connect-nodes-to-node ,a-connection-class ,from-node-seq ,node ,@args))))



(defmacro connect-nodes-in-node-seq (a-connection-class
                                                                             node-seq self-connected &rest args)
      (let ((node-i (gensym))
              (node-j (gensym)))
          `(dolist (,node-i ,node-seq)
                 (dolist (,node-j ,node-seq)
                       (if (not (eq ,node-i ,node-j)) 
                          (make-connection ,a-connection-class ,node-i ,node-j ,@args)
                          (when ,self-connected 
                                 (make-connection ,a-connection-class ,node-i ,node-j ,@args)))))))
     


(defun feed-in (activation-list node-seq)
      (do ((current-a-list activation-list (cdr current-a-list))
              (current-n-list node-seq (cdr current-n-list)))
             ((or (null current-a-list) (null current-n-list)))
            (setf (activation (car current-n-list))
                      (car current-a-list))
            (setf (output (car current-n-list))
                      (compute-output (car current-n-list)))))
  


(defun feed-out (node-seq)
      (mapcar (function (lambda (node)
                                              (activation node)))
             node-seq))



(defun synch-update (node-seq)
      (dolist (node node-seq)
            (touch node))
      (dolist (node node-seq)
            (update node)))


(defun all-one-asynch-update (node-seq)
      (let* ((number-of-nodes (length node-seq))
                random-number
                (testvector (make-array number-of-nodes :initial-element 'nil)))
          (loop
              (loop
                  (setf random-number (random number-of-nodes))
                  (when (null (aref testvector random-number)) (return)))
              (let ((node (nth random-number node-seq)))
                  (touch node)
                  (update node))
              (setf (aref testvector random-number) t)
              (when (every (function identity) testvector) (return t)))))



(defun asynch-update (node-seq)
      (let* ((number-of-nodes (length node-seq))
                random-number)
          (dotimes (node-number number-of-nodes)
                (setf random-number (random number-of-nodes))
                (let ((node (nth random-number node-seq)))
                    (touch node)
                    (update node)))))



(defmacro cycle (n &rest body)
      (let ((dummy (gensym)))
          `(dotimes (,dummy ,n) ,@body)))


(defmacro cycle-with-action-each-nth-time (n fcn-list nth-time action-list)
      (let ((dummy (gensym)))
          `(dotimes (,dummy ,n)
                 ,@fcn-list
                 (when (zerop (mod ,dummy ,nth-time))
                        ,@action-list))))
	

;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
