;;; -*- Mode: LISP; Package: cky; Syntax: Common-lisp; -*-

(in-package :cky)


;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   cky-grammar-gui.cl
;;; Short Desc: general user interface for the grammar routines
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.2.92 - FB
;;; Author(s):  Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Rod Johnson wrote a character oriented version of the interface
;;; Fabio Baj  wrote  the graphic user interface
;;; --------------------------------------------------------------------------


     
(defvar *new-entry)
(defvar *rhslist* )    
(defvar *delete-rule*)  
(defvar *find-rule*)
(defvar *load-grammar*) 
(defvar *save-grammar*)
(defvar *exit-grammar*)   
(defvar *help-grammar*)
(defvar *convert-grammar*)
(defvar *view-grammar*)
(defvar *options-grammar*)



(defun loadsyn (filename)
      (catch 'cancel
            (with-open-file (infile filename :direction :input) 
                 (setq *grammar-filename* filename)
                 (setq *grammar*  nil)
                 (format-display *cky-msg-disp*
                      "Loading grammar from ~A~%" filename)
                 (do ((entry (read infile nil 'eof) (read infile nil 'eof)))
                        ((eq 'eof entry))
                       (if (not (is-a-g-rule entry))
                          (progn
                               (format-display *cky-msg-disp*
                                    "BAD RULE: ~A" entry)
                               (throw 'cancel nil))
                          (setq *grammar* (newsyn entry *grammar*)))))
            (view-grammar)))



;;;===========================================================================

(defun is-a-g-rule (r)
      (cond ( (atom r) nil)
                  ( t (and (is-lhs (car r) )
                                (is-rhs (cdr r))))))



(defun is-lhs (s) 
      (atom s))



(defun is-rhs (s)
      (or (null s)(is-seq s)))



(defun is-seq(s)
      (cond ((atom s) nil)
                  ((and (is-node (car s))(null (cdr s))) t)
                  ((and (is-node (car s))(is-seq (cdr s))) t)))
		


(defun is-node (s) 
      (or
            (is-self s)
            (is-alts s)
            (is-star s )
            (is-opts s)))



(defun is-self (s)
      (cond ((atom s) nil)
                  (t (and (eq 'self (car s))
                               (atom  (cadr s))))))



(defun is-star(s)
      (cond ((atom s) nil)
                  (t (and (eq 'star (car s))
                               (is-rhs (cdr s))))))



(defun is-alts(s)
      (cond ((atom s) nil)
                  (t (and (eq 'alts (car s))
                               (all-rhs(cdr s))))))



(defun all-rhs (s)
      (and (mapcar #'is-rhs s)))
			

      
(defun is-opts (s)
      (cond ((atom s) nil)
                  (t (and (eq 'opts (car s))
                               (is-rhs(cdr s))))))



;;=========================================================

(defun newsyn (g-rule grammar)
      (when g-rule
             (ordered-insert1 g-rule
                  grammar)))
		      


(defun g-alphalessp (x y) 
      (string< (string x) (string y)))

  

;;=============================================================

(defun findsyn (cat)
      (let ((rules (mapcan
                                  #'(lambda (prod-rule)
                                          (cond ((eq (car prod-rule) cat)
                                                       (list prod-rule))))
                                  *grammar*)))
          (cond (rules
                           (seerules rules ))
                      (t
                         (format nil  "No rules for ~A" cat)))))
	    


(defun map-conc (fn list)
      (cond ((null list) "")
                  (t (concatenate 'string
                           (funcall fn (car list))
                           (map-conc fn (cdr list))))))



(defun view-grammar ()
      (create-cky-grammar-window (grammar-to-sequence *grammar*)))



(defun create-cky-grammar-window (sequence)
      (setq *grammar-dialog*
            (open-dialog ()
                  'cky-grammar-window *cky-main-window* :pop-up-p nil
                  :background-color (make-rgb :red 0 :green 128 :blue 128) 
                  :user-closable nil :user-scrollable nil :user-resizable t
                  :user-shrinkable t 
                  :sequence sequence
                  :button-menu-bar '(
                                                   ("Close" 'close-cky-grammar-window)
                                                  ("Add" 'new-grammar-rule)
                                                  ("Delete" 'delete-grammar-rule))
                  :title (concatenate 'string
                                 "Grammar : " 
                                 (file-namestring *grammar-filename*))
                  :visible-box (make-box 0 0 0 0))))



(defun close-cky-grammar-window (item)
      (when (window-close *grammar-dialog*)
             (progn
                  (setq *grammar* nil)
                  (set-grammar-menu-before-load)
                  (setq *grammar-dialog* nil)))
      t)

      

(defun seerules (grammar)
      (cond ((null grammar) (format nil "~%"))
                  (t (concatenate 'string 
                           (showrule (car grammar)) (format nil "~%")
                           (seerules (cdr grammar))))))

 

(defun showrule (g-rule)
      (format nil  "~A -> ~A"
           (string (car g-rule))
           (cond ((cdr g-rule)
                        (cond ((eq (caadr g-rule) 'alts)
                                     (concatenate 'string
                                           (showrhs (cadadr g-rule))
                                           (map-conc (function (lambda (seq)
                                                                                       (concatenate 'string
                                                                                             (format nil " | ")
                                                                                             (showrhs seq))))
                                                  (cddadr g-rule))))
                                    (t (showrhs (cdr g-rule))))))))



(defun showrhs (rhs)
      (concatenate 'string
            (showcat (car rhs))
            (map-conc
                   (function
                        (lambda (cat) 
                            (concatenate 'string " " (showcat cat))))
                   (cdr rhs))))



(defun showcat (cat)
      (cond ((eq (car cat) 'self)
                   (concatenate 'string
                         (format nil "~A" (cadr cat))
                         (showfeats (cddr cat))))
                  ((eq (car cat) 'star)
                   (cond ((null (cddr cat)) 
                                (concatenate 'string
                                      (showcat (cadr cat))
                                      "*"))
                               (t (concatenate 'string
                                        "("
                                        (showrhs (cdr cat))
                                        ")*"))))
                  ((eq (car cat) 'alts)
                   (concatenate 'string  
                         "("
                         (showrhs (cadr cat))
                         (map-conc (function
                                                   (lambda (node)
                                                       (concatenate 'string
                                                             " | "
                                                             (showrhs node))))
                                (cddr cat))
                         ")"))
                  ((eq (car cat) 'opts) 
                   (concatenate 'string "("
                         (showrhs (cdr cat))
                         ")"))
                  (t  (concatenate 'string 
                            "??? "
                            (format nil "~A" cat)
                            " ???"))))



(defun showfeats (features)
      (cond ((and *syn-features* features)
                   (concatenate 'string 
                         " {"
                         (showeqn (car features))
                         (map-conc (function
                                                   (lambda (eqn) 
                                                       (concatenate 'string
                                                             ", "
                                                             (showeqn eqn))))
                                (cdr features))
                         "}"))
                  (t "")))



(defun showeqn (eqn)
      (cond ((atom eqn)
                   (format nil "~A" eqn))
                  (t (concatenate 'string 
                           (showpath (car eqn))
                           (cond 
                                       ((cdr eqn) (concatenate 'string
                                                                " = "
                                                                (showpath (cdr eqn))))
                                       ( t ""))))))
  


(defun showpath (path)
      (cond ((atom path)
                   (format nil "~A"  path))
                  (t (concatenate 'string 
                           "<"
                           (cond ((eq (car path) 'up)
                                        "up")
                                       ((eq (car path) 'down)
                                        "down")
                                       (t (showpath (car path))))
                           (map-conc (function
                                                     (lambda (node)
                                                         (concatenate 'string " " (showpath node))))
                                  (cdr path))
                           ">"))))



;-------------------------------------------------------------

(defun convert ()
      (mapcar (function (lambda (g-rule)
                                              (cons (car g-rule) (convertrhs (cdr g-rule)))))
             *grammar*))



(defun convertrhs (rhs)
      (cond ((null rhs) nil)
                  (t (cond ((eq (caar rhs) 'self)
                                   (cons (list 'self (cdar rhs)) (convertrhs (cdr rhs))))
                                 ((eq (caar rhs) 'opts) (cons (cons 'opts (convertrhs (cdar rhs)))
                                                                               (convertrhs (cdr rhs))))
                                 ((eq (caar rhs) 'alts) 
                                  (cons (cons 'alts (mapcar 'convertrhs (cdar rhs)))
                                        (convertrhs (cdr rhs))))
                                 ((eq (caar rhs) 'star)
                                  (cons (cons 'star (convertrhs (cdar rhs)))
                                        (convertrhs (cdr rhs))))
                                 (t (cons (car rhs) (convertrhs (cdr rhs))))))))



;;;====================================================================
       
(defun makesyn (lhs rhs)  
      (let ((rules (add-to-sequence
                                 *grammar-dialog*
                                 (concatenate 'string lhs " -> " rhs))))
          (when rules
                 (setf *grammar* (sequence-to-grammar rules)))))
  
    

(defun set-up-new-rule (lhs rhs)
      (setq *rhslist* rhs)
      (myratom)
      (let ((rhs (readrhs nil)))
          (cons (read-from-string lhs) rhs)))

	

(defun readcat ()
      (let ((cat lookahead))
          (setq lookahead (myratom ))
          (cond ((eq lookahead '\{)
                       (cons 'self (cons cat (readfeats nil ))))
                      (t (list 'self (read-from-string cat))))))



(defun readrhs (rhs  )
      (cond ((null lookahead)
                   (reverse rhs))
                  (t 
                     (cond ((string= lookahead "|")
                                  (setq lookahead (myratom  ))
                                  (let* ((alt (readrhs nil  ))
                                            (this (reverse rhs))
                                            (tail (cond ((and (eq (caar alt) 'alts)
                                                                           (null (cdr alt)))
                                                                 (cons 'alts (cons this (cdar alt))))
                                                                (t (list 'alts this	alt)))))
                                      (list tail)))
                                 ((string= lookahead "*")
                                  (setq lookahead (myratom  ))
                                  (let ((this (car rhs)))
                                      (cond ((eq (car this) 'opts)
                                                   (readrhs (cons (cons 'star (cdr this))
                                                                           (cdr rhs)) ))
                                                  (t
                                                     (readrhs (cons (list 'star this)
                                                                             (cdr rhs))
                                                      )))))
                                 ((string= lookahead "(")
                                  (setq lookahead (myratom  ))
                                  (let ((seq (readrhs nil  )))
                                      (cond ((string= lookahead ")")
                                                   (setq lookahead (myratom  ))
                                                   (cond ((and (null (cdr seq))
                                                                          (eq (caar seq) 'alts))
                                                                (readrhs (cons (car seq) rhs)  ))
                                                               (t (readrhs (cons (cons 'opts seq) rhs)
                                                                    ))))
                                                  (t (badrule ") expected after" seq)))))
                                 ((string= lookahead ")")
                                  (reverse rhs))
                                 (t (readrhs (cons (readcat) rhs)  ))))))



;; disgusting function -> lookahead is global

(defun myratom ()
      (setq lookahead (pop *rhslist*)))



;;;==================================================

#|
(defun find-rules (cat)
      (let* ((current-grammar-window
                  (find-window
                     (concatenate 'string
                           "Grammar : " 
                           (enough-namestring *grammar-filename*))))
                (current-rule-list (when current-grammar-window
                                                        (table-sequence (wsequence current-grammar-window))))
                (result nil))
          (when current-grammar-window
                 (dolist (item current-rule-list)
                       (when (string= cat item :end2 (length (coerce cat 'list)))
                              (push item result))))
          (if result
             (reverse result)
             nil)))
|#


;;===================================================

(defun savesyn (filename)
      (when filename
             (progn
                  (with-open-file 
                       (port filename :direction :output :if-exists :supersede)  
                       (format-display *cky-msg-disp* "Saving grammar to ~A~%" filename)
                       (dumpsyn *grammar* port))
                  (setf (save-p (wsequence *grammar-dialog*)) nil)
                  t)))
    


(defun dumpsyn (grammar port)
      (format port "
; Grammar source file

; Rule format is:-

; <rule>	::= ( <lhs> . <rhs> )
; <lhs>		::= <symbol>
; <rhs>		::= <nil> | ( <seq> )
; <seq>		::= <node> | <node> <seq> 
; <node>	::= <self> | <alts> | <star> | <opts>
; <self>	::= ( self . <category> )
; <alts>	::= ( alts . <rhs> ... <rhs> )
; <star>	::= ( star . <rhs> )
; <opts>	::= ( opts . <rhs> )
" )
      (format port"
; <category>	::= ( <symbol> . <features> )
; <features>	::= <nil> | <feature> ... <feature>
; <feature>	::= <symbol> | ( <function> . <value> )
; <function>	::= ( <variable> ) | ( <variable> <symbol> ... <symbol> )
; <variable>	::= up | down
; <value>	::= <function> | <symbol> | <number>

"  ) 
      (mapc #'(lambda (entry)
                           (print entry port))
             grammar))



(defun select-rules-to-delete (lista)
      (when lista
             (select-item-from-list lista 
                   :window-title "Select rules to delete:"
                   :selection-type :disjoint
                   :action-function (function delete-rules))))

         

(defun sequence-to-grammar (lista)
      (when lista
             (let ((result nil))
                 (dolist (item lista)
                       (setq item (scan-string item))
                       (push (set-up-new-rule (car item) (cdr item)) result))
                 (reverse result))))

(defun grammar-to-sequence (grammar)
      (mapcar (function showrule) grammar))



(defun set-grammar-menu-after-load ()
      (set-menu-item-available-p
           (find-named-object ':load-grammar *cky-grammar-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':load-grammar *cky-file-menu*) nil)
      (set-dialog-item-available-p
           (find-named-object ':load-grammar *out-dialog*) nil)
      (set-menu-item-available-p
           (find-named-object ':save-cky-grammar *cky-grammar-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':new-grammar-rule *cky-grammar-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':delete-grammar-rule *cky-grammar-menu*) t))
      


(defun set-grammar-menu-before-load ()
      (set-menu-item-available-p
           (find-named-object ':load-grammar *cky-grammar-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':load-grammar *cky-file-menu*) t)
      (set-dialog-item-available-p
           (find-named-object ':load-grammar *out-dialog*) t)
      (set-menu-item-available-p
           (find-named-object ':save-cky-grammar *cky-grammar-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':new-grammar-rule *cky-grammar-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':delete-grammar-rule *cky-grammar-menu*) nil))



       