;;; -*- Mode: LISP; Package: cky; Syntax: Common-lisp; -*- 
 
(in-package :cky) 
 
;;; ************************************************************************ 
;;; 
;;; PORTABLE AI LAB - IDSIA LUGANO 
;;; 
;;; ************************************************************************ 
;;; 
;;; Filename:   cky-init.lisp 
;;; Short Desc: general user interface for the  CKY tool 
;;;             
;;; Version:    1.0 
;;; Status:     New 
;;; Last Mod:   13 January 1993 - PC 
;;; Author(s):  Paolo Cattaneo 
;;; 
;;; -------------------------------------------------------------------------- 
;;; Change History: 
;;;  
;;;  
;;; -------------------------------------------------------------------------- 
 
(defun cky () 
      (setq pail-lib::*cky-module-loaded* t) 
      (in-package :cky) 
      (create-cky-window)
      (set-cky-menubar))
   
 
 
(defun create-cky-window () 
      (unless (windowp *cky-main-window*)
            (progn
                 (setq *cky-main-window*
                       (open-dialog () 'cky-main-dialog *lisp-main-window* :pop-up-p nil
                             :title "CKY Main"
                             :name ':cky-main
                             :user-closable nil
                             :background-color (make-rgb :red 0 :green 128 :blue 128) 
                             :window-exterior (clipping-box *lisp-main-window*)))
                 (add-to-window-menu
                       (stream-title *cky-main-window*)
                       (object-name *cky-main-window*))
                 (create-cky-dialogs))))



(defun create-cky-dialogs ()
     (let* ((*number-of-buttons* 4)           
                (box (window-interior *cky-main-window*))
                (h-indent (round (* 0.2 (box-width box))))
                (v-indent (round (* 0.2 (- (box-height box) 20))))
                (h-size (- (box-width box) (* 2 h-indent)))
                (button-width (round (/ h-size *number-of-buttons*)))
                (button-height 25)
                (help-button nil)
                (out-dialog nil))
          (setq out-dialog (setq *out-dialog*
                (open-dialog () 'dialog *cky-main-window*
                      :pop-up-p nil
                      :title "CKY: Output"
                      :user-closable nil
                      :user-shrinkable t
                      :user-resizable nil
                      :user-scrollable nil
                      :user-movable t
                      :window-border :dialog-box
                      :window-interior
                      (make-box 
                             h-indent 
                             (+ v-indent button-height)
                             (- (box-width box) h-indent)
                             (- (box-height box) v-indent)))))
         (setq *cky-subdisp* (setq *cky-msg-disp*
              (open-stream
                   'text-edit-window out-dialog :output
                    :name ':cky-output
                   :user-clossable nil
                   :user-shrinkable nil
                   :user-resizable nil
                    :user-scrollable t
                    :user-movable nil
                   :font (make-font :modern nil 14)
                   :window-exterior
                   (make-box
                             0 button-height
                             (box-width (visible-box out-dialog))
                             (box-height (visible-box out-dialog))))))
         (update-dialog *out-dialog*
               (list
                  (make-dialog-item :widget 'button
                       :title "Help"
                       :set-value-fn #'(lambda (&rest x) (stepper-help))
                       :font (make-font nil :arial 16 '(:bold))
                       :available-p nil
                       :box (make-box
                                       0 0
                                        (round (/ (box-width (visible-box out-dialog)) *number-of-buttons*))
                                        button-height))
                  (make-dialog-item :widget 'button
                         :title "Parse"
                         :set-value-fn #'(lambda (&rest x) (parse-sentence))
                         :font (make-font nil :arial 16 '(:bold))
                         :available-p t
                         :box
                         (make-box
                                (round (/ (box-width (visible-box out-dialog)) *number-of-buttons*))
                                0
                                (round (/ (* 2 (box-width (visible-box out-dialog))) *number-of-buttons*))
                                button-height))
                  (make-dialog-item :widget 'button
                         :title "Load Grammar"
                         :name ':load-grammar
                         :set-value-fn #'(lambda (&rest x) (load-cky-grammar))
                         :font (make-font nil :arial 16 '(:bold))
                         :available-p t
                         :box
                         (make-box
                                (round (/ (* 2 (box-width (visible-box out-dialog))) *number-of-buttons*)) 0
                                (round (/ (* 3 (box-width (visible-box out-dialog))) *number-of-buttons*)) button-height))
                  (make-dialog-item :widget 'button
                       :title "Load Lexicon"
                       :name ':load-lexicon
                       :set-value-fn #'(lambda (&rest x) (load-cky-lexicon))
                       :font (make-font nil :arial 16 '(:bold))
                       :available-p t
                       :box (make-box
                                       (round (/ (* 3 (box-width (visible-box out-dialog))) *number-of-buttons*)) 0
                                       (box-width (visible-box out-dialog)) button-height))))))     

(defun cky-new ()
     (:new *cky-main-window*))

(defun cky-open ()
      (pail-open *cky-main-window*))                  
             
(defun cky-save ()
      (pail-save *cky-main-window*))                  
             
(defun cky-save-as ()
      (pail-save-as *cky-main-window*))         

(defun cky-close ()
     (let ((window (front-window *cky-main-window*)))
         (if window (window-close window)
            (window-close *cky-main-window*))))

(defun cky-print ()
     (:print (front-window *cky-main-window*)))

  
(setq *cky-file-menu*
      (make-menu-item :title "~File" :value
             (open-menu 
                   (list 
                      (make-menu-item :title "~New"
                             :value 'cky-new :available-p t
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\N))
                      (make-menu-item :title "~Open"
                             :value 'cky-open :available-p t
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\O))
                      (make-menu-item :title "~Close"
                             :value 'cky-close :available-p t
                             :selected-p nil :font nil)
                      menu-separator
                      (make-menu-item :title "Load Grammar..."
                             :name ':load-grammar
                             :value 'load-cky-grammar :available-p t 
                             :selected-p nil :font nil)
                      (make-menu-item :title "Load Lexicon..."
                             :name ':load-lexicon
                             :value 'load-cky-lexicon :available-p t 
                             :selected-p nil :font nil)
                      menu-separator
                      (make-menu-item :title "~Save"
                             :value 'cky-save :available-p nil
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\S))
                      (make-menu-item :title "Save ~As"
                             :value 'cky-save-as :available-p t
                             :selected-p nil :font nil)
                      menu-separator
                      (make-menu-item :title "~Print"
                             :value 'cky-print :available-p t
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\S))
                      menu-separator
                      (make-menu-item :name :help :title "~Help"
                             :value 'cky-file-help :available-p nil
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\H))
                      (make-menu-item :title "E~xit from CKY"
                             :value 'cky-exit :available-p t
                             :selected-p nil :font nil :event-synonym 
                             '(control-key #\X)))
                   'pop-up-menu *lisp-main-window* :name '*cky-file-menu* :title '"" 
                   :selection-function 'funcall-menu-item)))
 
 
 
(defun cky-exit () 
      (window-close *cky-main-window*) 
     )

(defun quit-cky ()
      (setq *sentence-list* nil) 
      (setq *grammar-filename* nil) 
      (setq *grammar* nil) 
      (setq *lexicon-filename* nil) 
      (setq *lexicon* nil) 
      (setq *current-sentence* "") 
      (setq *cky-main-window* nil) 
      (setq pail-lib::*cky-module-loaded* nil)
      (make-available :cky)
     (select-window pail-lib::*pail-main-window*)
     (in-package :pail-lib))
  
 

(setq *cky-parser-menu* 
      (make-menu-item :title "~Parser" :value
             (open-menu 
                   (list 
                      (make-menu-item :title "Parse Sentence"
                             :value 'parse-sentence :available-p t)
                      (make-menu-item :title "Parse Predefined Text"
                             :name ':parse-predefined-text
                             :value 'parse-predefined-text :available-p nil)
                      (make-menu-item :title "Repeat Last Parse"
                             :name ':repeat-last-parse
                             :value 'repeat-last-parse :available-p nil)
                      menu-separator
                      (make-menu-item :title "Choose Parser..."
                             :value 'select-algorithm :available-p t)
                      (make-menu-item :title "Set Options..."
                             :value 'options :available-p t)
                      menu-separator
                      (make-menu-item :title "Environment Info"
                             :value 'display-information :available-p t))
                   'pop-up-menu *lisp-main-window* :name '*cky-parser-menu* :title '"" 
                   :selection-function 'funcall-menu-item)))
 
                      

                      
(defun get-new-sentence () 
      (if (and *grammar* *lexicon*) 
         (multiple-value-bind (new-sent other value)
                 (ask-user-for-string
                       "Please, enter sentence to parse:"
                       *current-sentence*
                       "OK" "Cancel")
                (when (and (string= value "OK") (string/= new-sent ""))
                       (progn
                            (set-menu-item-available-p
                                  (find-named-object ':repeat-last-parse *cky-parser-menu*) t)
                            (set-menu-item-available-p
                                  (find-named-object ':parse-predefined-text *cky-parser-menu*) t)
                            (setq *current-sentence* new-sent) 
                            (pushnew new-sent *sentence-list* :test #'string=))))
         (setq *current-sentence* "")))
 

 
(defun parse-sentence () 
      (when (get-new-sentence) 
             (when (string/= *current-sentence* "")
                    (repeat-last-parse))))
 


(defun parse-predefined-text () 
      (let ((sentence (select-item-from-list *cky-main-window*
                                         *sentence-list*
                                         "Sentence Selection")))
          (when sentence 
                 (setq *current-sentence* sentence) 
                 (repeat-last-parse))))
 


(defun repeat-last-parse () 
      (initialise) 
      (if (and (string/= "" *current-sentence*) 
                    *grammar* *lexicon*) 
         (progn 
              (when *show-chart* 
                     (when (windowp *chart-window*)
                            (window-close *chart-window*) 
                            (setq *chart-window* nil)) 
                     (graph-chart-init)) 
              (analyse (scan-string (string-upcase *current-sentence*))))     
         (progn
              (if (not *grammar*)
                 (pop-up-message-dialog *cky-main-window*
                       nil "No grammar loaded!" nil "OK")
                 (if (not *lexicon*)
                    (pop-up-message-dialog *cky-main-window*
                          nil "No lexicon loaded!" nil "OK")
                    (if (string= "" *current-sentence*)
                       (pop-up-message-dialog *cky-main-window*
                             nil "No previous sentence!" nil "OK")))))))
   


(defun analyse (sent) 
      (cond ((not (and sent *grammar* *lexicon* *parser*)) 
                   (when (equal *lexicon* '(nil)) 
                          (pop-up-message-dialog *cky-main-window*
                                nil "There is nothing in the lexicon" nil "OK")) 
                   (when (null *grammar*) 
                          (pop-up-message-dialog *cky-main-window* 
                                nil "There are no grammar rules" nil "OK"))) 
                  (t 
                     (format-display *cky-subdisp*
                          " ~%~% STARTING PARSER ~%~%") 
                     (setq *chart* nil) 
                     (let ((CkyTable nil)) 
                         (parse sent) 
                         (setq CkyTable (cky-answer)) 
                         (when *show-chart*
                                (further-inform "Parsing Completed.")) 
                         (when *verbose* 
                                (format-display *cky-subdisp* 
                                     "~%~%NUMBER OF PARSES: ~D~%~%" (length CkyTable)))
                         (when (> (length CkyTable) 0)
                                (when *show-trees* (prtree CkyTable)))
                         (chartstats))))) 
 
 

(setq *cky-lexicon-menu* 
      (make-menu-item :title "~Lexicon" :value
             (open-menu 
                   (list 
                      (make-menu-item :title "New Lexicon"
                             :name ':new-lexicon
                             :value 'new-cky-lexicon :available-p t)
                      (make-menu-item :title "Load Lexicon..."
                             :name ':load-lexicon
                             :value 'load-cky-lexicon :available-p t)
                      (make-menu-item :title "Save Lexicon"
                             :name ':save-cky-lexicon
                             :value 'save-cky-lexicon :available-p nil)
                      menu-separator
                      (make-menu-item :title "New Lexical Entry"
                             :name ':new-word
                             :value 'new-word :available-p nil)
                      (make-menu-item :title "Delete Word"
                             :name ':delete-word
                             :value 'delete-word :available-p nil)
                      (make-menu-item :title "Find Word"
                             :name ':find-word
                             :value 'find-word :available-p nil))
                   'pop-up-menu *lisp-main-window* :name '*cky-lexicon-menu* :title '"" 
                   :selection-function 'funcall-menu-item)))
                       
                      
(defun new-cky-lexicon ()
     (catch 'cancel
            (when *lexicon*
                   (when (not (window-close *lexicon-dialog*))
                          (throw 'cancel nil)))
     (let ((lfile (concatenate 'string *pail-directory* "cky\\lexicon\\untitled.lex")))
         (setq *current-lex-file* lfile)
         (setq *lexicon-filename* lfile)
         (setq *lexicon* nil)
         (viewlex)
         (set-lexicon-menu-after-load)
         )))
                       
(defun load-cky-lexicon () 
      (catch 'cancel
            (when *lexicon*
                   (when (not (window-close *lexicon-dialog*))
                          (throw 'cancel nil)))
            (let ((file 
                       (ask-user-for-existing-pathname
                        "Lexicon to load"
                        :stream  *cky-main-window*
                        :allowed-types '(("Lexicon Files" . "*.lex")("All Files"  . "*.*"))
                        :host (concatenate 'string *pail-directory* "cky\\lexicae\\")
                        )))
                (when file
                       (progn
                            (loadlex file)
                            (set-lexicon-menu-after-load))))))


(defun m-new-word (item) 
      (declare (ignore item)) 
      (new-word)) 
 


(defun m-delete-word (item) 
      (declare (ignore item)) 
      (delete-word)) 
 


(defun m-find-word (item) 
      (declare (ignore item)) 
      (find-word)) 


 
(defun delete-word () 
      (if (dialog-item-value (wsequence *lexicon-dialog*))
         (let ((rules (delete-from-sequence
                               *lexicon-dialog*
                               (dialog-item-value (wsequence *lexicon-dialog*)))))
             (when rules
                    (setf *lexicon* (sequence-to-lexicon rules))))
         (pop-up-message-dialog *cky-main-window* nil
               "You must select a item first!" nil "OK")))
 

 
(defun find-word () 
      (multiple-value-bind (word other value)
              (ask-user-for-string
                    "Which word would you like to find?"
                    nil "Find" "Cancel")
             (when (and (string= value "Find") (string/= word ""))
                    (let ((entries (find-entries (string-upcase word)))) 
                        (if entries
                           (list-entries word entries) 
                           (pop-up-message-dialog *cky-main-window*
                                 nil
                                 (concatenate 'string
                                       "Sorry, no entries for " 
                                       word ".")
                                 nil "OK"))))))
 


(defun new-word () 
      (multiple-value-bind (word other value)
              (ask-user-for-string
                    "Please, enter new word?"
                    nil "Add" "Cancel")
             (when (and (string= value "Add") (string/= word ""))
                    (new-entry word))))
 


(defun m-save-cky-lexicon (item) 
      (declare (ignore item)) 
      (save-cky-lexicon)) 
 


(defun save-cky-lexicon () 
      (when *lexicon-dialog* 
             (when (save-lexicon *lexicon-dialog*) 
                    (setf (stream-title *lexicon-dialog*) 
                              (concatenate 'string 
                                    "Lexicon : "  
                                    (file-namestring *lexicon-filename*))))))
        
       
 
(setq *cky-grammar-menu* 
      (make-menu-item :title "~Grammar" :value
             (open-menu 
                   (list 
                      (make-menu-item :title "New Grammar"
                             :name ':new-grammar
                             :value 'new-cky-grammar :available-p t)
                      (make-menu-item :title "Load Grammar"
                             :name ':load-grammar
                             :value 'load-cky-grammar :available-p t)
                      (make-menu-item :title "Save Grammar"
                             :name ':save-cky-grammar
                             :value 'save-cky-grammar :available-p nil)
                      menu-separator
                      (make-menu-item :title "New Grammar Rule"
                             :name ':new-grammar-rule
                             :value 'new-grammar-rule :available-p nil)
                      (make-menu-item :title "Delete Grammar Rule"
                             :name ':delete-grammar-rule
                             :value 'delete-grammar-rule :available-p nil)
                      (make-menu-item :title "Expand Category"
                             :name ':expand-category
                             :value 'expand-category :available-p nil))
                   'pop-up-menu *lisp-main-window* :name '*cky-grammar-menu* :title '"" 
                   :selection-function 'funcall-menu-item)))
 


(defun new-cky-grammar ()
      (catch 'cancel
            (when *grammar*
                   (when (not (window-close *grammar-dialog*))
                          (throw 'cancel nil)))
            (let ((file (concatenate 'string *pail-directory* "cky\\grammar\\untitled.grm")))
                (when file
                       (progn
                           (setq *grammar-filename* file)
                           (setq *grammar* nil)
                           (view-grammar)
                           (set-grammar-menu-after-load))))))

(defun load-cky-grammar ()
      (catch 'cancel
            (when *grammar*
                  (when (not (window-close *grammar-dialog*))
                        (throw 'cancel nil)))
            (let ((file 
                       (ask-user-for-existing-pathname
                        "Grammar to load"
                        :stream  *cky-main-window*
                        :allowed-types '(("Grammar Files" . "*.grf")("All Files"  . "*.*"))
                        :host (concatenate 'string *pail-directory* "cky\\grammars\\")
                        )))
                 (when file
                       (progn
                            (loadsyn file)
                            (set-grammar-menu-after-load))))))
 


(defun save-cky-grammar () 
      (when *grammar-dialog* 
             (when (save-grammar *grammar-dialog*) 
                    (setf (stream-title *grammar-dialog*)
                              (concatenate 'string 
                                    "Grammar : "  
                                    (file-namestring *grammar-filename*))))))
     
 

(defun new-grammar-rule (&optional item) 
      (multiple-value-bind (lhs other value)
              (ask-user-for-string
                    "Left hand category for the new rule?"
                    "" "Add" "Cancel")
             (when (and (string= value "Add") (string/= lhs ""))
                    (multiple-value-bind (rhs other value)
                            (ask-user-for-string
                                  (concatenate 'string 
                                        "Enter the right-hand side now... " 
                                        lhs 
                                        " ->")
                                  "" "Add" "Cancel")
                           (when (and (string= value "Add") (string/= rhs ""))
                                  (makesyn lhs rhs))))))
 


(defun delete-grammar-rule (&optional item) 
      (if (dialog-item-value (wsequence *grammar-dialog*))
         (let ((rules (delete-from-sequence
                               *grammar-dialog*
                               (dialog-item-value (wsequence *grammar-dialog*)))))
             (when rules
                    (setf *grammar* (sequence-to-grammar rules))))
         (pop-up-message-dialog *cky-main-window* nil
               "You must select a item first!" nil "OK")))
 


(defun expand-category ())       
 


(setq *cky-demo-menu* 
      (make-menu-item :title "~Demos" :value
             (open-menu 
                   (list 
                      (make-menu-item :title "Demo 1"
                             :value 'cky-demo-1 :available-p t)
                      (make-menu-item :title "Demo 2"
                             :value 'cky-demo-2 :available-p t))
                   'pop-up-menu *lisp-main-window* :name '*cky-file-menu* :title '"" 
                   :selection-function 'funcall-menu-item)))
 


 
(defun cky-demo-1 () 
      (start-demo-1)) 

(defun cky-demo-2 () 
      (start-demo-2)) 
 

 
                                                    
(defun set-cky-menubar ()
     (clean-up-menubar)
;;;      (setq *lisp-menu-bar*
;;;           (open-stream 'menu-bar *lisp-main-window* :io))
     (add-to-menu *lisp-menu-bar*
          (make-menu-item
                :title '~Tools
                :value *pail-tool-menu*))
     (add-to-menu *lisp-menu-bar*
          *cky-file-menu*)
     (add-to-menu *lisp-menu-bar*
          (make-menu-item
                :title '~Edit
                :value *pail-edit-menu*))
     (add-to-menu *lisp-menu-bar*
          *cky-parser-menu*)
     (add-to-menu *lisp-menu-bar*
          *cky-grammar-menu*)
     (add-to-menu *lisp-menu-bar*
          *cky-lexicon-menu*)
     (add-to-menu *lisp-menu-bar*
          *cky-demo-menu*)
     (add-to-menu *lisp-menu-bar*
          *pail-window-menu*)  
     )
 
 
 
 