;;; -*- Mode: LISP; Package: cky; Syntax: Common-lisp; -*-

(in-package :cky)


;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   cky-lexicon-gui.cl
;;; Short Desc: general user interface for the lexicon routines
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.2.92 - FB
;;; Author(s):  Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Rod Johnson wrote a character oriented version of the interface
;;; Fabio Baj  wrote  the graphic user interface
;;; --------------------------------------------------------------------------

(defun loadlex (filename)
      (format-display *cky-msg-disp*
           "Loading lexicon from ~A~%" filename)
      (when filename
             (with-open-file (infile filename :direction :input) 
                  (setq *lexicon-filename* filename)
                  (setq *lexicon* nil)
                  (do ((entry (read infile nil 'eof)
                                (read infile nil 'eof)))
                         ((eq 'eof entry) (setq *lexicon* (nreverse *lexicon*)))
                        (when entry (push entry *lexicon*)))
                  (viewlex))))
	  



(defmethod perform-save-lexicon ((clw cky-lexicon-window) filename)
      (with-open-file
           (port filename
            :direction :output
            :if-exists :supersede
            :if-does-not-exist :create)  
           (format-display *cky-msg-disp*
                "Saving lexicon to ~A~%" filename)
           (setq *lexicon-filename* filename)
           (setf (save-p (wsequence clw)) nil)
           (mapcar #'(lambda (entry)
                                    (let ((second 0))
                                        (format  port "(~a  ~a)~%"
                                             (multiple-value-bind (a b)
                                                     (read-from-string entry)
                                                    (setq second b)
                                                    a)
                                             (read-from-string entry nil 'done :start (+ 2 second)))))
                  (dialog-item-range (wsequence clw)))))

 
  

(defun lookup (word)
      (let ((entries (search-word (read-from-string word) *lexicon*)))
          (if (not entries)
             (progn
                  (when (yes-or-no-p
                                     (format nil 
                                          "   HELP! '~A' is not in the lexicon:~% is it a spelling error?"
                                          word))
                         (multiple-value-bind (new-word other value)
                                 (ask-user-for-string
                                       "Please, give the correct form"
                                       word
                                       "Cancel" "OK")
                                (when (eql value "OK")
                                       (progn
                                            (new-entry word)
                                            (lookup (read-from-string new-word)))))))
             (progn
                  (let ((result nil))
                      (dolist (item entries)
                            (push (list (cadr item))
                                  result))
                      (reverse result))))))



(defun search-word (word lexicon)
      (let ((result nil))
          (dolist (item lexicon)
                (when (eq (car item) word)
                       (push item result)))
          (when result (reverse result))))



#|

(defun sort-lexicon (lexicon)
      (sort  lexicon (function (lambda (x y)
                                                       (string< x y)))))
  
|#



;;; =============================================================
;;; ATTENZIONE: qui veniva usata la funzione my-read

(defun getfeatures (features)
      (cond ((string= features "")  nil)	
                  (t (car (word-cky (coerce   features 'list))))))



;;; =============================================================

(defun viewlex ()
      (let ((wordlist *lexicon*))
          (when *lexicon-filename*
                 (create-cky-lexicon-window
                      (lexicon-to-sequence wordlist)))))



(defun create-cky-lexicon-window (sequence)
      (setq *lexicon-dialog*
            (open-dialog ()
                  'cky-lexicon-window *cky-main-window* :pop-up-p nil
                  :background-color (make-rgb :red 0 :green 128 :blue 128) 
                  :user-closable nil :user-scrollable nil :user-resizable t
                  :sequence sequence
                  :button-menu-bar '(
                                                       ("Close" 'close-cky-lexicon-window)
                                                       ("Save" 'm-save-cky-lexicon)
                                                       ("Find" 'm-find-word)
                                                       ("Add" 'm-new-word)
                                                       ("Delete" 'm-delete-word))
                  :title (concatenate 'string
                                 "Lexicon : " 
                                 (file-namestring *lexicon-filename*))
                  :visible-box (make-box 0 0 0 0))))
            


(defun close-cky-lexicon-window (item)
      (when (window-close *lexicon-dialog*)
             (progn
                  (setq *lexicon* nil)
                  (set-lexicon-menu-before-load)
                  (setq *lexicon-dialog* nil)))
      t)



(defun find-entries (word)
      (let* ((current-word-list (when *lexicon-dialog*
                                                          (dialog-item-range (wsequence *lexicon-dialog*))))
                (result nil))
          (when *lexicon-dialog*
                 (dolist (item current-word-list (reverse result))
                       (when (string= word item :end2 (length (coerce word 'list)))
                              (push item result))))))



(defun new-entry (word)
      (when word
             (multiple-value-bind (category other value)
                     (ask-user-for-string "Category?" nil "Cancel" "OK")
                    (when (string= value "OK")
                           (let* ((table (wsequence *lexicon-dialog*))
                                     (cells (ordered-insert (show-entry (list (string-upcase word)
                                                                                                           (string-upcase category)))
                                                     (dialog-item-range table))))
                               (print cells)
                               (set-dialog-item-range table cells)
                               (setf (save-p table) t)
                               (ordered-insert1 (list (read-from-string word)
                                                                   (read-from-string category)) *lexicon*))))))
            



(defun list-entries (word lista)
      (when lista
             (let ((item (select-item-from-list
                                      *cky-main-window* 
                                      lista
                                      (concatenate 'string "Entries for " word))))
                 (when item
                        (set-dialog-item-value (wsequence *lexicon-dialog*) item)))))
                         
  

(defun select-entries-to-delete (lista)
      (when lista
             (select-item-from-list
                  *cky-main-window* 
                   lista 
                   "Select entry to delete:")))
  

                           
(defun delete-entries (lista)
      (when lista
             (let ((table (wsequence *lexicon-dialog*)))
                 (set-dialog-item-range table
                       (my-delete lista (dialog-item-range table)))
                 (reset-table table)
                 (setf *lexicon*
                           (sequence-to-lexicon (dialog-item-range table))))))


    
(defun lexicon-to-sequence (lexicon)
      (mapcar (function show-entry)
             lexicon))       



(defun show-entry (entry)
      (if (cddr entry)
         (format nil  "~A : ~A  ~A"
              (car entry)
              (cadr entry)
              (cddr entry))
         (format nil  "~A : ~A"
              (car entry)
              (cadr entry))))



(defun sequence-to-lexicon (sequence)
      (let ((result nil)
              (second 0))
          (dolist (entry sequence)
                (push (list (multiple-value-bind (a b)
                                            (read-from-string entry)
                                           (setq second b)
                                           a)
                               (read-from-string entry nil 'done :start (+ 2 second)))
                      result))
          (reverse result)))



(defun set-lexicon-menu-after-load ()
      (set-menu-item-available-p
           (find-named-object ':load-lexicon *cky-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':load-lexicon *cky-file-menu*) nil)
      (set-dialog-item-available-p
           (find-named-object ':load-lexicon *out-dialog*) nil)
      (set-menu-item-available-p
           (find-named-object ':save-cky-lexicon *cky-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':new-word *cky-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':delete-word *cky-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':find-word *cky-lexicon-menu*) t))
      


(defun set-lexicon-menu-before-load ()
      (set-menu-item-available-p
           (find-named-object ':load-lexicon *cky-lexicon-menu*) t)
      (set-menu-item-available-p
           (find-named-object ':load-lexicon *cky-file-menu*) t)
      (set-dialog-item-available-p
           (find-named-object ':load-lexicon *out-dialog*) t)
      (set-menu-item-available-p
           (find-named-object ':new-word *cky-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':delete-word *cky-lexicon-menu*) nil)
      (set-menu-item-available-p
           (find-named-object ':find-word *cky-lexicon-menu*) nil))

