;;; -*- Mode: LISP; Package: csp; Syntax: Common-lisp;      -*-


(in-package :csp)



(defun line-match (l1 l2)
      (every #'(lambda (x) x)
            (mapcar #'fit l1 (reverse l2))))


(defun fit (x y)
      (= 0 (+ x y)))



(defun can-stay-aside (side1 side2 piece1 piece2)
  (line-match
     (nth side1 (pattern piece1))
     (nth side2 (pattern piece2))))



(defun flip (pair)
      (cons (cdr pair) (car pair)))



(defun pattern (piece)
      (cdr piece))



;;----------------------------------
;;
;; A Piece is a list like '(NAME (1  1)(1  -1)(-1  0)(0 0))
;;
;; for a piece '(Name a b c d) we must generate as possible values also
;; (Name b c d a)(Name c d a b)(Name d a b c) representig all possible rotations
;; of a piece



(defun out-of-range (x y dim)
  (or
        (= 0 x)
        (= 0 y)
        (> x dim)
        (> y dim)))



(defun pos-neigh (x y dim)
      (remove-if
           #'(lambda (pair)
                   (out-of-range (car pair) (cdr pair) dim))
           (list
              (cons (1+ x) y)
              (cons x (1+ y))
              (cons (1- x) y) 
              (cons x (1- y)))))



(defun side-size (pieces)
      (round (sqrt (length pieces))))



;;---------------------------------------------

(defun generate-p1 (dimension)
      (format nil
           "
(defun p-1 (var1 try1)
      (cond ~{~a~}
                  (t t)))"
           (generate-p1-cond dimension)))



(defun generate-p1-cond (dimension)
      (let ((result nil))
          (do ((row 1 (+ row 1)))
                 ((= row (1+ dimension)))
                (do ((col 1 (+ col 1)))
                       ((= col (1+ dimension)))
                      (if (is-angle row col dimension)	   
                         (push (p1-angle-clause row col) result)
                         (if (is-side row col dimension)
                            (push (p1-side-clause row col)  result)))))
          result))



(defun p1-angle-clause (row col)
      (format nil
           "
((eq var1 'pos~A-~a) (must-be-angle ~A ~A try1 ))" 
           row col row col))



(defun p1-side-clause (row col)
      (format nil
           "
((eq var1 'pos~A-~a) (must-be-side ~A ~A try1 ))" 
           row col row col ))



(defun is-angle (row col dim)
      (cond
                  ((and (eq row dim) (eq col dim)) 1)
                  ((and (eq row 1) (eq col dim)) 0)
                  ((and (eq row dim) (eq col 1)) 2)
                  ((and (eq row 1) (eq col 1)) 3)))



(defun is-side(r c d)
      (cond
                  ((eq r 1) 0)
                  ((eq r d) 2)
                  ((eq c 1) 3)
                  ((eq c d) 1)))



(defun find-angle (row col)
      (cond
                  ((and (> row 1) (> col 1)) 1)
                  ((and (eq row 1) (> col 1)) 0)
                  ((and (> row 1) (eq col 1)) 2)
                  ((and (eq row 1) (eq col 1)) 3)))



(defun find-side (r c)
      (cond
                  ((eq r 1) 0)
                  ((eq c 1) 3)
                  ((> r c) 2)
                  ((> c r) 1)))



(defun must-be-side (row col try)
      (let ((side (find-side row col)))
          (equal (nth side (pattern try)) '(0  0))))



(defun must-be-angle(row col try)
      (let ((side (find-angle row col)))
          (and
                   (equal (nth side (pattern try)) '(0  0))
                   (equal (nth
                                     (remainder (1+ side) 4)
                                     (pattern try)) '(0  0)))))



(defun remainder (x y)
      (multiple-value-bind (q r)
              (floor x y) r))



;;---------------------------------------------

(defun generate-p2 (dimension)
      (format nil
           "
(defun p-2 (var1 try1 var2 try2)
      (cond ~{~a~}))"
      (generate-p2-cond dimension)))



(defun generate-p2-cond (dimension)
      (let ((result nil))
          (do ((row 1 (+ row 1)))
                 ((= row (1+ dimension)))
                (do ((col 1 (+ col 1)))
                       ((= col (1+ dimension)))
                      (dolist (pair (pos-neigh row col dimension))
                            (let* ((nrow (car pair))
                                      (ncol (cdr pair))
                                      (side1 (common-side row col nrow ncol))
                                      (side-nb (common-side  nrow ncol row col )))
                                (push
                                      (format nil
                                           "
((and (eq var1 'pos~A-~a) (eq var2 'pos~A-~a))
       (can-stay-aside ~A ~A try1 try2)) " 
                                           row col nrow ncol side1 side-nb)
                                      result)))))
            result))



(defun generate-net-def (dimension)
      (format nil
           "
(construct-network
      '( ~{~A~}))"
           (generate-net-def-node dimension)))



(defun generate-net-def-node (dimension)
      (let ((result nil))
          (do ((row 1 (+ row 1)))
                 ((= row (1+ dimension)))
                (do ((col 1 (+ col 1)))
                       ((= col (1+ dimension)))
                      (let* ((nbs (pos-neigh row col dimension)))
                          (push
                                (format nil
                                     "
(pos~A-~a DOM ~A)"
                                    row col (nbs-string nbs))
                                result))))  
            result))


       
(defun nbs-string (nbs)
      (format nil 
           "~{~A ~}"
           (mapcar #'nb-to-string nbs)))



(defun nb-to-string (nb)
      (format nil
           "pos~A-~A"
           (car nb)
           (cdr nb)))



;;;-------------------------------    

(defun rotate-n (list n)
      (let ((res list))
          (dotimes (x n res) 
                (setq res (rotate res)))))



(defun rotate (list)
      (append (cdr list) (list (car list))))

(defun common-side (r1 c1 r2 c2)
      (cond
                  ((> r1 r2) 0)
                  ((> c1 c2) 3)
                  ((< r1 r2) 2)
                  ((< c1 c2) 1)))


    
;;;-------------------------------  

(defun generate-domain (pieces)
      (mapcan #'all-rotations pieces))



(defun all-rotations (piece)
      (mapcar
             #'(lambda(r)
                     (cons
                           (car piece)
                           (rotate-n (pattern  piece) r)))
             '(0 1 2 3)))



;;---------------------------------------
;; Here we redefine the equal-value for pieces,
;; to be used to remove values from domains when a unique
;;value has been determined for a variable.
;; When a a places has be assigned with a piece, with a particular rotation
;; we must impose that this pieces AND ALL ITS DIFFERENT ROTATIONS must be
;; deleted from the possible values of the other places.
;; So the equal-value is redefined to consider equal two values
;; by considering only the pattern, and not the rotation.



(defun equal-value (v1 v2)
      (equal  (car v1) (car v2)))



;;--------------------------------------
;;
;; MAIN FUnction

(defun construct-network-from-pieces(dim)
      (compile (eval (read-from-string (generate-p2 dim))))
      (compile (eval (read-from-string (generate-p1 dim))))
      (eval (read-from-string (generate-net-def dim))))