(in-package :csp)



(defclass time-problem ()
	  ((days     :initarg :days    :accessor days    :initform nil)
	   (timings     :initarg :timings     :accessor timings     :initform nil)
	   (classrooms     :initarg :classrooms    :accessor classrooms    :initform nil)
	   (subjects     :initarg :subjects    :accessor subjects    :initform nil)
	   (syllabus     :initarg :syllabus    :accessor syllabus    :initform nil)
	   (time-variables    :accessor time-variables    :initform nil)
	   (time-values  :accessor time-values    :initform nil)))



(defclass time-variable ()
	  ((time-var-name     :initarg :time-var-name    :accessor time-var-name    :initform nil)
	   (day     :initarg :day    :accessor day    :initform nil)
	   (time-id     :initarg :time-id    :accessor time-id    :initform nil)
	   (classroom     :initarg :classroom    :accessor classroom    :initform nil)
	   (unary-constr     :initarg :unary-constr    :accessor unary-constr    :initform nil)
	   (neighbors     :initarg :neighbors    :accessor neighbors    :initform nil)))
	   


(defclass time-value ()
	  ((subject     :initarg :subject    :accessor subject    :initform nil)
	   (number-id     :initarg :number-id    :accessor number-id    :initform nil)
	   (classroom     :initarg :classroom    :accessor classroom    :initform nil)))



(defclass neigh-entry ()
	  ((neigh-name :initarg :neigh-name   :accessor neigh-name   :initform nil)
	   (bin-constraints :initarg :bin-constraints   :accessor bin-constraints     :initform nil)))



;;==============================================================

(defmethod generate-variables ((time-p time-problem))
      (dolist (d (days time-p))
            (dolist (tm (timings time-p))
                  (dolist (cl (classrooms time-p))
                        (push (generate-variable d tm cl)
                              (time-variables time-p))))))



(defun generate-variable (d tm cl)
      (make-instance 'time-variable
             :time-var-name (construct-time-name d tm cl)
             :classroom cl
             :day d
             :time-id tm))



(defun construct-time-name (d tm cl)
      (atom-conc (list d '-  tm '- cl)))



;;==============================================================

(defmethod generate-values ((time-p time-problem))
      (dolist (sy (syllabus time-p))
            (dotimes (numb (syllabus-amount sy))
                  (push
                        (make-instance 'time-value 
                               :subject (syllabus-subject sy)
                               :number-id numb
                               :classroom (syllabus-classroom sy ))
                        (time-values time-p)))))



(defun syllabus-subject (sy)
          (nth 0 sy))

(defun syllabus-classroom  (sy)
      (nth 1 sy))

(defun syllabus-amount (sy) 
      (nth 2 sy))



;;==============================================================

(defmethod  add-non-overlap-constraints((time-p time-problem))
      (add-non-overlap-c (time-variables time-p)))



(defun add-non-overlap-c(varlist)
      (if varlist
         (let ((first-var (car varlist)))
             (mapcar 
                    #'(lambda (var) 
                            (when (contemporary-p first-var var)
                                   (add-neigh var first-var)
                                   (add-constr var first-var 'not-eq)
                                   (add-neigh first-var var)
                                   (add-constr first-var var 'not-eq)))
                    (cdr varlist))
             (add-non-overlap-c (cdr varlist)))))
        


(defmethod add-neigh ((v1 time-variable) (v2 time-variable))
      (if (not (find  v1 (neighbors v2) :test #'var-present))
         (push
               (make-instance 'neigh-entry
                      :neigh-name (time-var-name v1))
               (neighbors v2))))



(defmethod add-constr ((v1 time-variable) (v2 time-variable) constraint-id)
      (let ((nb-cond (find  v1 (neighbors v2) :test #'var-present)))
          (push 
                (generate-constr v1 v2  constraint-id)
                (bin-constraints nb-cond))))



(defmethod generate-constr ((v1 time-variable) (v2 time-variable) constraint-id)
      (case constraint-id 
            ((not-eq) `(not (equal (subject value1) (subject  value2))))))


        
(defmethod contemporary-p ((tv1 time-variable ) (tv2 time-variable))
      (and (equal (day tv1) (day tv2))
               (equal (time-id tv1) (time-id tv2))
               (not (equal (classroom tv1) (classroom tv2)))))



(defmethod var-present ((v time-variable) (nb neigh-entry))    
      (equal (time-var-name v) (neigh-name nb)))



;;===========================================================
;; Generation of a CSProblem from a timetable problem
;;===========================================================

(defmethod generate-net-definition ((time-p time-problem))
      (list  'construct-network 
         (list 'quote (mapcar
                                     #'generate-node-definition
                                     (time-variables time-p)))))



(defmethod generate-node-definition ((tv time-variable))
      (append (list (time-var-name tv) 'Teachers )
            (mapcar #' neigh-name (neighbors tv))))
  


;;==============================================================

(defmethod p2-generator ((time-p time-problem))
      (list 'defun 'p-2 (list 'var1 'value1 'var2 'value2)
         (cons 'cond
               (mapcan #'p2-generator (time-variables time-p)))))



(defmethod p2-generator ((tv time-variable))
      (mapcar
             #'(lambda (nb) 
                     (p2-generator-cond tv nb))
             (neighbors tv)))


       
(defmethod p2-generator-cond ((tv time-variable)(nb neigh-entry))
      (list 
         (list
            'and    
            (list  'eq 'var1 (list 'quote (time-var-name tv)))
            (list  'eq 'var2 (list 'quote (neigh-name nb))))
         (cons 'and (bin-constraints nb))))



;;==============================================================		

(defun p-1 (var val)
      (right-classroom var val))



(defun right-classroom (var val)
      (let ((time-var (find var (time-variables mtp) :key #'time-var-name)))
          (equal (classroom val) (classroom time-var))))



(defun variable-domain (x)
      (time-values mtp))



(defun domain-dependent-show-solution(solution)
      (let ((reordered (mapcar
                                            '(lambda(cg)
                                                 (mapcar
                                                        '(lambda(og)
                                                             (ordina-per-giorni og))
                                                        cg))
                                            (mapcar
                                                   'ordina-per-ore
                                                   (mapcar 'raggruppa-per-ore
                                                          (raggruppa-per-classi (time-variables mtp)))))))
          (dolist (class-g reordered)
                (class-message class-g)
                (dolist (hour-g class-g)
                      (print-assigned-subjects hour-g solution)))))



(defun print-assigned-subjects (hg solution)
      (format t 
           "| ~2D  "
           (time-id (car hg) ))
      (dolist (tv hg)
            (format t
                 "|  ~A  "
                 (assigned-subj (time-var-name tv) solution)))
      (format t "|")
      (nl))



(defun assigned-subj (vn solution)
      (subject (car (possible-values (find-node vn solution)))))
 	


(defun class-message (cg)
      (nl) (nl)
      (dolist (tv (cons nil (car cg)))
            (format t "--------")) (nl)
      (format t "          ~A~%"   (classroom (caar cg)))
      (dolist (tv (cons nil (car cg)))
            (format t "--------")) (nl)
      (format t "      ")
      (dolist (tv (car cg))
            (format t "|  ~A  " (day tv)))
      (format t "|") (nl)  (format t"------")
      (dolist (tv (car cg))
            (format t "--------")) (nl))
	

(defun nl ()
      (format t "~%"))     



(defun raggruppa-per-classi (lista)
      (raggruppa lista nil :key 'classroom))



(defun raggruppa-per-ore (lista)
      (raggruppa lista nil :key 'time-id))



(defun ordina-per-giorni(lista)
      (sort lista #'atom< :key #'(lambda (x) (day x))))



(defun ordina-per-ore(lista)
      (sort lista #'< :key #'(lambda (x) (time-id (car  x)))))



(defun atom< (x y)
      (string< (format nil "~A" x) (format nil "~A" y)))



(defun raggruppa (lista &optional (res nil) &key (key 'id ))
      (cond ((null lista) res)
                  (t (let ((gruppo (find
                                                  (funcall key (car lista))
                                                  res
                                                  :key #'(lambda (group)
                                                                   (funcall key (car group))))))
                          (if gruppo
                             (progn
                                  (push (car lista) gruppo)
                                  (setq res (remove
                                                          gruppo res
                                                          :test #'(lambda(x y)
                                                                           (equal (funcall key(car  x))
                                                                                 (funcall key (car y))))))
                                  (push gruppo res))
                             (push (list (car lista)) res))
                          (raggruppa (cdr lista) res :key key )))))



;;==============================================================
;;                      T E S T
;;==============================================================

; (setq tv (car (time-variables mtp)))
; (setq tv2 (caddr (time-variables mtp)))

;(describe tv)
;(setq nb (car (neighbors tv)))
;(describe nb)


(defmethod print-object ((tv time-value) stream)
      (format stream "~a" (subject tv)))