;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   ga-functions.cl
;;; Short Desc: A simple set of functions to be used in the Genetic Algorithm
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   May 1991
;;; Author:     Nick Almassy
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History:
;;;        14.2.92  Comments describing how to do new functions   -dta
;;; Files required: ga.cl	  (this-file)
;;;                 ga-dialog.cl  (the window Interface)
;;;		    acl-gin	  (the window-functions)
;;; --------------------------------------------------------------------------
;;; User defined fitness functions
;;; --------------------------------------------------------------------------
;;; PAIL users may define their own optimization functions.  It is
;;; recommended that they be defined in this file, so that they need
;;; not be concerned with package problems.  Defining functions
;;; requires some limited LISP competence.
;;;
;;; A fitness function takes a single argument, which is a list of
;;; parameters.  The parameters themselves can be referenced with the
;;; LISP functions first, second, third, etc.  These will always be
;;; numeric values.  For the predefined examples below, the range of
;;; the parameters is specified by a call to (make-instance
;;; 'ga-function ...).  This call is not necessary for user defined
;;; functions, since the range of the parameters can also also be
;;; defined interacively in the GA module with the "New Function"
;;; button (see the Help in the Parameter window for more information
;;; on the "New Function" button).
;;;
;;; Thus the user needs only to define the new function in this file,
;;; then select it with the "New Function" button in the "GA:Parameter"
;;; window.  The value returned by the function must be a single
;;; numeric value which will be optimized by the Genetic Algorithm.
;;;
;;; If the range of the results of this function is known it is useful
;;; to enter these values either interacively in the "GA:New Fitness
;;; Function" window or with the method of defining the ga-function
;;; object.  By setting these values propperly the evolution can
;;; precisely be observed during runtime in the "GA:Monitor" window.
;;; --------------------------------------------------------------------------




;;; ==========================================================================
;;; Predefined Test functions:
;;; ==========================================================================
;;;
;;; ----------intervall-----dim---bits/p---gray------name-------------
;;;
;;; SQR    0.0   ..  1.0     1      5        n     Square
;;; DSQR   0.0   ..  1.0     1      5        n     Deceptive Square
;;;
;;; --------------- De Jong Five-Functions Test Bed ------------------
;;;
;;; F1    -5.12  ..  5.12    3     10        y     Paraboloid
;;; F2    -2.048 ..  2.048   2     12        n     Rosenbrock's ridge
;;; F3    -5.12  ..  5.12    5     10        n     Step function
;;; F4    -1.28  ..  1.28   30      8        n     x**4 with noise
;;; F5   -65.536 .. 65.536   2     17        n     Shekel's Foxholes
;;;
;;; ------------------------------------------------------------------


(defparameter *known-functions* nil)


; ==========================================================================

(defun f1 (paramlist)
  (+ (expt (first paramlist) 2)
     (expt (second paramlist) 2)
     (expt (third paramlist) 2)))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :lower-bound -5.12
			   :upper-bound 5.12
			   :nparameter 3
			   :bits-per-parameter 10
			   :fitness-function 'f1
			   :min-f 0		; At (0 0 0)
			   :max-f 78.643196)	; At (5.12 5.12 5.12)
			 *known-functions*))

;;; ==========================================================================

(defun f2 (paramlist)
  (+ (* 100.0 (expt (- (expt (first paramlist) 2)
		       (second paramlist)) 2))
     (expt (- 1 (first paramlist)) 2)))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :lower-bound -2.048
			   :upper-bound 2.048
			   :nparameter 2
			   :bits-per-parameter 12
			   :fitness-function 'f2
			   :min-f 0		; At (1 1)
			   :max-f 3905.9272)	; really At (-2.048 -2.048) bug?
			 *known-functions*))

;;; ==========================================================================

(defun f3 (paramlist)
  (+ 30 (eval (cons '+ (mapcar #'floor paramlist)))))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :lower-bound -5.12
			   :upper-bound 5.12
			   :nparameter 5
			   :bits-per-parameter 10
			   :fitness-function 'f3
			   :min-f 0	; At (-5.12 -5.12 -5.12 -5.12 -5.12 )
			   :max-f 55)	; At (5.12 5.12 5.12 5.12 5.12)
			 *known-functions*))

;;; ==========================================================================

(defun f4 (paramlist &optional (i 1))
  (if paramlist
      (+ (* (expt (car paramlist) 4) i)
	 (f4 (cdr paramlist) (1+ i)))
    (noise)))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :lower-bound -1.28
			   :upper-bound 1.28
			   :nparameter 30
			   :bits-per-parameter 8
			   :fitness-function 'f4
			   :min-f 0		; At xj=0 for all j, noise = 0
			   :max-f 1248.2247)	; At xj=1.28 for all j, noise = 0
			 *known-functions*))

;;; ==========================================================================

(defparameter *a* (make-array '(2 25) :initial-contents
'((-32.0 -16.0   0.0  16.0  32.0 -32.0 -16.0   0.0  16.0  32.0 -32.0 -16.0 0.0 16.0 32.0 -32.0 -16.0  0.0 16.0 32.0 -32.0 -16.0  0.0 16.0 32.0 )
  (-32.0 -32.0 -32.0 -32.0 -32.0 -16.0 -16.0 -16.0 -16.0 -16.0   0.0   0.0 0.0  0.0  0.0  16.0  16.0 16.0 16.0 16.0  32.0  32.0 32.0 32.0 32.0))))

(defun f5 (x)				
  (do ((ans 0.002)
       (j 0 (1+ j))
       fj)
      ((>= j 25) ans)
    (setq fj (1+ j))
    (incf ans (do ((i 0 (1+ i)))
		  ((>= i 2) (/ 1.0 fj))
		(incf fj (expt (- (car (nthcdr i x)) (aref *a* i j)) 6))))))
    
(setf *known-functions* (cons
			 (make-instance 'ga-function
 			   :lower-bound -65.536
			   :upper-bound 65.536
			   :nparameter 2
			   :bits-per-parameter 17
			   :fitness-function 'f5
			   :max-f (/ 1 0.9980038379)	; At (? ?)
			   :min-f 0.002)		; At (? ?)
			 *known-functions*))

;;; ==========================================================================

(defun sqr (x)
  (expt (first x) 2))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :nparameter 1
			   :bits-per-parameter 5
			   :fitness-function 'sqr)
			 *known-functions*))

;;; ==========================================================================

(defun dsqr (x)
  (if (eq (first x) 0)
      1.2
    (expt (first x) 2)))

(setf *known-functions* (cons
			 (make-instance 'ga-function
			   :nparameter 1
			   :bits-per-parameter 5
			   :fitness-function 'dsqr
			   :max-f 1.2)
			 *known-functions*))



;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
