;;; *******************************
;;; *    PORTABLE  AI  LAB   -   IDSIA   *
;;; *******************************
;;;
;;; Filename:      ga.cl
;;; Short Desc:  Genetic Algorithm for Classifier System
;;; Version:         1.0
;;; Status:           Review
;;; Last Mod:      09.01.94
;;; Author:          Roberto Limeres
;;;
;;; Copyright (c) 1994 Instituto Dalle Molle (IDSIA), University  of
;;; Zurich, Swiss Federal Institte of Technology Lausanne
;;;
;;; Permision is granted to any individual  or  institution  to  use,
;;; copy, modify, and distribute this software, provided that this
;;; complete  copyright  and  permission  notice  is   maintained,
;;; intact, in all copies and supporting documentation.
;;;
;;; IDSIA provides this  software  "as  is"  without  expresed  or
;;; implied warranty.
;;;

;;; --------------------------------------------------
;;; Change History:
;;; Files required:       ga.cl                (this file)
;;;                                    declare.lsp    (declaration file)
;;;                                    util.lsp             (random and utility functions)
;;; --------------------------------------------------

(in-package :lcs)


;;; ====================================================
;;; Method GA
;;; Main body of genetic algorithm
;;; ====================================================

(defmethod ga ((p population))
      (clear-parents (population p))
      (let ((mates nil) (mort nil))
          (dotimes (i (round (/ (* *proportion-select* (length (population p))) 2.0)))
                (setf mates (crossover p (select* p) (select* p)))
                (setf mort (crowding p (first mates)))
                (replace (population p) mates
                     :start1 (position mort (population p))
                     :end1 (1+ (position mort (population p)))
                     :start2 0
                     :end2 1)
                (setf mort (crowding p (second mates)))
                (replace (population p) mates
                     :start1 (position mort (population p))
                     :end1 (1+ (position mort (population p)))
                     :start2 1
                     :end2 2))))

(defun clear-parents (p)
     (dolist (cl p)
          (setf (parent1 cl) 0)
          (setf (parent2 cl) 0)))
          
;;; ====================================================
;;; Method SELECT
;;; Selection of parent rules
;;; ====================================================

(defmethod select* ((p population))
      (select1 (population p)
       (rnd (sum-strength p))
       0.0))

(defun select1 (p rand partsum)
      (if (>= (setf partsum (+ partsum (strength (car p))))
                rand)
         (car p)
         (select1 (cdr p) rand partsum)))


;;; ====================================================
;;; Method CROSSOVER
;;; Creation of two new rules by crossover procedure
;;; ====================================================

(defmethod crossover ((p population)
                                              (parent1 classifier)
                                              (parent2 classifier))
      (let ((cp (flip *crossover-probability*))
             (c-xsite (rnd *condition-length*))
             (inheritance (/ (+ (strength parent1) (strength parent2)) 2.0))
             (p1 (1+ (position parent1 (population p))))
             (p2 (1+ (position parent2 (population p)))))
          
          (if cp
             (progn
                  (setq *crossover-events* (1+ *crossover-events*))
                  (list
                     (make-instance 'classifier
                            :condition
                            (mutation *trit* c-xsite
                             (nconc (firstn c-xsite (the-condition parent1))
                                   (copy-list (nthcdr c-xsite (the-condition parent2)))))
                            :action
                            (mutation *bit* *action-length* (copy-list (the-action parent1)))                             
                           :strength
                            inheritance
                           :parent1 p1
                           :parent2 p2)
                     (make-instance 'classifier
                            :condition
                            (mutation *trit* c-xsite
                             (nconc (firstn c-xsite (the-condition parent2))
                                   (copy-list (nthcdr c-xsite (the-condition  parent1)))))
                            :action
                            (mutation *bit* *action-length* (copy-list (the-action parent2)))
                            :strength
                            inheritance
                           :parent1 p1
                           :parent2 p2)))
             (list 
                (make-instance 'classifier
                       :condition
                       (mutation *trit* *condition-length*
                        (copy-list (the-condition parent1)))
                       :action
                       (mutation *bit* *action-length* (copy-list (the-action parent1)))
                       :strength
                       inheritance)
                (make-instance 'classifier
                       :condition
                       (mutation *trit*  *condition-length*
                        (copy-list (the-condition parent2)))
                       :action
                       (mutation *bit* *action-length* (copy-list (the-action parent2)))
                       :strength
                       inheritance)))))

	
;;; ====================================================
;;; Method MUTATION
;;; Aplication of meutation proceure to the child rules
;;; ====================================================

(defun mutation (allele-type n l)
      (if (> n 0)
         (if (flip *mutation-probability*)
            (progn
                 (setq *mutation-events* (1+ *mutation-events*))
                 (cons (rnd-element-1 allele-type (car l))
                       (mutation allele-type (1- n) (cdr l))))
            (cons (car l) (mutation allele-type(1- n) (cdr l))))
         l))


;;; ====================================================
;;; Method CROWDING
;;; Selection of worst and more similar rules 
;;; ====================================================
  
(defmethod crowding ((p population) (child classifier))
      (when (< *crowding-factor* 1) (setf *crowding-factor* 1))
      (let ((nmatch -1) (nmatch-max -1)
              (the-worst nil) (most-similar nil))
          (dotimes (i *crowding-factor* the-worst)
                (push (worstofn p) the-worst))
          (dolist (pop-member the-worst most-similar)
                (setf nmatch (match-count pop-member child))
                (when (> nmatch nmatch-max)
                       (progn
                            (setf nmatch-max nmatch)
                            (setf most-similar pop-member))))))


;;; ====================================================
;;; Method WORSTOFN
;;; Probabilistic selection of worst rules
;;; ====================================================

(defmethod worstofn ((p population))
      (when (< *crowding-subpop* 1) (setf *crowding-subpop* 1))
      (let ((strength -1) (strength-max 1000)
              (candidates nil) (the-worst nil))
          (dotimes (i *crowding-factor* candidates)
                (push (nth (rnd (length (population p))) (population p)) candidates))
          (dolist (candidate candidates the-worst)
                (setf strength (strength candidate))
                (when (< strength strength-max)
                       (progn
                            (setf strength-max strength)
                            (setf the-worst candidate))))))


;;; ====================================================
;;; Method MATCH-COUNT
;;; Count the number of equal contitions betwen two classifiers
;;; ====================================================

(defmethod match-count ((cl1 classifier) (cl2 classifier))
      (+ (compare (the-condition cl1) (the-condition cl2))
          (compare (the-action cl1) (the-action cl2))))

(defun compare (l1 l2 &optional (accu 0))
      (if (or (not l1) (not l2))
         accu
         (compare (cdr l1)
          (cdr l2)
          (if (= (car l1) (car l2)) (1+ accu) accu)))) 
  
  
;;; ====================================================
;;; Output functions (Not here in the end)
;;; ====================================================

(defun format-genotype (l)
      (when l	
             (case (car l)
                   (0 (concatenate 'string "0" (format-genotype (cdr l))))
                   (1 (concatenate 'string "1" (format-genotype (cdr l))))
                   (-1 (concatenate 'string "#" (format-genotype (cdr l)))))))

#|
(defmethod report ((i individual) stream)
      (format stream 
           "~A           ~5,2F          ~A            ~A            ~A~%"
           (format-genotype (genotype i))
           (fitness-value i)
           (when (parent1 i) (format-genotype (genotype (parent1 i))))
           (when (parent2 i) (format-genotype (genotype (parent2 i))))
           (xsite i)))

(defmethod report ((p population) stream)
      (format stream "~%-------------------------------------------------------------------~%")
      (format stream "Statistics of population~%")
      (format stream "-------------------------------------------------------------------~%")
      (format stream "Individual      Fitness        Parent1        Parent2        Xsite~%")
      (dolist (i (population p) nil)
            (report i stream))
      (format stream "-------------------------------------------------------------------~%")
      (format stream "Average-fitness:   ~4,2F~%" (average-fitness p))
      (format stream "Maximum-fitness:   ~4,2F~%" (maximum-fitness p))
      (format stream "Minimum-fitness:   ~4,2F~%" (minimum-fitness p))
      (format stream "-------------------------------------------------------------------~%~%~%"))
|#


