;;; -*- Mode: LISP; Package: pail-lib; Syntax: Common-lisp; -*-
;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   browser.cl
;;; Short Desc: implements class for browsers
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   12.5.92 dta
;;; Author:     DTA
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; Written 4.1.91
;;; Extended 10.6.91 to support some default menues
;;; Extended 22.10.91 to add scrollbars and multiple trees
;;; 12.5.92 Added overall browser functions (title items) 
;;; --------------------------------------------------------------------------
;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;
;;;
(in-package :pail-lib)

(export '(browser 
          wtree
          browser-help
          recompute-browser
          starting-tree
          browser-node-click-event-handler
          push-button))


(defparameter *default-font*		   (make-font :modern nil 10))
(defparameter *default-display-border*     1)
(defparameter *default-display-left*	   10)
(defparameter *default-display-bottom*	   10)
(defparameter *default-display-width*	   400)
(defparameter *default-display-height*     300)
(defparameter *default-display-title*	   "A pAILab Window")

(defvar *tree-button-bar-height* 25)
(defvar *tree-button-height* 20)

(defparameter *default-pop-up-button-width*     30)
(defparameter *default-pop-up-button-size-p*    nil)
(defparameter *default-push-button-size-p*      nil)
(defparameter *default-push-button-width*       30)
(defparameter *default-radio-button-width*      16)


(defmethod height ((d dialog-item))
  (box-height (dialog-item-box d)))

(defmethod width ((d dialog-item))
  (box-width (dialog-item-box d)))


(defclass browser (dialog)
      ((bottom :initarg :bottom :accessor bottom)
       (left :initarg :left :accessor left)
       (width :accessor width)
       (height :initarg :height :initform 0 :accessor height)
       (wtree
           :initarg :wtree
           :accessor wtree
           :initform nil)
       (starting-tree :initarg :starting-tree
           :initform nil
           :accessor starting-tree
           :type tree)	; can also be a list of trees
       (font :initarg :font
           :initform (make-font :modern nil 10)
           :accessor font)
       (offset :initarg :offset
           :initform 0
           :accessor offset
           :type integer)
       (border :initarg :border
            :initform 0
            :accessor border
            :type integer)
       (title :initarg :title
          :initform "Browser "
          :accessor title
          :type string)
       (buttons 
            :initform nil
            :accessor buttons
            :type list)
       (items
          :initform nil
          :accessor items
          :type list)))



(defclass tree-dialog (dialog)
       ((lines :initarg :lines
          :initform nil
          :accessor lines)))



;;; closes the display associated with the browser

(defmethod close-browser ((b browser))
  (window-close b))



;;; Returns the descendants of an item.  For a plain browser, this is
;;; simply the value of the descendants slot.  It could be specialized
;;; to allow a different view of an object.

(defmethod find-descendants ((b browser) item)
  (descendants item))



(defun font-character-height (font)
     (font-size font))



;;; Initialization of a browser involves computing an offset and
;;; border (if none are supplied) from the font information.  The size
;;; of the browser is computed, then the display is opened to that
;;; size.  Finally the browser is displayed.  All keyword args that
;;; were passed to the make-instance call that are not defined by
;;; the browser class are passed on to the make-display.

(defconstant make-display-keyword-arguments
    '(:left :bottom :width :height :borders :icon :title :active
      :x-scrollbar :y-scrollbar :inner-width :inner-height :parent
      :frame-menu :font :display-type :reshape-method :flush-method
      :button-region :from-button :filename))



(defmethod resize-window :after ((b browser) position)
      (when (dialog-items b)
             (set-dialog-item-box (first (dialog-items b))
                   (make-box
                          0 0 (round (/ (position-x position) 2))
                          *tree-button-bar-height*))
             (set-dialog-item-box (second (dialog-items b))
                   (make-box
                          (round (/ (position-x position) 2)) 0
                          (position-x position) *tree-button-bar-height*))
             (setf (window-exterior (wtree b))
                       (make-box
                              0 *tree-button-bar-height*
                              (position-x position)
                              (position-y position)))))
      
            
            
(defmethod initialize-instance :after ((b browser)  &rest keys)
     (setf (font b) (make-font :modern nil 10 '(:bold)))
     (if (zerop (offset b)) (setf (offset b) (font-character-height (font b))))
     (if (zerop (border b)) (setf (border b) (* 2 (font-character-height (font b)))))
     (if (listp (starting-tree b))
        (setf (starting-tree b) (remove-if 'null (starting-tree b))))
     (let* ((specs (size-tree-vertical b (starting-tree b) :left 0 :offset (offset b)))
             (x (round (+ (width specs) 100)))
             (y (round (+ (height specs) 40)))
             (center-x (/ (box-width (clipping-box *screen*)) 2))
             (center-y (/ (box-height (clipping-box *screen*)) 2)))
         (setf (stream-title b) (title b))
         (resize-window b
              (make-position
                     (min x (round (* 1.5 center-x)))
                     (+ (min y (round (* 1.5 center-y))) *tree-button-bar-height*)))
;         (set-scroll-range b x y 0 0)
         (setf (height b) (box-height (visible-box b)))
         (setf (left b) (box-left (visible-box b)))
         (setf (bottom b) (box-bottom (visible-box b)))
         (setf (width b) (box-width (visible-box b)))
         (process-pending-events)
         (display-tree-vertical b (starting-tree b))
         (update-dialog b
               (list
                  (make-dialog-item :widget 'button
                         :box (make-box 0 0 (round (/ (width b) 2)) *tree-button-bar-height*)
                         :title "Help"
                         :set-value-fn '(lambda (item new old) (browser-help item))
                         :font (make-font nil :arial 16 '(:bold)))
                  (make-dialog-item :widget 'button
                         :box (make-box
                                         (round (/ (width b) 2)) 0
                                         (width b) *tree-button-bar-height*)
                         :title "Close"
                         :set-value-fn `(lambda (item new old) (close-browser ,b))
                         :font (make-font nil :arial 16 '(:bold)))))))



(defmethod browser-help ((b button))
  ())



;;; Computes the size of a browser for a list of trees, displayed vertically.

(defmethod size-tree-vertical ((browser browser) trees  &key
                                                (left (border browser))
                                                offset)
     (let* ((bestlayout 0)
             (bestsum 0)
             (thisy 0)
             (subbuttons (loop for tree in trees
                                     collect (let ((tr (size-tree-vertical browser tree :left left :offset offset)))
                                                    (setf bestsum (+ 1 bestsum (height tr) (width tr)))
                                                    tr)))
             subbutton)
         (loop for tree in trees as layout in subbuttons do
             (setf thisy (+ thisy (height layout)))                      ; try y dimension to be trees so far
             (setf subbutton (packall thisy  subbuttons))
             (when (< (+ (height subbutton) (width subbutton)) bestsum)
                  (setf bestsum (+ (height subbutton) (width subbutton)))
                  (setf bestlayout subbutton)))         
         bestlayout))
    

      
;;; finds size of the display yielded by packing all given layouts
;;; into a display with y dimension given by ydim.
      
(defun packall (ydim layouts)
     (let ((ysofar 0)
            (xsofar 0)
            (xmax 0)
            (ymax ydim))
         (loop for layout in layouts do
             (cond ((> (+ ysofar (height layout)) ydim)
                       (setf ysofar (height layout))
                       (setf xsofar (+ xsofar xmax))
                       (setf xmax (width layout))
                       (if (> (height layout) ydim) (setf ymax (height layout))))
                      (t (setf ysofar (+ ysofar (height layout)))
                         (if (> (width layout) xmax) (setf xmax (width layout))))))
         (make-instance 'button-spec
               :height ymax
               :width (+ xsofar xmax))))
    


;; (defun font-string-width (stream string)
;;   (+ 7 (stream-string-width stream string)))
 


;;; Computes the size of a browser for a single tree.  This is done by
;;; virtually laying out the buttons, and doing all the computations
;;; required.   

(defmethod size-tree-vertical ((browser browser) (tree tree)  &key
                                                (left (border browser))
                                                offset bottom)
     (declare (ignore bottom))
     (let ((fullwidth 0)
            (maxheight 0))
         (if (find-descendants browser tree)
            (prog (subbutton)
                (dolist (desc (find-descendants browser tree) fullwidth) 
                     (setf subbutton (size-tree-vertical browser desc :left (+ fullwidth left)
                                                  :offset offset))
                     (setf fullwidth (+  fullwidth offset (width subbutton)))
                     (setf maxheight (max  maxheight  (height subbutton))))
                (setf fullwidth
                        (max
                              (max
                                     (* 10 (length (if (stringp (tlabel tree)) (tlabel tree) (write-to-string (
                                                                                                               tlabel tree)))))
                                     *default-pop-up-button-width*)
                              (- fullwidth offset))))
            (setf fullwidth (max
                                               (* 10 (length (if (stringp (tlabel tree)) (tlabel tree) (write-to-string (
                                                                                                                         tlabel tree)))))
                                               *default-pop-up-button-width*)))
         (make-instance 'button-spec
               :width fullwidth
               :height (+ offset maxheight 20))))



;;; Display a list of trees in a matrix.

(defmethod display-tree-vertical ((browser browser) trees  &key
                                                     (left (border browser) )
                                                     (bottom nil)
                                                     (connect-list nil)
                                                     (*default-push-button-size-p* nil))
     (declare (ignore connect-list bottom left))
     (let ((bot (- (height browser) (border browser) (font-character-height (font browser))))
            (ysofar 0)
            (xsofar (offset browser))
            (xmax 0)
            (ymax (height browser))
            subbutton)
         (loop for tree in trees do
             (setf subbutton (size-tree-vertical browser tree :offset (offset browser)))
             (cond ((> (+ ysofar (height subbutton)) (height browser))
                       (setf xsofar (+ xsofar xmax (offset browser)))
                       (setf ysofar 0)
                       (display-tree-vertical browser tree :left xsofar :bottom
                           (- bot ysofar))
                       (setf ysofar (height subbutton))
                       (setf xmax (width subbutton))
                       (if (> (height subbutton) (height browser)) (setf ymax (height subbutton))))
                      (t (display-tree-vertical browser tree :left  xsofar :bottom (- bot  ysofar))
                         (setf ysofar (+ ysofar (height subbutton)))
                         (if (> (width subbutton) xmax) (setf xmax (width subbutton))))))
         (make-instance 'button-spec
               :height ymax
               :width (+ xsofar xmax))))
	     
	     
      
(defmethod label ((tree decision-tree))
  (if (or
       (> (length (descendants tree)) 1)
       (and (= (length (descendants tree)) 1)
	    (null (descendants (first (descendants tree))))))
      (symbol-name (content tree))
    (format nil "[~a]" (content tree))))



;;; Display a tree (vertically).  The active items are push-buttons.

(defmethod display-tree-vertical ((browser browser) (tree tree)  &key
                                                                 (left (border browser) )
                                                                 (bottom nil)
                                                                 (connect-list nil)
                                                                 (*default-push-button-size-p* nil))
      (let ((fullwidth 0)
              (maxheight 0)
              button 
              (offset (offset browser)))
          (unless (wtree browser)
                (setf (wtree browser)
                          (open-dialog () 'tree-dialog browser :pop-up-p nil
                                :window-exterior (make-box 
                                                                         0 *tree-button-bar-height*
                                                                         (width browser)
                                                                         (height browser))
                                :user-resizable nil :user-movable nil
                                :user-closable nil :user-scrollable t
                                :user-shrinkable nil)))
          (if (null bottom)
             (setf bottom
                       (- (height browser) (border browser) (font-character-height (font browser)))))
          (push (setf button			; keeps track of all buttons in the browser
                                (make-dialog-item :widget 'push-button
                                       :box (make-box 0 0 0 0)
                                       :font (make-font :roman :times 10 '(:bold))
                                       :set-value-fn 
                                       `(lambda (item new old)
                                             (browser-select-item ,browser item new))
                                       :label (if (stringp (tlabel tree)) (tlabel tree) (write-to-string (
                                                                                                          tlabel tree)))
                                       :node tree
                                       :subtree (descendants tree)))
                (buttons browser))
          (push tree (items browser))		;keeps track of items in the browser
          ;; The general strategy is to recursively lay out all the
          ;; descendants of a node first, then to add up the space that they
          ;; need.  This subtree will require as much width as the maximum
          ;; of the current button and the combined subs. The button for the
          ;; current node is centered in this horizontal space. 
          (if (find-descendants browser tree)
             (prog (subbutton)
                  (dolist (desc (find-descendants browser tree) fullwidth) 
                        (setf subbutton (display-tree-vertical browser desc :left (+ fullwidth left)
                                                        :bottom (- bottom offset (height button))
                                                        ))
                        (setf fullwidth (+  fullwidth offset (width subbutton)))
                        (setf maxheight (max  maxheight  (height subbutton)))
                        ; keep track
                        ; of the centerpoints of the
                        ; subs, for connecting with
                        ; lines later on.
                        (push (centerpoint subbutton)  connect-list))
                  (setf fullwidth (max (- fullwidth offset) (width button))))
             (setf fullwidth (width button)))
          ;; set the button.  The action item is to call the
          ;; browser-select-item (see above).
          (set-button button browser
               :left (floor (- (+ left (/ fullwidth 2) ) (/ (width button) 2)))
               :bottom bottom
               :border nil)
          (dolist (connection connect-list)	; connect the nodes
                (draw-right browser
                 (round (car connection))
                 (round (cadr connection))
                 (round (+ left (/ fullwidth 2.0)))
                 (round bottom)))
          ;; return a width and a centerpoint to the next level up
          (make-instance 'button-spec
                 :height (+ offset maxheight (+ 6 (font-character-height (font browser) )))
                 :width fullwidth
                 :centerpoint (list (floor (+ left (/ fullwidth 2) )) (+ (height button)  bottom))
                 )))



(defmethod browser-select-item ((b browser) item mousestate)
     ())



;;; The return value from a recursive call to display-tree or
;;; display-tree-horizontally has three things that must be returned.
;;; This is a structure that holds them.  

(defclass button-spec ()
	  ((width :initarg :width
		  :initform 0
		  :accessor width
		  :type integer)
	   (height :initarg :height
		  :initform 0
		  :accessor height
		  :type integer)
	   (centerpoint :initarg :centerpoint
			:initform '(0 0)
			:accessor centerpoint
			:type list)))


;;; ==========================================================================
;;; DRAWING
;;; ==========================================================================


(defclass line-object ()
      ((x1               :accessor x1       :initarg :x1       :initform 0)
       (y1               :accessor y1       :initarg :y1       :initform 0)
       (x2               :accessor x2       :initarg :x2       :initform 0)
       (y2               :accessor y2       :initarg :y2       :initform 0))
     (:documentation "no doc"))


(defmethod draw-object ((w window) (obj line-object))
  (draw-line w
      (make-position (x1 obj) (y1 obj))
      (make-position (x2 obj) (y2 obj))))

  
(defmethod draw-right ((b browser) x1 y1 x2 y2)
  (let* ((disp (wtree b))
         (obj (make-instance 'line-object 
                             :x1 x1 :y1 (- (height b) y1) 
                             :x2 x2 :y2 (- (height b) y2))))
    (setf (lines disp) (cons obj (lines disp)))
    (draw-object disp obj)))


(defmethod redisplay-window :after ((td tree-dialog) &optional box)
     (dolist (item (lines td))
          (draw-object td item)))



;;; ==========================================================================
;;; PUSH-BUTTON
;;; ==========================================================================


(defclass push-button (button)
      ((region :initarg :region  
           :initform nil
           :accessor region)
       (node  :initarg :node  
            :initform nil
            :accessor node)
       (subtree :initarg :subtree
            :initform nil
            :accessor subtree)
       (left :type integer
           :initarg :left
           :accessor left)
       (label :type integer
           :initarg :left
           :accessor label)
       (bottom :type integer
            :initarg :bottom
            :accessor bottom))
     (:documentation "This is a push Button"))



(defmethod initialize-instance :after ((p push-button) &key 
                                                           (label "click here")
                                                           (width nil)
                                                           (active t) ;enabled/disabled
                                                           (font *fred-default-font-spec*))
      (set-dialog-item-box p
            (make-box 0 0
                   (max (* 10 (length label))
                            *default-pop-up-button-width*)
                   *tree-button-height*))
     (set-dialog-item-title p label) 
     (set-dialog-item-font p font)
     (set-dialog-item-available-p p active)
     (setf (region p) (dialog-item-box p)))



(defmethod set-button ((p push-button) display &key
                                     (left 0)
                                     (bottom 0)
                                     (active t)
                                     (action nil)
                                     (border 0)
                                     (default nil))
     (let ((width (max (* 10 (length (dialog-item-title p))) *default-pop-up-button-width*)))
         (declare (ignore border action active))
         (if (not (slot-boundp p 'left)) (setf (left p) left))
         (if (not (slot-boundp p 'bottom)) (setf (bottom p) bottom))
         (setf bottom (bottom p))   
         (setf left (left p))      
         (set-dialog-item-box p
              (make-box
                    left 
                    (- (height display)
                       bottom 
                       *tree-button-height*)
                    (+ left width)
                    (- (height display)
                           bottom)))
         (open-dialog-item p (wtree display))
         p))


     
(defmethod window-update-cursor ((b browser) point)
  (declare (ignore point))
  (set-cursor 129))



#|
(defmethod view-click-event-handler :after ((p push-button) where)
  (browser-node-click-event-handler p where))


(defmethod browser-node-click-event-handler ((self push-button) where)
  (declare (ignore where)))
 


(defmethod view-click-event-handler :after ((b button-dialog-item) where)
  (declare (ignore where))
  (when (eql (dialog-item-text b) "Recompute")
    (view-draw-contents b)))

|#
;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
