;;; -*- Mode: LISP; Package: pail-lib; Syntax: Common-lisp; -*-

(in-package :pail-lib)

(export '(table-editor
                 attributes
                 editor-to-table
                 table-to-editor
                 insert-new-attribute
                 get-rid-of-attribute
                 insert-new-row
                 contents
                 table-name
                 get-rid-of-row
                 table-needs-saving-p
                 ))



(defconstant *s-h-size* 100)        
(defconstant *s-v-size* 30)
(defconstant *s-cell-size* (make-position *s-h-size* *s-v-size*))
(defconstant *s-h-ribbon* 3)       
(defconstant *s-v-ribbon* 3)

(defconstant *button-height* 25)
(defconstant *buttons-height* 50)



(defclass table-editor (dialog)
       ((contents
              :initarg :contents
              :initform nil
              :accessor contents)
        (functions 
             :initarg :functions
             :initarg nil
             :accessor functions)
        (table-needs-saving-p
             :initarg :table-needs-saving-p
             :initform nil
             :accessor table-needs-saving-p)
        (table-name
             :initarg :table-name
             :initform ""
             :accessor table-name)))



(defclass editable-cell-dialog-item (editable-text)
       ((discard 
            :initarg :discard
            :initform nil
            :accessor discard)))



(defclass non-editable-cell-dialog-item (static-text)
       ((discard
             :initarg :discard
             :initform t
             :accessor discard)))



(defmethod ss-size ((s table-editor) &optional cell-size)
      (let ((cols (length (contents s)))
              (rows (length (car (contents s)))))
          (if cell-size
                 (make-position
                        (+ (* (+ (position-x cell-size) *s-h-ribbon*) rows) *s-h-ribbon*)
                        (+ (* (+ (position-y cell-size) *s-v-ribbon*) cols) *s-v-ribbon* *buttons-height*))
                 (make-position
                        (+ (* (+ *s-h-size* *s-h-ribbon*) rows) *s-h-ribbon*)
                        (+ (* (+ *s-v-size* *s-v-ribbon*) cols) *s-v-ribbon* *buttons-height*)))))
    


;(defmethod initialize-instance :before ((s table-editor) &rest whatever)
;  (declare (ignore whatever))
;  :window-show nil)
  
  
  
(defmethod initialize-instance :after ((s table-editor) &key cell-size)
      (if cell-size
             (resize-window s (ss-size s cell-size))
         (if (<  (length (car (contents s))) 1)
            (resize-window s (make-position 100 100))
            (resize-window s (ss-size s))))
      (setf (stream-title s) (table-name s))
      (let* ((top-row t)
                (input (contents s))
                (output nil)
                (table-width (box-width (window-interior s)))
                (table-button-width (round (/ table-width 4)))
                (s-cell-size (if cell-size
                                            cell-size
                                            *s-cell-size*))
                (h-counter *s-v-ribbon*)
                (v-counter (+ *s-v-ribbon* *buttons-height*)))
          (push (make-dialog-item :widget 'button
                             :box (make-box 0 *button-height* table-button-width *buttons-height*)
                             :font (make-font :modern nil 16 '(:bold))
                             :set-value-fn `(lambda (item new old)
                                                           (funcall (nth 0 (functions ,s))))
                             :title "Add ex.")
                output)
          (push (make-dialog-item :widget 'button
                             :box (make-box
                                             table-button-width *button-height*
                                             (* 2 table-button-width) *buttons-height*)
                             :font (make-font :modern nil 16 '(:bold))
                             :set-value-fn `(lambda (item new old)
                                                           (funcall (nth 1 (functions ,s))))
                             :title "Del ex.")
                output)
          (push (make-dialog-item :widget 'button
                             :box (make-box
                                             (* 2 table-button-width) *button-height*
                                             (* 3 table-button-width) *buttons-height*)
                             :font (make-font :modern nil 16 '(:bold))
                             :set-value-fn `(lambda (item new old)
                                                           (funcall (nth 2 (functions ,s))))
                             :title "Add att.")
                output)
          (push (make-dialog-item :widget 'button
                             :box (make-box
                                             (* 3 table-button-width) *button-height*
                                             table-width *buttons-height*)
                             :font (make-font :modern nil 16 '(:bold))
                             :set-value-fn `(lambda (item new old)
                                                           (funcall (nth 3 (functions ,s))))
                             :title "Del att.")
                output)
          (dolist (row input output)
                (setq top-row nil)
                (setq h-counter 3)
                (dolist (item row)
                      (if (or top-row (< h-counter 10))
                         (push (make-dialog-item :widget 'non-editable-cell-dialog-item
                                            :box (make-box
                                                            h-counter 
                                                            v-counter
                                                            (+ h-counter (position-x s-cell-size))
                                                            (+ v-counter (position-y s-cell-size)))
                                            :value item
                                            :font (make-font :modern nil 14 '(:bold)))
                               output)
                         (push (make-dialog-item :widget 'editable-cell-dialog-item
                                            :box (make-box
                                                            h-counter 
                                                            v-counter
                                                            (+ h-counter (position-x s-cell-size))
                                                            (+ v-counter (position-y s-cell-size)))
                                            :set-value-fn `(lambda (item new old) 
                                                                           (setf (table-needs-saving-p ,s) t))
                                            :value item
                                            :font (make-font :modern nil 14 '(:bold)))
                               output))
                      (setq h-counter (+ h-counter (position-x s-cell-size) *s-h-ribbon*)))
                (setq v-counter (+ v-counter (position-y s-cell-size) *s-v-ribbon*)))
          (push
                (make-dialog-item :widget 'button
                       :box (make-box 0 0 60 *button-height*)
                       :font (make-font :modern nil 16 '(:bold))
                       :available-p nil
                       :set-value-fn `(lambda (item new old)
                                                      (table-editor-help))
                       :title "Help")
                output)
          (push
                (make-dialog-item :widget 'button
                       :box (make-box
                                       (- table-width 60) 0
                                       table-width *button-height*)
                       :font (make-font :modern nil 16 '(:bold))
                       :set-value-fn `(lambda (item new old)
                                                      (funcall (nth 4 (functions ,s))))
                       :title "Close")
                output)
          (update-dialog s (reverse output)))
      s)


(defmethod attributes ((te table-editor))
      (let ((len (1- (length (car (contents te)))))
              (result nil))
          (when (> len 0)
                 (dolist (item (cdar (contents te)))
                       (push (read-from-string item) result))
                 (reverse result))))



(defmethod editor-to-table ((te table-editor))
      (update-contents te)
      (let ((working-contents (symbol-contents (contents te)))
              (current-values nil))
          (when working-contents
                 (dolist (item (cdr working-contents))
                       (push (cdr item) current-values))
                 (make-instance 'table
                        :table-name (table-name te)
                        :attributes (cdr (car working-contents))
                        :rows (reverse current-values)))))


(defun symbol-contents (lista)
     (let ((result nil)
            (partial nil))
         (dolist (item lista)
              (dolist (item1 item)
                   (push (read-from-string item1) partial))
              (push (reverse partial) result)
              (setq partial nil))
         (reverse result)))

(defmethod editor-to-table-id3 ((te table-editor))
      (update-contents te)
      (let ((working-contents (contents te))
              (current-values nil))
          (when working-contents
                 (dolist (item (cdr working-contents))
                       (push  (cdr item) current-values))
                 (make-instance 'table
                        :table-name (table-name te)
                        :attributes (cdr (car working-contents))
                        :rows (reverse current-values)))))
      

(defmethod table-to-editor ((tab table) win functions &optional cell-size)
      (let* ((counter 0)
                (result nil)
                (fixed-rows (dolist (item (rows tab) (reverse result))
                                             (incf counter) 
                                             (push (cons counter item) result))))
          (open-dialog () 'table-editor win :pop-up-p nil
                :user-closable nil :user-shrinkable nil
                :cell-size cell-size
                :window-border :dialog-box
                :background-color (make-rgb :red 0 :green 128 :blue 128) 
                :contents
                (list-to-string-list
                 (push (cons "Ex. #" (attributes tab)) fixed-rows))
                :functions functions
                :table-name (table-name tab))))





(defun list-to-string-list (lista)
      (let ((fresult nil)
              (iresult nil))
          (cond ((null lista) nil)
                      (t (dolist (sublist lista)
                                (dolist (item sublist)
                                      (if (stringp item)
                                         (push item iresult)
                                         (push (write-to-string item) iresult)))
                                (push (reverse iresult) fresult)
                                (setq iresult nil))))
          (reverse fresult)))



(defmethod update-contents ((s table-editor))
      (let* ((lista nil)
                (old-contents (contents s))
                (len (length (car old-contents)))
                (new-contents nil))
          (dolist (item (dialog-items s))
                (when (stringp (dialog-item-value item))
                       (progn
                            (push (dialog-item-value item) lista)
                            (when (= (length lista) len)
                                   (push (reverse lista) new-contents)
                                   (setq lista nil)))))
          (setf (contents s) (reverse new-contents))))



(defun delete-dialog-row (text)
      (yes-or-no-p
            (concatenate 'string 
                  "Are you sure you want to delete example #"
                  text)))


(defmethod get-rid-of-row ((te table-editor) text win &optional cell-size)
      (update-contents te)
      (let ((old-contents (contents te))
              (new-contents nil)
              (counter 0)
              (functions-te (functions te))
              (name (table-name te)))
          (dolist (item old-contents)
                (cond ((equal text (car item)))
                            ((numberp (read-from-string (car item)))
                                 (progn
                                      (incf counter)
                                      (push
                                            (cons (write-to-string counter) (cdr item))
                                            new-contents)))
                            (t (push item new-contents))))
          (close te)
          (psetf te
                (open-dialog () 'table-editor win :pop-up-p nil
                      :cell-size cell-size
                      :window-border :dialog-box
                      :user-closable nil :user-shrinkable nil
                      :background-color (make-rgb :red 0 :green 128 :blue 128) 
                      :contents (reverse new-contents)
                      :functions functions-te
                      :table-needs-saving-p t
                      :table-name name))
          te))



(defun insert-dialog-row ()
      (pop-up-message-dialog "About to insert a new example"))



(defmethod insert-new-row ((te table-editor) win &optional cell-size (initial "NOTHING"))
      (update-contents te)
      (let* ((old-contents (contents te))
                (last-example (length old-contents))
                (example-size (length (car old-contents)))
                (functions-te (functions te))
                (name (table-name te))
                (new-contents
                      (cons
                            (write-to-string last-example)
                            (make-list (1- example-size) :initial-element initial))))
          (close te)
          (psetf te
                (open-dialog () 'table-editor win :pop-up-p nil
                      :cell-size cell-size
                      :window-border :dialog-box
                      :user-closable nil :user-shrinkable nil
                      :background-color (make-rgb :red 0 :green 128 :blue 128) 
                      :contents (append old-contents (list new-contents))
                      :functions functions-te
                      :table-needs-saving-p t
                      :table-name name))
          te))



(defun delete-dialog-attribute (text)
      (yes-or-no-p (concatenate 'string 
                                    "Would you like to delete the " 
                                    text
                                    " attribute?")))



(defmethod get-rid-of-attribute ((te table-editor) text win &optional cell-size)
      (update-contents te)
      (let* ((old-contents (contents te))
                (attribute-position (position text (car old-contents)
                                                          :test (function eql)))
                (functions-te (functions te))
                (name (table-name te)))
          (dolist (item old-contents)
                (delete (nth attribute-position item) item 
                      :start attribute-position
                      :end (1+ attribute-position)))
          (close te)
          (psetf te
                (open-dialog () 'table-editor win :pop-up-p nil
                      :cell-size cell-size
                      :window-border :dialog-box
                      :user-closable nil :user-shrinkable nil
                      :background-color (make-rgb :red 0 :green 128 :blue 128) 
                      :functions functions-te
                      :table-needs-saving-p t
                      :contents old-contents
                      :table-name name))
          te))



(defmethod insert-new-attribute ((te table-editor) win &optional cell-size)
      (update-contents te)
      (let ((old-contents (contents te))
              (functions-te (functions te))
              (name (table-name te)))
          (multiple-value-bind (new-att other-att value)
                  (ask-user-for-string
                        "Please enter the name of the new attribute"
                        ""
                        "OK" "Cancel")
                 (setq new-att (string-upcase new-att))
                 (when (string= value "OK")
                        (cond ((string= "" new-att))
                                    ((member new-att (car old-contents) :test #'string=)
                                     (pop-up-message-dialog
                                           win "Message"
                                           (concatenate 'string 
                                                 "Sorry, "
                                                 new-att " already exists!")
                                           nil
                                           "OK"))
                                    (t (nconc (car old-contents)
                                             (list new-att))
                                       (dolist (item (cdr old-contents))
                                             (nconc item (list "NOTHING")))
                                       (close te)
                                       (psetf te
                                             (open-dialog () 'table-editor win :pop-up-p nil
                                                   :cell-size cell-size
                                                   :window-border :dialog-box
                                                   :user-closable nil :user-shrinkable nil
                                                   :background-color (make-rgb :red 0 :green 128 :blue 128) 
                                                   :functions functions-te
                                                   :table-needs-saving-p t
                                                   :contents old-contents
                                                   :table-name name)))))
                 te)))

#|

(defmethod view-key-event-handler :around ((n non-editable-cell-dialog-item) char)
  (let ((text (dialog-item-text n))
        (current-table-editor (view-container n)))
    (cond ((and (or (char= char #\DEL) (char= char #\Delete))
                (member (read-from-string text) (attributes current-table-editor)))
           (when (delete-dialog-attribute text) (get-rid-of-attribute current-table-editor text)))
          ((and (or (char= char #\DEL) (char= char #\Delete))
                (numberp (read-from-string text)))
           (when (delete-dialog-row text) (get-rid-of-row current-table-editor text)))
          ((and (char= char #\Help)
                (numberp (read-from-string text)))
           (when (insert-dialog-row) (insert-new-row current-table-editor)))
          ((and (char= char #\Help)
                (member (read-from-string text) (attributes current-table-editor)))
           (insert-new-attribute current-table-editor)))))

|#