;;; -*- Mode: LISP; Package: PAIL-LIB; Syntax: Common-lisp;  
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   tables.lsp
;;; Short Desc: class definition for tables
;;; Version:    2.0
;;; Status:     Review
;;; Last Mod:   16.4.91 TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------
;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;


;;; Supports a class called table, instances of which hold sets of
;;; rows, where each row is a list of values for an attribute.


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================
;;;
(in-package :pail-lib)


(export '(table
                     make-table
                     table-name
                     get-values
                     get-possible-values
                     get-row
                     get-value
                     get-nth-value
                     count-occurences
                     count-occurences-if
                     get-subtable
                     get-table
                     clash-p
                     expand-wildcards
                     copy-table
                     cumulated-values
                     cut-value
                     substitute-values
                     table-to-sexpr
                     sexpr-to-table
                     attributes
                     number-of-rows
                     rows
                     set-values
                     set-row
                     set-nth-row
                     set-value
                     get-xy-value
                     set-xy-value
                     collect-values))

;;; ==========================================================================
;;; CLASS DEFINITION FOR TABLE
;;; ==========================================================================
;;;

(defclass table ()
       ((table-name      
              :initarg :table-name
              :initform nil
              :accessor table-name)
        (attributes       
             :initarg :attributes
             :initform nil
             :accessor attributes
             :type list)
        (number-of-rows       ; number-of-rows is
                                                ; automatically maintained
                                                ; to reflect the length of
                                                ; the list given in rows. 
             :initform nil	
             :accessor number-of-rows
             :type integer)
        (rows         
             :initarg :rows
             :initform nil
             :accessor rows
             :type list))
      (:documentation "The class for attribute-value pair like example tables"))



(defmethod initialize-instance :after ((data table) &rest arg)
      (declare (ignore arg))
      (setf (number-of-rows data) (length (rows data))))



(defmethod (setf rows) :after (arg (data table))
      (declare (ignore arg))
      (setf (number-of-rows data) (length (rows data))))



;;; A table is contructed by giving a list of attribute names
;;; (symbols), and a list of rows, where a row is given as a list of
;;; values.  The length of each row should match the length of the
;;; attribute list.

(defvar *list-of-tables* nil)

(defun make-table (&key (table-name nil table-name-p)
                                           (attributes nil attributes-p)
                                           (rows nil rows-p))
      (when (not (and table-name-p attributes-p rows-p))
             (error "~&Usage of make-table: maketable :table-name :attributes attributes :rows rows"))
      (make-instance 'table
                         :table-name table-name
                         :attributes attributes
                         :rows rows))



(defun list-of-tables (tables)
      (let ((result nil))
          (dolist (item tables)
                (push (table-name item) result))
          result))



(defun select-table-by-name (name tables)
      (let ((result nil))
          (dolist (item tables)
                (when (string= (table-name item) name)
                       (setq result item)))
          result))



;;; Harlequin CL has a bug in the format: It does relative tabulation
;;; instead of relative. You have to insert ~% after "Attributes" and "Values"

(defmethod print-table ((table table) &optional (stream t))
      (format stream "~%Table Name" (table-name table))
      (format stream "~%Attributes~18T(~{~13a~})" (attributes table))
      (format stream "~%Values~16T(~{~18T(~{~13a~})~%~})" (rows table)))



(defmethod print-object ((table table) stream)
      (if *readable*
         (print-table table stream)
         (print-unreadable-object 
          (table stream :type t :identity t))))



;;; Given an attribute name and a table, returns a function
;;; definition.  The returned function takes a row as an argument, and
;;; returns the value of the given attribute for that row.  It takes a
;;; bit of computation to work out how to extract the value for an
;;; attribute from a row.  Some of that computation can be done once
;;; for each attribute. 

(defmethod x-access-function (attribute (data table))
      `(lambda (seq)
            (nth ,(search (list attribute) (attributes data) :test #'equal) seq)))



(defmethod x-access-function (attribute (data table))
      (let ((the-search (search (list attribute) (attributes data) :test #'equal)))
          #'(lambda (seq) (nth the-search seq))))
  


;;; Given an attribute name, a value, and a table, returns a function
;;; definition.  If the m is the smallest number for which the mth row
;;; of the table has the specified value for the specified attribute,
;;; the resulting function selects the mth item from a given sequence.

(defmethod y-access-function (attribute value (data table))
      (let ((x-access (x-access-function attribute data)))
          `(lambda (seq)
                (nth ,(search (list value) (rows data)
                               :test #'(lambda (o1 o2)
                                                (equal o2 (funcall x-access o1))))
                     seq))))



(defmethod y-access-function (attribute value (data table))
      (let* ((x-access (x-access-function attribute data))
                (the-search (search (list value) (rows data)
                                             :test #'(lambda (o1 o2)
                                                              (equal o2 (funcall x-access o1))))))
          #'(lambda (seq) (nth the-search seq))))



;;; Returns a list of all the values of the given attribute.

(defmethod get-values (attribute (data table))
      (mapcar (x-access-function attribute data) (rows data)))



;;; Fills values for an attribute in a table

(defmethod set-values (attribute (data table) values)
      (let ((pos (search (list attribute) (attributes data) :test #'equal)))
          (do ((values values (rest values))
                  (rows (rows data) (rest rows)))
                 ((or (null values) (null rows)))
                (setf (nth pos (first rows)) (first values)))))



;;; Returns a list of all the values that appear in the table for a
;;; given attribute, duplicates deleted. 

(defmethod get-possible-values (attribute (data table))
      (remove-duplicates (get-values attribute data) :test (function equal)))



;;; Returns the first row which has the given value for the specified
;;; attribute. 

(defmethod get-row (attribute value (data table))
      (funcall (y-access-function attribute value data) (rows data)))



(defmethod set-row (attribute value (data table) row)
      (let ((pos (search (list value) (get-values attribute data) :test #'equal)))
          (setf (nth pos (rows data)) row)))



(defmethod set-nth-row (nth (data table) row)
      (setf (nth nth (rows data)) row))



;;; Returns the value of the given attribute in the first row for
;;; which the key attribute has the key value.

(defmethod get-value (key-attribute key-value attribute (data table))
      (funcall (x-access-function attribute data)
           (funcall (y-access-function key-attribute key-value data)
                (rows data))))



(defmethod collect-values (key-attribute key-value attribute (data table))
      (do ((key-values (get-values key-attribute data) (rest key-values))
              (values (get-values attribute data) (rest values))
              (collection nil))
             ((null key-values) collection)	; reverse order !
            (when (equal key-value (first key-values))
                   (push (first values) collection))))



(defmethod set-value (key-attribute key-value attribute (data table) value)
      (let ((pos-y (search (list key-value)
                                 (get-values key-attribute data) :test #'equal))
              (pos-x (search (list attribute) (attributes data) :test #'equal)))
          (setf (nth pos-x (nth pos-y (rows data))) value)))



;;; Returns the value of the specified attribute in the row specified
;;; by row-number (an integer between 0 and number-of-rows - 1).

(defmethod get-nth-value (attribute row-number (data table))
      (apply (x-access-function attribute data) 
            (list (nth row-number (rows data)))))



(defmethod set-nth-value (attribute row-number (data table) value)
      (let ((pos-x (search (list attribute) (attributes data) :test #'equal)))
          (setf (nth pos-x (nth row-number (rows data))) value)))



(defmethod get-xy-value (x y (data table))
      (nth x (nth y (rows data))))



(defmethod set-xy-value (x y (data table) value)
      (setf (nth x (nth y (rows data))) value))



;;; How many times does the specified attribute take the specified
;;; value?

(defmethod count-occurences (attribute value (data table))
      (let ((values (get-values attribute data)))
          (count value values :test (function equal))))



;;; How many times does attribute1 take value1, AND attribute2 takes
;;; value2?

(defmethod count-occurences-if (attribute1 value1 attribute2 value2
                                                                 (data table) 
                                                                 &optional (values1 
                                                                                      (get-values attribute1 data))
                                                                 (values2 
                                                                  (get-values attribute2 data)))
      (cond ((null values1) 0)
                  ((and (equal (car values1) value1) (equal (car values2) value2))  
                   (1+ (count-occurences-if attribute1 value1 attribute2 value2 
                           data (cdr values1) (cdr values2))))
                  (t (count-occurences-if attribute1 value1  attribute2 value2 
                       data (cdr values1) (cdr values2)))))



;;; Return the subtable of the original table, of entries for which
;;; the given attribute takes the given value.  That is, in the new
;;; table, only the rows for which attribute takes value are included,
;;; and the column for attribute is excluded.

(defmethod get-subtable (attribute value (data table))
      (let ((index (search (list attribute) (attributes data)
                                 :test (function equal)))
              (access-fcn (x-access-function attribute data)))
          (make-instance 'table
                 :table-name (concatenate 'string (table-name data)
                                               (write-to-string (gensym)))
                 :attributes (remove-nth index (attributes data))
                 :rows (do* ((seq (rows data) (cdr seq))
                                       (res nil))
                                      ((null seq) res)
                                   (when (equal (apply access-fcn (list (car seq))) value)
                                          (setf res (cons (remove-nth index (car seq)) res)))))))



;;; Construct a table from the original, including only the specified
;;; attributes.

(defmethod get-table (attribute-list (data table))
      (make-instance 'table
             :table-name (print (concatenate 'string (table-name data)
                                                      (write-to-string (gensym))))
             :attributes attribute-list
             :rows (apply #'mapcar
                               (cons
                                     (function (lambda (&rest args) args))
                                     (mapcar #'(lambda (attribute)
                                                              (get-values attribute data))
                                            attribute-list)))))



;;; Read an S-expression into a table.  The first item in the list is
;;; the list of attribute names, the rest of the list is the row
;;; specification. 

(defun sexpr-to-table (sexpr)
      (make-instance 'table
             :table-name "what"
             :attributes (car sexpr)
             :rows (cdr sexpr)))



;;; Return a list representation of the table.  car of the list is a
;;; list of attribute names, the rest of the list is a list of rows.

(defmethod table-to-sexpr ((table table))
      (cons (attributes table) (rows table)))



;;; write interface with current pool 24.11 dta

(defmethod dump-editable ((table table))
      (intern-all (table-to-sexpr table) :dump))



;;; read interface with current pool 24.11 dta

(defmethod read-instance ((table table) desc name)
      (declare (ignore name))
      (let ((readtable (sexpr-to-table desc)))
          (setf (attributes table) (attributes readtable))
          (setf (rows table) (rows readtable))
          table))



;;; Returns another table exaclty like the given table.

(defmethod copy ((table table))
      (make-instance 'table
             :table-name (table-name table)
             :attributes (copy-tree (attributes table))
             :rows (copy-tree (rows table))))



(defmethod change-type ((table table) (table2 table)) table)

;;; Sorry - but what is this for ? (TW)



;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
