;;; -*- Mode: LISP; Package: pail-lib; Syntax: Common-lisp;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   trees.cl  
;;; Short Desc: class definition for trees
;;; Version:    2.0
;;; Status:     Review
;;; Last Mod:   5.2.91 TW
;;; Author:     Thomas Wehrle
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------

(in-package :pail-lib)

(export '(tree
          get-paths
          depth
          sexpr-to-tree
          tree-to-sexpr
          copy
          leaves
	  decision-tree
          query
          content
          descendants
          tlabel
          ))


(defclass tree ()
  ((content          :initarg :content
		     :initform nil
		     :accessor content
		     :type symbol)
   (descendants      :initarg :descendants
		     :initform nil
		     :accessor descendants
		     :type list))
  (:documentation "Foundation of trees"))


(defmethod print-tree ((tree tree) &optional (stream t) (level 0))
  (format stream "~%")
  (dotimes (dummy level) (format stream "  "))
  (format stream "(~a" (content tree))
  (if (null (descendants tree)) 
      (format stream ")")
    (dolist (desc (descendants tree) (format stream ")"))
      (print-tree desc stream (1+ level)))))


(defmethod print-object ((tree tree) stream)
      (if *readable*
         (print-tree tree stream)
         (print-unreadable-object 
              (tree stream :type t :identity t))))


(defmethod get-paths ((tree tree) &optional (path nil))
  (if (null (descendants tree)) (list (cons (content tree) path))
    (apply (function append)
	   (mapcar (function (lambda (sub-tree)
			       (get-paths sub-tree 
					  (cons (content tree) path))))
		   (descendants tree)))))


; a tree without descendants has depth 1
(defmethod depth ((tree tree))
  (apply (function max)
         (mapcar (function length)
                 (get-paths tree))))


(defun sexpr-to-tree (sexpr)
  (if (atom sexpr)
      (make-instance 'tree :content sexpr)
    (make-instance 'tree
      :content (car sexpr)
      :descendants (mapcar (function sexpr-to-tree) (cdr sexpr)))))


(defmethod tree-to-sexpr ((tree tree))
  (if (null (descendants tree))
      (list (content tree))
    (cons (content tree)
          (mapcar (function tree-to-sexpr)
                  (descendants tree)))))


(defmethod copy ((tree tree))
  (make-instance 'tree 
    :content (content tree)
    :descendants
    (mapcar (function copy)
	    (descendants tree))))


(defmethod leaves ((tree tree))
  (if (null (descendants tree)) (list (content tree))
    (apply (function append)
	   (mapcar (function leaves)
		   (descendants tree)))))


; ***************

(defclass decision-tree (tree)
  nil
  (:documentation "Foundation of decision trees"))


; gives the maximum number of questions to answer
(defmethod depth ((tree decision-tree))
  (1- (/ (call-next-method tree) 2)))


(defmethod find-sub-tree ((tree decision-tree))
  (let ((possible-values (mapcar (function content)
				 (descendants tree))))
    (format t "~%What is the value of ~a ?~%   Possible values: ~a~%   ==> "
	    (content tree) possible-values)
    (let ((value (read)))
      (when (member value possible-values :test #'equal)
	(do* ((desc-list (descendants tree) (rest desc-list))
	      (sub-tree (first desc-list) (first desc-list)))
	    ((eq value (content sub-tree)) 
	     (first (descendants sub-tree))))))))


(defmethod query ((tree decision-tree))
  (if (<= (length (descendants tree)) 1)
      (format t "~%From what I know I expect ~a to be ~a."
	      (content tree)
              (content (first (descendants tree))))
    (query (find-sub-tree tree))))


(defmethod query (something)
  (when (null something) 
    (format t "~%No knowledge about this - sorry.")))


(defmethod tlabel ((tree tree))
  (format nil "~A" (content tree)))
;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
