;;; -*- Mode: LISP; Package: rbs; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA
;;;
;;; ************************************************************************
;;;
;;; Filename:   database.cl
;;; Short Desc: primitives for database handling.
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   94.2
;;; Author:     M. Rosner
;;;
;;; Copyright (c) 1994 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------

;;; --------------------------------------------------------------------------
;;; INTERFACE

;; show <database> [<stream>]
;; read-database <filename> [<tool>] => object of type database
;; fetch <pattern list> <database> <initial environment set>
;; store <assertion> <db> <reason> 
;;; --------------------------------------------------------------------------


(in-package :rbs)



;;; class DB-ITEM

(defclass db-item ()
       ((sexp   
         :accessor sexp
         :initarg  :sexp
         :initform nil)
        
        (valid  
         :accessor valid
         :initform nil)
        
        (index   
         :accessor index
         :initform nil)))



;;; class ASSERTION
;;  an immediate subclass of db-item

(defclass assertion (db-item) ())



(defun is-assertion (x)
      (and (listp x) (symbolp (car x))))



;;; store <db-item> <database> => <database> if something is added
;;;                                                         =>    nil                 if item is already present

(defmethod store ((item db-item) db &optional (info t))
      (let ((buck (get-bucket item db)))
          (cond
                      ((in-bucket item buck)
                       nil)
                      (t (add item info buck)
                         db))))



(defmethod store ((r rule) db &optional (info t))
      (setf (rnumber r) (1+ (itemcount db)))
      (setf (itemlist db) (append (itemlist db)(list r)))
      (setf (itemcount db) (1+ (itemcount db))))



;(defmethod store ((r horn-rule) db &optional (info t))

(defmethod store ((r rule) db &optional (info t))
      (let ((buck (get-bucket r db)))
          (cond
                      ((in-bucket r buck)
                       nil)
                      (t (setf (rnumber r) (1+ (itemcount db)))
                         (setf (itemcount db) (1+ (itemcount db)))
                         (setf (itemlist db) (append (itemlist db)(list r)))
                         (add r info buck)
                         db))))





;;; check-syntax <object> => t     if syntax of object is OK
;;                                               => nil otherwise

(defmethod check-syntax ((a assertion))
      (is-assertion (sexp a)))



(defun make-assertion (sexp)
      (make-instance 'assertion :sexp sexp))



(defmethod initialize-instance :after ((a assertion) &rest restargs)
      (declare (ignore restargs))
      (cond ((check-syntax a)
                   (setf (index a) 
                             (list (car (sexp a)) (length (cdr (sexp a)))))
                   (setf (valid a) t))))



(defmethod in-bucket ((item assertion) bucket)
      (assoc (sexp item) (cdr bucket) :test 'equal))



;(defmethod in-bucket ((r horn-rule) bucket)

(defmethod in-bucket ((r rule) bucket)
      (member r (cdr bucket) :test 'same-rule))



(defun same-rule (x y)
      nil) ;provisional



(defmethod add ((item assertion) derivation bucket)
      (push (list (sexp item) derivation) (cdr bucket)))



(defmethod add ((r rule) derivation bucket)
      (push r (cdr bucket)))



(defmethod get-bucket ((item db-item) db &aux x buck)
      (let ((pred (car (index item)))
              (numbargs (cadr (index item)))
              (assertions (contents db)))
          (or (setq x (assoc pred assertions))
                (setf (contents db) 
                          (setq assertions
                                (cons (setq x (list pred)) assertions))))
          (or (setq buck (assoc numbargs (cdr x)))
                (progn
                     (setf (cdr x) (cons (setq buck (list numbargs)) (cdr x)))
                     (add-index item db)))
          buck))




    
;class DATABASE

(defclass database ()
       ((sexp        
              :accessor sexp
              :initarg  :sexp
              :initform nil)
        
        (contents 
              :initarg :contents
              :accessor contents
              :initform nil)
        
        (content-type
              :accessor content-type
              :initarg  :content-type)
        
        (itemcount	 
           :accessor itemcount
           :initarg :itemcount
           :initform 0)
        
        (itemlist
           :accessor itemlist
           :initarg :itemlist
           :initform nil)
        
        (indices    
            :initarg :indices
            :accessor indices
            :initform nil)))




(defclass assertion-database (database) ()
      (:default-initargs
       :content-type  'assertion))



(defclass rule-database (database) ()
      (:default-initargs
       :content-type  'rule))



(defclass horn-rule-database (database) ()
      (:default-initargs
       :content-type  'horn-rule))
  


(defmethod initialize-instance :after ((db assertion-database) &rest restargs)
      (declare (ignore restargs))
      (unless (contents db)
            (mapc
                   (function
                        (lambda (s)
                            (let ((item (make-instance (content-type db) :sexp s)))
                                (cond ((valid item)
                                             (progn 
                                                  (unless (itemlist db)
                                                        (format-display *rbs-output-window*
                                                             " Adding assertion ~A~%" s))
                                                  (store item db)))
                                            (t (unless (itemlist db)
                                                      (format-display *rbs-output-window*
                                                           " Ignoring bad assertion ~A~%" s)))))))
                   (sexp db))))




(defmethod initialize-instance :after ((rb rule-database) &rest restargs)
      (declare (ignore restargs))
      (mapc
             (function
                  (lambda (s)
                      (let ((item (make-instance (content-type rb) :sexp s)))
                          (cond ((valid item) (progn 
                                                                   (format-display *rbs-output-window*
                                                                        " Adding rule ~A~%" (rule-form item))
                                                                   (store item rb)))
                                      (t (format-display *rbs-output-window*
                                               " Ignoring bad rule~%"))))))
             (sexp rb)))




(defmethod initialize-instance :after ((rb horn-rule-database) &rest restargs)
      (declare (ignore restargs))
      (mapc
             (function
                  (lambda (s)
                      (let ((item (make-instance (content-type rb) :sexp s)))
                          (cond ((valid item) (progn 
                                                                   (format-display *rbs-output-window*
                                                                        " Adding rule ~A~%" (rule-form item))
                                                                   (store item rb)))
                                      (t (format-display *rbs-output-window*
                                               " Ignoring bad rule~%"))))))
             (sexp rb)))



(defun add-index (item db)
      (setf (indices db)
                (sort 
                      (cons (index item) (indices db))
                      (function (lambda (x y) (string-lessp (car x)(car y)))))))



(defmethod list-of-assertions ((db database))
      (let (list-assertions)
          (dolist (assertion (contents db) list-assertions)
                (dolist (item (cdadr assertion))
                      (push (format nil "~A" (car item)) list-assertions)))))
     


(defmethod list-of-rules ((rset database))
      (mapcar #'(lambda (rule)
                               (rule-form rule))
             (itemlist rset)))



;(defmethod show :before ((db database) &optional (s t))



(defmethod show ((db database) &optional (s t))
      (format s "~%database contents:~%")
      (mapc 
             (function (lambda (x) (print x)))
             (indices db))
      t)



(defmethod show ((rset rule-database) &optional s)
      (or s (setq s t))
      (mapc 
             (function (lambda (x) (print-rule x s)))
             (itemlist rset))
      t)




;;; store-assertion: <assertion><reason><database> => {t,nil}
;; rule is the rule that gave rise to this assertion
;; environment is the alist in force when the rule was fired
;; result t means that the assertion was added
;; result nil means that it was already present. In this case
;; the assertion is not added and no trace of the new derivation
;; is recorded.


;;; fetch <list of conjuncts> <database> <list of (e,i) pairs>
;; where e is an environments and i is other information
;; indicating which database items were matched in producing bindings
;; for that environment.
;; the empty environment is always (list nil)
;;
;; ==>
;; nil if fetch fails
;; a list ((e1.i1)..(en.in)} of pairs if fetch succeeds.
;; 
;; In the case of a fetch where no new bindings are created, the input
;; environment is returned.




(defun fetch (listconj d &optional (listenv (list (cons *newenv* nil))))
      (cond
                  ((null listenv) nil)
                  ((null listconj) listenv)
                  (t (fetch (cdr listconj) d (fetch1 (car listconj) d listenv)))))
    
    

(defun fetch1 (simplepat d listenv &aux result)
      (cond 
                  ((is-special simplepat)
                   (mapc
                          (lamda (e)
                           (let ((env-after (levalpat simplepat (car e))))
                               (or (eq env-after *fail*)
                                     (push 
                                           (mk-fetch-result env-after t e)
                                           result))))
                          listenv))
                  (t
                     (let ((bucket (get-candidates simplepat d)))
                         (cond (bucket 
                                      (mapc 
                                             (lamda (fres)
                                              (mapc
                                                     (lamda (candidate)
                                                      (let 
                                                             ((alist
                                                               (match-fetch simplepat candidate fres)))
                                                          (or
                                                                (eq alist *fail*) 
                                                                (push
                                                                      (mk-fetch-result alist candidate fres)
                                                                      result))))
                                                     bucket))
                                             listenv))))))
      result)




(defun match-fetch (pat dbnode fresult)
      (cond ((typep dbnode 'horn-rule-invocation)
                   (unify (cdr pat) (cdr (car (head dbnode))) (car fresult)))
                  (t (match (cdr pat) (cdr (car dbnode)) (car fresult))))) ;*


;* we need only match cdrs since cars must match



(defun mk-fetch-result (alist dbnode fresult)
      (cond ((eq (type-of dbnode) 'horn-rule-invocation)
                   (setf (env dbnode) alist)
                   dbnode)
                  (t (list alist (cons dbnode (cadr fresult))))))



;(defun mk-fetch-result (alist dbnode fresult)
;  (list alist (cons dbnode (cadr fresult))))



;;; get-candidates <pattern> <database>
;; yields the bucket accessed by the pattern index
;; (ie the predicate name and the number of arguments
;; a la prolog

(defmethod get-candidates (pattern (db assertion-database) &aux x)
      (let ((pred (car pattern))
              (numbargs (length (setq pattern (cdr pattern)))))
          (and (setq x (assoc pred (contents db)))
                   (cdr (assoc numbargs (cdr x))))))



(defmethod get-candidates (pattern (db horn-rule-database) &aux x)
      (let ((pred (car pattern))
              (numbargs (length (setq pattern (cdr pattern)))))
          (and (setq x (assoc pred (contents db)))
                   (mapcar (function invocation) (cdr (assoc numbargs (cdr x)))))))



(defun get-dbref (fact db)
      (assoc (cdr fact) (get-candidates fact db) :test 'equal))



;;;;;;;;;;;;; backward-chainer;;;;;;;;;;;;;;;

(defun backchain (q rdb adb)
      (process-pending-events)
      (format-display  *rbs-output-window*
           "~% Backward chaining ...~%~%")
      (let ((result (prove-all q rdb adb))
              (qvars (varsin q nil))
              (more-solutions t))
          (format-display  *rbs-output-window*
               "~% Backward chainer finished.~%")
          (cond ((eq result *fail*)
                       (pop-up-message-dialog
                             *rbs-main-window* "Backward chaining"
                            "       No solutions.     "
                             nil "OK"))
                      ((null qvars)
                        (pop-up-message-dialog
                             *rbs-main-window* "Backward chaining"
                              (format nil "~A is TRUE" (car q))
                              nil "OK"))
                      (t
                         (mapc (lamda (e)
                                          (when more-solutions
                                                 (let ((result (format nil "Solution:~%")))
                                                     (terpri)
                                                     (mapc (lamda (v)
                                                                      (let ((x (lookupval v e)))
                                                                          (setq result (concatenate 'string
                                                                                                       result
                                                                                                       (format nil "~%~A = ~A"
                                                                                                            v
                                                                                                            (if (is-special x)
                                                                                                               (evalspecial (car x) (cdr x) nil nil)
                                                                                                               x))))))
                                                            qvars)
                                                     (setq result (concatenate 'string
                                                                                  result
                                                                                  (format nil "~%~%More solutions?")))
                                                     (unless (yes-or-no-p result) (setq more-solutions nil)))))
                                result)
                         t))))



(defun prove-all (goals db adb &optional (env '((t . t))))
      "Find a solution to the conjunction of goals."
      (cond ((eq env *fail*) *fail*)
                  ((null goals) (list env))
                  (t
                     (mapcan
                            (function 
                                 (lambda (e)
                                     (prove-all (cdr goals) db adb e)))
                            (prove1 (car goals) db adb env)))))



(defun prove1 (goal db adb env)
      "Return a list of possible solutions to goal."
      (when (listp goal)
             (format-display *rbs-output-window*
                  "~% Goal ~A~%" (enveval goal env))
             (setq goal (renveval goal)))
      (cond 
                  ((is-special goal)
                   (let ((alist (levalpat goal env)))
                       (cond ((eq alist *fail*) nil)
                                   (t (list alist)))))
                  ((eq (car goal) 'not)
                   (prove-not (cadr goal) db adb env))
                  (t
                     (mapcan
                            #'(lambda (clause)
                                    (cond 
                                                ((typep clause 'horn-rule-invocation)
                                                 (let ((e1 (unify goal (car (head clause)) env)))
                                                     (cond 
                                                                 ((eq e1 *fail*) nil)
                                                                 (t
                                                                    (format-display *rbs-output-window*
                                                                         " Try rule ~A: ~A =>~%             ~A~%" 
                                                                         (rnumber (parent-rule clause))
                                                                         (enveval (subgoals clause) e1)
                                                                         (enveval (head clause) e1))
                                                                    (prove-all (subgoals clause) db adb e1)))))
                                                (t 
                                                   (let ((e1 (unify goal (car clause) env)))
                                                       (cond 
                                                                   ((eq e1 *fail*) nil)
                                                                   (t (list e1)))))))
                            (append
                                  (get-candidates goal db)
                                  (get-candidates goal adb))))))
	


(defun prove-not (goal db adb env)
      (let ((try (prove1 goal db adb env)))
          (cond ((eq try *fail*) (list env))
                      (t *fail*))))



#|
;;;;;;;;;;;;; tracing ;;;;;;;;;;;;;;;

;;; for the time being the global variable *spied*
;;; lives in pail-lib/global.cl

(defun spied (pred)
      (memq pred *spied*))

(defun spy (pred)
      (or (spied pred)
            (push pred *spied*))
      *spied*)



(defun unspy (&optional pred)
      (cond ((and pred (spied pred))
                   (delete pred *spied*))
                  (t (setq *spied* nil)))
      *spied*)



(defun spyall (db)
      (mapc (lamda (x) (spy (car x)))
             (indices db))
      *spied*)

  

|#