;;; -*- Mode: LISP; Package: rbs; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA
;;;
;;; ************************************************************************
;;;
;;; Filename:   db.cl
;;; Short Desc: routines for simple matching
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   2.94
;;; Author:     M. Rosner
;;;
;;; Copyright (c) 1994 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------

(in-package :rbs)

;;; note - many aspects of this file were inspired by Peter
;;; Norvig's book

(defconstant *fail* nil)
(defconstant *newenv* '((t . t)))
(defvar *occurs-check* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  matching and unification ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;




;;; lookup <var> <env> => *unbound* if var is unbound
;;                     => first binding in e otherwise
;;  n.b we need to distinguish the case where 
;;  a variable is bound to nil so we simply return the
;;  pair delivered by assoc

(defun lookup (x e)
      (let ((v (assoc x e)))
          (cond (v (cdr v))
                      (t *unbound*))))



(defun lookup (x e)
      (assoc x e :test 'eq))



;;; lookupval <var> <env> => self if unbound
;;                        => sexpression otherwise
;; note that lookupval takes care of the problem of
;; dereferencing variables that are bound to other variables

(defun lookupval (x e)
      (let ((v (lookup x e)))
          (cond 
                      ((null v) x)
                      (t (setq v (cdr v))
                         (cond 
                                     ((isvar v) (lookupval v e))
                                     (t (enveval v e)))))))




;;;enveval <pattern> <env> => element (ie list with no variables)
;;;simply substitutes for variables in the rhs
;;;used to make an assertion before adding it to the db
;;;when executing the rhs of a rule.

(defun enveval (p e)
      (cond ((isvar p) (lookupval p e))
                  ((atom p) p)
                  (t (cons (enveval (car p) e) (enveval (cdr p) e)))))



;;;a variable is a symbol beginning with '?'

(defun isvar (v)
      (and (symbolp v)
               (char= (schar (symbol-name v) 0) #\?)))
	 


;;; extracts variables in an expression

(defun varsin (e res)
      (cond ((isvar e)
                   (cond ((member e res) res)
                               (t (cons e res))))
                  ((atom e) res)
                  (t (setq res (varsin (car e) res))
                     (varsin (cdr e) res))))



;;; bind <var> <val> <env> => <env>
;;  takes account of special case where
;; env = *newenv*

(defun bind (var val env)
      (cond 
                  ((eq env *newenv*) (list (cons var val)))
                  (t (cons (cons var val) env))))
	


;;; simple matcher -- no variables in data

;;; match <pattern><data><env> => *fail* if no match
;;                             => binding list ((var.val)...)  

(defun match (pat dat env)
      (cond
                  ((eq env *fail*) *fail*)
                  ((isvar pat)
                   (let ((val (lookup pat env)))
                       (cond
                                   (val (match (cdr val) dat env))
                                   (t (bind pat dat env)))))
                  ((symbolp pat)
                   (cond ((eq pat dat) env)
                               (t *fail*)))
                  ((and (consp pat)(consp dat))
                   (match (car pat)
                    (car dat) 
                    (match (cdr pat)(cdr dat) env)))
                  (t *fail*)))




(defun unify (x y env)
      (cond ((eq env *fail*) *fail*)
                  ((eql x y) env)
                  ((isvar x) (unify-var x y env))
                  ((isvar y) (unify-var y x env))
                  ((and (consp x) (consp y))
                   (unify (cdr x) (cdr y) 
                    (unify (car x) (car y) env)))
                  (t *fail*)))



(defun unify-var (var x env &aux val)
      (cond ((setq val (lookup var env))
                   (unify (cdr val) x env))
                  ((and (isvar x) (setq val (lookup x env)))
                   (unify var (cdr val) env))
                  ((and *occurs-check* (occurs-check var x env))
                   *fail*)
                  (t (bind var x env))))



(defun occurs-check (var x env)
      (cond ((eq var x) t)
                  ((and (isvar x) (get-binding x env))
                   (occurs-check var (lookup x env) env))
                  ((consp x) (or (occurs-check var (car x) env)
                                            (occurs-check var (cdr x) env)))
                  (t nil)))



