;;; -*- Mode: LISP; Package: rbs; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA
;;;
;;; ************************************************************************
;;;
;;; Filename:   rule.cl
;;; Short Desc: objects for rule sets and associated methods/functions
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   94.6
;;; Author:     M. Rosner
;;;
;;; Copyright (c) 1994 Istituto Dalle Molle (IDSIA)
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; replaces ruleset.cl
;;;	
;;; --------------------------------------------------------------------------

;;; --------------------------------------------------------------------------
;;; INTERFACE

;; make-instance 'rule
;; make-instance 'horn-rule
;; print-rule <rule> [<stream>]
;; trac/utrac <rule-db> [<rule-number>]
;;; --------------------------------------------------------------------------

;;; --------------------------------------------------------------------------
;;; Package 

(in-package :rbs)

;;; 
;;;	
;;; --------------------------------------------------------------------------


(defun nextvar ()
  (gensym "?"))


;;;class RULE

(defclass rule (db-item)
       
       ((tag	     
         :accessor tag		;user supplied name (optional)
         :initarg :tag
         :initform nil)
        
        (rnumber	
         :accessor rnumber	;system supplied rule number
         :initform 1)
        
        (traceflag	
         :accessor traceflag	;trace flag for debugging
         :initform nil)
        
        (lhs		
         :accessor lhs
         :initarg :lhs
         :initform nil)
        
        (rhs		
         :accessor rhs
         :initarg :rhs
         :initform nil)
        
        (vars		
         :accessor vars
         :initarg :vars)))



(defclass horn-rule (rule) ())


(defclass horn-rule-invocation ()
       
       ((parent-rule	
         :accessor parent-rule
         :initarg :parent-rule
         :initform nil)
        
        (vtab		
         :accessor vtab
         :initarg :vtab)
        
        (ivtab	
         :accessor ivtab
         :initarg :ivtab)
        
        (env          :accessor env
         :initform nil)
        
        
        (subgoals	
         :accessor subgoals
         :initarg :subgoals
         :initform nil)
        
        (head		
         :accessor head
         :initarg :head
         :initform nil)))



(defmethod invocation ((r rule))
      (make-instance 'horn-rule-invocation
             :parent-rule r))




(defmethod initialize-instance :after ((h horn-rule-invocation) &rest restargs)
      (declare (ignore restargs))
      (let ((parent-rule (parent-rule h)))
          
          (setf (vtab h) (mkvtab (vars parent-rule)))
          (setf (ivtab h)
                    (mapcar (function 
                                          (lambda (pair)
                                              (cons (cdr pair) (car pair))))
                           (vtab h)))
          (setf (subgoals h) (sublis (ivtab h) (lhs parent-rule)))
          (setf (head h) (sublis (ivtab h) (rhs parent-rule)))))



(defmethod initialize-instance :after ((r rule) &rest restargs)
      (declare (ignore restargs))
      (cond 
                  ((check-syntax r)
                   (let ((sexp (sexp r)))
                       (setf (valid r) t)
                       (setf (lhs r) (nth 0 sexp))
                       (setf (rhs r) (nth 1 sexp))
                       (setf (index r) 
                                 (list (caar (rhs r)) (length (cdar (rhs r)))))
                       (setf (tag r) (nth 2 sexp))
                       (setf (vars r) (varsin (append (lhs r) (rhs r)) nil))))))



 ;(defmethod initialize-instance :after ((r horn-rule) &rest restargs)
 ; (declare (ignore restargs))
 ; (let ((head (car (rhs r))))
 ;   (setf (index r)
 ;	  (list (car head)(length (cdr head))))))
    



(defmethod rule-form ((r rule))
      (apply
            'concatenate
            (append
                  (list 'string "IF  ")
                  (mapcar
                         #'(lambda (item)
                                 (format nil "~A AND " item))
                         (reverse (cdr (reverse (lhs r)))))
                  (list (format nil "~A  THEN  " (car (last (lhs r)))))
                  (mapcar
                         #'(lambda (item)
                                 (format nil "~A AND " item))
                         (reverse (cdr (reverse (rhs r)))))
                  (list (format nil "~A" (car (last (rhs r))))))))



(defmethod rule-form2 ((r rule))
      (list
         (apply
               'concatenate
               (append
                     (list 'string "IF            ")
                     (mapcar
                            #'(lambda (item)
                                    (format nil "~A AND " item))
                            (reverse (cdr (reverse (lhs r)))))
                     (list (format nil "~A" (car (last (lhs r)))))))
         (apply
               'concatenate
               (append
                     (list 'string "THEN    ")
                     (mapcar
                            #'(lambda (item)
                                    (format nil "~A AND " item))
                            (reverse (cdr (reverse (rhs r)))))
                     (list (format nil "~A" (car (last (rhs r)))))))))
                  



(defun print-rule (r stream)
      (format stream "~%~A~0,2T~A: ~A~1,12T=> ~A" 
           (cond ((traceflag r) "*")(t " "))
           (rnumber r)
           (lhs r)
           (rhs r)))




(defmethod check-syntax ((r rule))
      (let ((sexp (sexp r)))
          (and (listp sexp)
                   (listp (car sexp))
                   (and (caar sexp) (every 'is-assertion (car sexp)))
                   (listp (cadr sexp))
                   (every 'is-assertion (cadr sexp)))))



(defmethod check-syntax ((r horn-rule))
      (let ((sexp (sexp r)))
          (and (listp sexp)
                   (listp (car sexp))
                   (every 'is-assertion (car sexp))
                   (listp (cadr sexp))
                   (every 'is-assertion (cadr sexp))
                   (eq (length (cadr sexp)) 1))))



#|
;;; trac/utrac <ruleset> [<ruleno>]
;;  trace/untrace specific rule or entire ruleset

(defun trac (rset &optional n) (set-trace n rset t))

(defun utrac (rset &optional n) (set-trace n rset nil))


(defun set-trace (n rset val)
 (cond (n
	(cond ((and (> (itemcount rset) n) (> n 0))
	       (setf (traceflag (nth (1- n) (itemlist rset))) val))
	      (t (warn *no-such-rule* n))))
       (t (mapc (function (lambda (r) (setf (traceflag r) val)))
		(itemlist rset))))
 t)

|#

(defun mkvtab (vars)
  (mapcar
   (function (lambda (v) (cons (nextvar) v)))
   vars))










