IMPLEMENTATION MODULE ErrorX68;
(* Displays error messages for X68000 cross assembler *)

   FROM Terminal IMPORT
      WriteString, WriteLn;

   IMPORT Terminal;   (* for Read/Write *)

   FROM FileSystem IMPORT
      File;

   IMPORT FileSystem;   (* for WriteChar *)

   FROM Strings IMPORT
      Length;

   FROM Conversions IMPORT
      ConvertCardinal;

   IMPORT ASCII;

   FROM Parser IMPORT
      Line, LineCount;


(*---
   TYPE
      ErrorType = (Dummy, TooLong, NoCode, SymDup, Undef, SymFull, Phase,
                   ModeErr, OperErr, BraErr, AddrErr, SizeErr, EndErr);

   VAR
      ErrorCount : CARDINAL;
                                                                  ---*)

   VAR
      FirstTime : BOOLEAN;



   PROCEDURE FileWriteString (VAR f : File; Str : ARRAY OF CHAR);

      VAR
         i : CARDINAL;
         top : CARDINAL;

      BEGIN
         top := HIGH (Str);
         i := 0;
         WHILE (i <= top) AND (Str[i] # 0C) DO
            FileSystem.WriteChar (f, Str[i]);
            INC (i);
         END;
      END FileWriteString;



   PROCEDURE Error (Pos : CARDINAL; ErrorNbr : ErrorType);
   (* Displays Error #ErrorNbr, then waits for any key to continue *)

      VAR
         i : CARDINAL;
         c : CHAR;
         CntStr : ARRAY [0..6] OF CHAR;

      BEGIN
         WriteLn;
         ConvertCardinal (LineCount, 1, CntStr);
         WriteString (CntStr);
         WriteString ("   ");
         WriteString (Line);   WriteLn;

         (* Make up for LineCnt so ^ in right spot *)
         FOR i := 1 TO Length (CntStr) DO
            Terminal.Write (' ');
         END;
         WriteString ("   ");

         IF Pos > 0 THEN
            FOR i := 1 TO Pos DO
               Terminal.Write (' ');
            END;
            Terminal.Write ('^');   WriteLn;
         END;
         
         CASE ErrorNbr OF
            TooLong : WriteString ("Identifier too long -- Truncated!");
         |  NoCode  : WriteString ("No such op-code.");
         |  SymDup  : WriteString ("Duplicate Symbol.");
         |  Undef   : WriteString ("Undefined Symbol.");
         |  SymFull : WriteString ("Symbol Table Full -- Maximum = 500!");
                      WriteLn;
                      WriteString ("Program Terminated.");   WriteLn;
                      HALT;
         |  Phase   : WriteString ("Pass 1/Pass 2 Address Count Mis-Match.");
         |  ModeErr : WriteString ("This addressing mode not allowed here.");
         |  OperErr : WriteString ("Error in operand format.");
         |  BraErr  : WriteString ("Error in relative branch.");
         |  AddrErr : WriteString ("Address mode error.");
         |  SizeErr : WriteString ("Operand size error.");
         |  EndErr  : WriteString ("Missing END Pseudo-Op.");
         ELSE
            WriteString ("Unknown Error.");
         END;
         WriteLn; 

         IF FirstTime THEN
            WriteString ("Hit any key to continue (^C to Terminate) .... ");
            Terminal.Read (c);   
            WriteLn;
            FirstTime := FALSE;
         ELSE
            Terminal.Read (c);
         END;
         
         INC (ErrorCount);
         IF ErrorCount > 500 THEN
            WriteString ("Too many errors!");   WriteLn;
            WriteString ("Program Terminated.");   WriteLn;
            HALT;
         END;
      END Error;



   PROCEDURE WriteErrorCount (VAR f : File);
   (* Error count output to Console & Listing file *)

      VAR
         CntStr : ARRAY [0..6] OF CHAR;
         Msg0 : ARRAY [0..25] OF CHAR;
         Msg1 : ARRAY [0..10] OF CHAR;
         Msg2 : ARRAY [0..20] OF CHAR;

      BEGIN
         Msg0 := "--->   END OF ASSEMBLY";
         Msg1 := "--->   ";
         Msg2 := " ASSEMBLY ERROR(S).";
         ConvertCardinal (ErrorCount, 1, CntStr);
            
         (* Messages to console *)
         WriteLn;
         WriteLn;
         WriteString (Msg0);   WriteLn;
         WriteString (Msg1);
         WriteString (CntStr);
         WriteString (Msg2);
         WriteLn;

         (* Messages to listing file *)
         FileSystem.WriteChar (f, ASCII.cr);   
         FileSystem.WriteChar (f, ASCII.lf);
         FileSystem.WriteChar (f, ASCII.cr);   
         FileSystem.WriteChar (f, ASCII.lf);

         FileWriteString (f, Msg0);
         FileSystem.WriteChar (f, ASCII.cr);   
         FileSystem.WriteChar (f, ASCII.lf);

         FileWriteString (f, Msg1);
         FileWriteString (f, CntStr);
         FileWriteString (f, Msg2);
         FileSystem.WriteChar (f, ASCII.cr);   
         FileSystem.WriteChar (f, ASCII.lf);

         FileSystem.WriteChar (f, ASCII.ff);   (* feed up next page *)
      END WriteErrorCount;


BEGIN   (* MODULE Initialization *)
   FirstTime := TRUE;
   ErrorCount := 0;
END ErrorX68.

