MODULE InitOperationCodes;
(* Module to construct the file containing the Operation Code Data Table *)

   FROM SYSTEM IMPORT
      ADR, TSIZE;

   FROM FileSystem IMPORT
      File, Response, Delete, Lookup, WriteNBytes, Close;

   FROM Terminal IMPORT
      WriteString, WriteLn;

   FROM Parser IMPORT
      TOKEN;


   CONST
      FIRST = 1;
      LAST = 118;

   TYPE
      ModeTypeA = (RegMem3,      (* 0 = Register, 1 = Memory *)
                   Ry02,         (* Register Ry -- Bits 0-2 *)
                   Rx911,        (* Register Rx -- Bits 9-11 *)
                   Data911,      (* Immediate Data -- Bits 9-11 *)
                   CntR911,      (* Count Register or Immediate Data *)
                   Brnch,        (* Relative Branch *)
                   DecBr,        (* Decrement and Branch *)
                   Data03,       (* Used for VECT only *)
                   Data07,       (* Branch & MOVEQ *)
                   OpM68D,       (* Data *)
                   OpM68A,       (* Address *)
                   OpM68C,       (* Compare *)
                   OpM68X,       (* XOR *)
                   OpM68S,       (* Sign Extension *)
                   OpM68R,       (* Register/Memory *)    
                   OpM37);       (* Exchange Registers *)
                   
      ModeTypeB = (Bit811,       (* BIT operations - bits 8/11 as switch *)
                   Size67,       (* 00 = Byte, 01 = Word, 10 = Long *)
                   Size6,        (* 0 = Word, 1 = Long *)
                   Size1213A,    (* 01 = Byte, 11 = Word, 10 = Long *)
                   Size1213,     (*            11 = Word, 10 = Long *)
                   Exten,        (* OpCode extension required *)
                   EA05a,        (* Effective Address - ALL *)
                   EA05b,        (* Less 1 *)
                   EA05c,        (* Less 1, 11 *)
                   EA05d,        (* Less 9, 10, 11 *)
                   EA05e,        (* Less 1, 9, 10, 11 *)
                   EA05f,        (* Less 0, 1, 3, 4, 11 *)
                   EA05x,        (* Dual mode - OR/AND *)
                   EA05y,        (* Dual mode - ADD/SUB *)
                   EA05z,        (* Dual mode - MOVEM *)
                   EA611);       (* Used only by MOVE *)
                   
      ModeA = SET OF ModeTypeA;
      ModeB = SET OF ModeTypeB;

      TableRecord = RECORD
                       Mnemonic : TOKEN;
                       Op : BITSET;
                       AddrModeA : ModeA;
                       AddrModeB : ModeB;
                    END;


   VAR
      Table68K : ARRAY [FIRST..LAST] OF TableRecord;
      i : CARDINAL;   (* index variable for initializing Table68K *)
      d : CARDINAL;   (* dummy for count on WriteNBytes *)
      f : File;   (* "OPCODE.DAT" *)


PROCEDURE One;   
BEGIN                                   
   i := 1;
   WITH Table68K[i] DO
      Mnemonic := "ABCD";
      Op := {15, 14, 8};
      AddrModeA := ModeA{Rx911, RegMem3, Ry02};
      AddrModeB := ModeB{};
   END;

   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ADD";
      Op := {15, 14, 12};
      AddrModeA := ModeA{OpM68D};
      AddrModeB := ModeB{EA05y};
   END;

   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ADDA";
      Op := {15, 14, 12};
      AddrModeA := ModeA{OpM68A};
      AddrModeB := ModeB{EA05a};
   END;

         
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ADDI";
      Op := {10, 9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e, Exten};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ADDQ";
      Op := {14, 12};
      AddrModeA := ModeA{Data911};
      AddrModeB := ModeB{Size67, EA05d};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ADDX";
      Op := {15, 14, 12, 8};
      AddrModeA := ModeA{RegMem3, Rx911, Ry02};
      AddrModeB := ModeB{Size67};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "AND";
      Op := {15, 14};
      AddrModeA := ModeA{OpM68D};
      AddrModeB := ModeB{EA05x};
   END;
   
   INC (i);             
   WITH Table68K[i] DO
      Mnemonic := "ANDI";
      Op := {9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e, Size67, Exten};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ASL";
      Op := {15, 14, 13, 8};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "ASR";
      Op := {15, 14, 13};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BCC";
      Op := {14, 13, 10};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BCHG";
      Op := {6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e, Exten, Bit811};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BCLR";
      Op := {7};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e, Exten, Bit811};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BCS";
      Op := {14, 13, 10, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BEQ";
      Op := {14, 13, 10, 9, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BGE";
      Op := {14, 13, 11, 10};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BGT";
      Op := {14, 13, 11, 10, 9};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BHI";
      Op := {14, 13, 9};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BLE";
      Op := {14, 13, 11, 10, 9, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BLS";
      Op := {14, 13, 9, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BLT";
      Op := {14, 13, 11, 10, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BMI";
      Op := {14, 13, 11, 9, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BNE";
      Op := {14, 13, 10, 9};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BPL";
      Op := {14, 13, 11, 9};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BRA";
      Op := {14, 13};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BSET";
      Op := {7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e, Exten, Bit811};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BSR";
      Op := {14, 13, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
END One;

PROCEDURE Two;
BEGIN   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "BTST";
      Op := {};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05c, Exten, Bit811};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "BVC";
      Op := {14, 13, 11};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "BVS";
      Op := {14, 13, 11, 8};
      AddrModeA := ModeA{Brnch};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "CHK";
      Op := {14, 8, 7};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05b};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "CLR";
      Op := {14, 9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "CMP";
      Op := {15, 13, 12};
      AddrModeA := ModeA{OpM68C};
      AddrModeB := ModeB{EA05a};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "CMPA";
      Op := {15, 13, 12};
      AddrModeA := ModeA{OpM68A};
      AddrModeB := ModeB{EA05a};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "CMPI";
      Op := {11, 10};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e, Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "CMPM";
      Op := {15, 13, 12, 8, 3};
      AddrModeA := ModeA{Rx911, Ry02};
      AddrModeB := ModeB{Size67};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBCC";
      Op := {14, 12, 10, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBCS";
      Op := {14, 12, 10, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBEQ";
      Op := {14, 12, 10, 9, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBF";
      Op := {14, 12, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;

   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBGE";
      Op := {14, 12, 11, 10, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBGT";
      Op := {14, 12, 11, 10, 9, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBHI";
      Op := {14, 12, 9, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBLE";
      Op := {14, 12, 11, 10, 9, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBLS";
      Op := {14, 12, 9, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBLT";
      Op := {14, 12, 11, 10, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBMI";
      Op := {14, 12, 11, 9, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBNE";
      Op := {14, 12, 10, 9, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBPL";
      Op := {14, 12, 11, 9, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBRA";
      Op := {14, 12, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;

   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBT";
      Op := {14, 12, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBVC";
      Op := {14, 12, 11, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DBVS";
      Op := {14, 12, 11, 8, 7, 6, 3};
      AddrModeA := ModeA{DecBr};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DIVS";
      Op := {15, 8, 7, 6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05b};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "DIVU";
      Op := {15, 7, 6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05b};
   END;

   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "EOR";
      Op := {15, 13, 12};
      AddrModeA := ModeA{OpM68X};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "EORI";
      Op := {11, 9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e, Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "EXG";
      Op := {15, 14, 8};
      AddrModeA := ModeA{OpM37};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "EXT";
      Op := {14, 11};
      AddrModeA := ModeA{OpM68S};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ILLEGAL";
      Op := {14, 11, 9, 7, 6, 5, 4, 3, 2};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;

   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "JMP";
      Op := {14, 11, 10, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05f};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "JSR";
      Op := {14, 11, 10, 9, 7};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05f};
   END;
END Two;
   
PROCEDURE Three;
BEGIN
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "LEA";
      Op := {14, 8, 7, 6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05f};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "LINK";
      Op := {14, 11, 10, 9, 6, 4};
      AddrModeA := ModeA{Ry02};
      AddrModeB := ModeB{Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "LSL";
      Op := {15, 14, 13, 9, 8, 3};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "LSR";
      Op := {15, 14, 13, 9, 3};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MOVE"; 
      Op := {};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size1213A, EA611};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MOVEA";
      Op := {6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{Size1213, EA05a};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MOVEM";
      Op := {14, 11, 7};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size6, EA05z, Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MOVEP";
      Op := {3};
      AddrModeA := ModeA{OpM68R};
      AddrModeB := ModeB{Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MOVEQ";
      Op := {14, 13, 12};
      AddrModeA := ModeA{Data07};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MULS";
      Op := {15, 14, 8, 7, 6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05b};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "MULU";
      Op := {15, 14, 7, 6};
      AddrModeA := ModeA{Rx911};
      AddrModeB := ModeB{EA05b};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "NBCD";
      Op := {14, 11};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);            
   WITH Table68K[i] DO
      Mnemonic := "NEG";
      Op := {14, 10};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "NEGX";
      Op := {14};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "NOP";
      Op := {14, 11, 10, 9, 6, 5, 4, 0};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "NOT";
      Op := {14, 10, 9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e};
   END;
   
   INC (i);            
   WITH Table68K[i] DO
      Mnemonic := "OR";
      Op := {15};
      AddrModeA := ModeA{OpM68D};
      AddrModeB := ModeB{EA05x};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ORI";
      Op := {};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e, Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "PEA";
      Op := {14, 11, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05f};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "RESET";
      Op := {14, 11, 10, 9, 6, 5, 4};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ROL";
      Op := {15, 14, 13, 10, 9, 8, 4, 3};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ROR";
      Op := {15, 14, 13, 10, 9, 4, 3};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ROXL";
      Op := {15, 14, 13, 10, 8, 4};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ROXR";
      Op := {15, 14, 13, 10, 4};
      AddrModeA := ModeA{CntR911};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "RTE";
      Op := {14, 11, 10, 9, 6, 5, 4, 1, 0};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "RTR";
      Op := {14, 11, 10, 9, 6, 5, 4, 2, 1, 0};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "RTS";
      Op := {14, 11, 10, 9, 6, 5, 4, 2, 0};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SBCD";
      Op := {15, 8};
      AddrModeA := ModeA{Rx911, RegMem3, Ry02};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SCC";
      Op := {14, 12, 10, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SCS";
      Op := {14, 12, 10, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SEQ";
      Op := {14, 12, 10, 9, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SF";
      Op := {14, 12, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SGE";
      Op := {14, 12, 11, 10, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;

   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SGT";
      Op := {14, 12, 11, 10, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
END Three;
   

PROCEDURE Four;
BEGIN
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SHI";
      Op := {14, 12, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SLE";
      Op := {14, 12, 11, 10, 9, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SLS";
      Op := {14, 12, 9, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SLT";
      Op := {14, 12, 11, 10, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SMI";
      Op := {14, 12, 11, 9, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SNE";
      Op := {14, 12, 10, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SPL";
      Op := {14, 12, 11, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "ST";
      Op := {14, 12, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "STOP";
      Op := {14, 11, 10, 9, 6, 5, 4, 1};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SUB";
      Op := {15, 12};
      AddrModeA := ModeA{OpM68D};
      AddrModeB := ModeB{EA05y};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SUBA";
      Op := {15, 12};
      AddrModeA := ModeA{OpM68A};
      AddrModeB := ModeB{EA05a};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SUBI";
      Op := {10};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e, Exten};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SUBQ";
      Op := {14, 12, 8};
      AddrModeA := ModeA{Data911};
      AddrModeB := ModeB{Size67, EA05d};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SUBX";
      Op := {15, 12, 8};
      AddrModeA := ModeA{Rx911, RegMem3, Ry02};
      AddrModeB := ModeB{Size67};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SVC";
      Op := {14, 12, 11, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SVS";
      Op := {14, 12, 11, 8, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "SWAP";
      Op := {14, 11, 6};
      AddrModeA := ModeA{Ry02};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "TAS";
      Op := {14, 11, 9, 7, 6};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "TRAP";
      Op := {14, 11, 10, 9, 6};
      AddrModeA := ModeA{Data03};
      AddrModeB := ModeB{};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "TRAPV";
      Op := {14, 11, 10, 9, 6, 5, 4, 2, 1};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{};
   END;
   
   INC (i);
   WITH Table68K[i] DO
      Mnemonic := "TST";
      Op := {14, 11, 9};
      AddrModeA := ModeA{};
      AddrModeB := ModeB{Size67, EA05e};
   END;
   
   INC (i);         
   WITH Table68K[i] DO
      Mnemonic := "UNLK";
      Op := {14, 11, 10, 9, 6, 4, 3};
      AddrModeA := ModeA{Ry02};
      AddrModeB := ModeB{};
   END;
END Four;


BEGIN
   One;   (* first part of initialization *)
   Two;   (* rest of initialization -- Split due implementation restriction *)
   Three;
   Four;

   Delete ("OPCODE.DAT", f);   (* Just in case file already exists *)
   Lookup (f, "OPCODE.DAT", TRUE); 
   IF f.res # done THEN  
      WriteString ("Unable to create OpCode File.");
      WriteLn;
      HALT;
   END;

   FOR i := FIRST TO LAST DO
      WriteNBytes (f, ADR (Table68K[i]), TSIZE (TableRecord), d);
   END;

   Close (f);
   IF f.res # done THEN 
      WriteString ("Unable to close OpCode File.");
      WriteLn;
   END;
END InitOperationCodes.

