IMPLEMENTATION MODULE Listing;
(* Creates a program listing, including Addresses, Code & Source. *)

   FROM FileSystem IMPORT
      File, WriteChar;

   FROM LongNumbers IMPORT
      LONG, LongPut;

   FROM Parser IMPORT
      TOKEN, Line;

   FROM SymbolTable IMPORT
      ListSymTab;

   FROM Conversions IMPORT
      ConvertCardinal;

   FROM Strings IMPORT
      Length;

   IMPORT ASCII;

      
   CONST
      LnMAX = 55;


   VAR
      LnCnt : CARDINAL;   (* counts number of lines per page *)
      PgCnt : CARDINAL;   (* count of page numbers *)



   PROCEDURE WriteStrF (VAR f : File; Str : ARRAY OF CHAR);
   (* Writes a string to the file *)

      VAR
         i : CARDINAL;
         top : CARDINAL;

      BEGIN
         top := HIGH (Str);
         i := 0;
         WHILE  (i <= top) AND (Str[i] # 0C) DO
            WriteChar (f, Str[i]);
            INC (i);
         END;
      END WriteStrF;



   PROCEDURE CheckPage (VAR f : File);
   (* Checks if end of page reached yet -- if so, advances to next page. *)

      VAR
         i : CARDINAL;
         PgCntStr : ARRAY [0..6] OF CHAR;
      
      BEGIN
         INC (LnCnt);
         IF LnCnt >= LnMAX THEN
            LnCnt := 1;
            INC (PgCnt);
            WriteChar (f, ASCII.ff);   (* Form Feed for new page *)
            ConvertCardinal (PgCnt, 1, PgCntStr);   (* Print New Page Number *)
            FOR i := 1 TO 60 DO
               WriteChar (f, ' ');
            END;

            WriteStrF (f, "Page ");
            WriteStrF (f, PgCntStr);

            FOR i := 1 TO 3 DO
               WriteChar (f, ASCII.cr);
               WriteChar (f, ASCII.lf);
            END;
         END;
      END CheckPage;



   PROCEDURE StartListing (VAR f : File);
   (* Sign on messages for listing file -- initialize *)
      BEGIN
         WriteChar (f, ASCII.ff);   (* Start on a clean page *)

         WriteStrF (f, "                 68000 Cross Assembler");
         WriteChar (f, ASCII.cr);
         WriteChar (f, ASCII.lf);

         WriteStrF (f, "         Copyright (c) 1985 by Brian R. Anderson");
         WriteChar (f, ASCII.cr);
         WriteChar (f, ASCII.lf);

         WriteChar (f, ASCII.cr);
         WriteChar (f, ASCII.lf);

         LnCnt := 1;
         PgCnt := 1;
      END StartListing;



   PROCEDURE WriteListLine (VAR f : File;
                            AddrCnt, ObjOp, ObjSrc, ObjDest : LONG;
                               nA,     nO,    nS,     nD    : CARDINAL);
   (* Writes one line to the Listing file, Including Object Code *)

      CONST
         ObjMAX = 30;

      VAR
         i : CARDINAL;

      BEGIN
         IF nA = 0 THEN   (* nA is always either 0 or 6.  Address field = 8 *)
            FOR i := 1 TO 8 DO
               WriteChar (f, ' ');
            END;
         ELSE
            LongPut (f, AddrCnt, 6);
            WriteChar (f, ' ');   
            WriteChar (f, ' ');
         END;
         LongPut (f, ObjOp, nO);   
         LongPut (f, ObjSrc, nS);  
         LongPut (f, ObjDest, nD); 
         i := 8 + nO + nS + nD;
         WHILE i < ObjMAX DO
            WriteChar (f, ' ');
            INC (i);
         END;

         WriteStrF (f, Line);
         WriteChar (f, ASCII.cr);
         WriteChar (f, ASCII.lf);
         
         CheckPage (f);
      END WriteListLine;



   PROCEDURE WriteSymTab (VAR f : File; NumSym : CARDINAL);
   (* Lists symbol table in alphabetical order *)

      VAR
         Label : TOKEN;
         Value : LONG;
         i, j, pos : CARDINAL;

      BEGIN
         LnCnt := 1;
         INC (PgCnt);

         WriteStrF (f, "             * * *  Symbolic Reference Table  * * *");
         FOR i := 1 TO 3 DO
            WriteChar (f, ASCII.cr);
            WriteChar (f, ASCII.lf);
         END;

         pos := 1;
         FOR i := 1 TO NumSym DO
            ListSymTab (i, Label, Value);
            WriteStrF (f, Label);
            FOR j := Length (Label) TO 8 DO
               WriteChar (f, ' ');
            END;
            WriteStrF (f, " : ");
            LongPut (f, Value, 8);
            INC (pos);
            IF pos > 3 THEN
               pos := 1;
               WriteChar (f, ASCII.cr);
               WriteChar (f, ASCII.lf);
               CheckPage (f);
            ELSE
               WriteStrF (f, "  |  ");
            END;
         END; 

         WriteChar (f, ASCII.cr);
         WriteChar (f, ASCII.lf);
         WriteChar (f, ASCII.ff);
      END WriteSymTab;

END Listing.

