IMPLEMENTATION MODULE OperationCodes;
(* Initializes lookup table for Mnemonic OpCodes.  Searches the table *)
(* and returns the bit pattern along with address mode information.   *)

   FROM SYSTEM IMPORT
      ADR, TSIZE;

   FROM FileSystem IMPORT
      File, Response, Lookup, ReadNBytes, Close;

   FROM Terminal IMPORT
      WriteString, WriteLn;

   FROM Strings IMPORT
      CompareStr;

   FROM Parser IMPORT
      TOKEN, STRING;

   FROM ErrorX68 IMPORT
      ErrorType, Error;


   CONST
      FIRST = 1;       (* First 68000 OpCode *)
      LAST = 118;      (* Last 68000 OpCode *)


(*---
(* These objects are declared in the DEFINITION MODULE *)

   TYPE
      ModeTypeA = (RegMem3,      (* 0 = Register, 1 = Memory *)
                   Ry02,         (* Register Rx -- Bits 0-2 *)
                   Rx911,        (* Register Ry -- Bits 9-11 *)
                   Data911,      (* Immediate Data -- Bits 9-11 *)
                   CntR911,      (* Count Register or Immediate Data *)
                   Brnch,        (* Relative Branch *)
                   DecBr,        (* Decrement and Branch *)
                   Data03,       (* Used for VECT only *)
                   Data07,       (* MOVEQ *)
                   OpM68D,       (* Data *)
                   OpM68A,       (* Address *)
                   OpM68C,       (* Compare *)
                   OpM68X,       (* XOR *)
                   OpM68S,       (* Sign Extension *)
                   OpM68R,       (* Register/Memory *)    
                   OpM37);       (* Exchange Registers *)
                   
      ModeTypeB = (Bit811,       (* BIT operations - bits 8/11 as switch *)
                   Size67,       (* 00 = Byte, 01 = Word, 10 = Long *)
                   Size6,        (* 0 = Word, 1 = Long *)
                   Size1213A,    (* 01 = Byte, 11 = Word, 10 = Long *)
                   Size1213,     (* 11 = Word, 10 = Long *)
                   Exten,        (* OpCode extension required *)
                   EA05a,        (* Effective Address - ALL *)
                   EA05b,        (* Less 1 *)
                   EA05c,        (* Less 1, 11 *)
                   EA05d,        (* Less 9, 10, 11 *)
                   EA05e,        (* Less 1, 9, 10, 11 *)
                   EA05f,        (* Less 0, 1, 3, 4, 11 *)
                   EA05x,        (* Dual mode - OR/AND *)
                   EA05y,        (* Dual mode - ADD/SUB *)
                   EA05z,        (* Dual mode - MOVEM *)
                   EA611);       (* Used only by MOVE *)
                   
      ModeA = SET OF ModeTypeA;
      ModeB = SET OF ModeTypeB;

                                                            ---*)
   TYPE
      TableRecord = RECORD
                       Mnemonic : TOKEN;
                       Op : BITSET;
                       AddrModeA : ModeA;
                       AddrModeB : ModeB;
                    END;


   VAR
      Table68K : ARRAY [FIRST..LAST] OF TableRecord;
      i : CARDINAL;   (* index variable for initializing Table68K *)
      d : CARDINAL;  (* dummy for ReadNBytes *)
      f : File;


   PROCEDURE Instructions (MnemonSym : TOKEN; 
                           OpLoc : CARDINAL; VAR Op : BITSET; 
                           VAR AddrModeA : ModeA; VAR AddrModeB : ModeB);
   (* Uses lookup table to find addressing mode & bit pattern of opcode. *)

      CONST
         GoLower = -1;
         GoHigher = +1;

      VAR
         Top, Bottom, Look : CARDINAL;   (* index to Op-code table *)
         Found : BOOLEAN;
         Search : INTEGER;

      BEGIN
         Bottom := FIRST;
         Top := LAST;
         Found := FALSE;
         
         REPEAT   (* Binary Search *)
            Look := (Bottom + Top) DIV 2;
            Search := CompareStr (MnemonSym, Table68K[Look].Mnemonic);
           
            IF Search = GoLower THEN
               Top := Look - 1;
            ELSIF Search = GoHigher THEN
               Bottom := Look + 1;
            ELSE   (* Got It! *)
               Found := TRUE;
            END;
         UNTIL (Top < Bottom) OR Found;

         IF Found THEN
            (* Return the instruction, mode, and address restristictions *)
            Op := Table68K[Look].Op;
            AddrModeA := Table68K[Look].AddrModeA;
            AddrModeB := Table68K[Look].AddrModeB;
         ELSE
            Error (OpLoc, NoCode);
         END;
      END Instructions;


BEGIN   (* MODULE Initialization *)
   Lookup (f, "OPCODE.DAT", FALSE);   (* Try default drive first *)
   IF f.res # done THEN
      Lookup (f, "A:OPCODE.DAT", FALSE);
      IF f.res # done THEN
         Lookup (f, "B:OPCODE.DAT", FALSE);
         IF f.res # done THEN
            WriteString ("Can't Find 'OPCODE.DAT'.");
            WriteLn;
            HALT;
         END;
      END;
   END;

   FOR i := FIRST TO LAST DO
      ReadNBytes (f, ADR (Table68K[i]), TSIZE (TableRecord), d);
   END;

   Close (f);
   IF f.res # done THEN
      (* Don't worry about it! *)
   END;
END OperationCodes.



