MODULE X68000;
(*------------------------------------------------------------------*)
(*                                                                  *)
(*                    MC68000 Cross Assembler                       *)
(*            Copyright (c) 1985 by Brian R. Anderson               *)
(*                                                                  *)
(*   This program may be copied for personal, non-commercial use    *)
(*   only, provided that the above copyright notice is included     *)
(*   on all copies of the source code.  Copying for any other use   *)
(*   without the consent of the author is prohibited.               *)  
(*                                                                  *)
(*------------------------------------------------------------------*)

   IMPORT Debug;

   FROM Terminal IMPORT
      WriteString, WriteLn, ReadString;

   FROM FileSystem IMPORT
      File, Response, Delete, Lookup, Reset, Close;

   FROM Strings IMPORT
      CompareStr, Assign, Concat, Length;

   IMPORT Strings;   (* For Delete *)

   IMPORT ASCII;

   FROM LongNumbers IMPORT
      LONG;

   FROM SymbolTable IMPORT
      SortSymTab;

   FROM Parser IMPORT
      TOKEN, OPERAND, STRING, LineCount, LineParts;

   FROM CodeGenerator IMPORT
      LZero, AddrCnt, Pass2, BuildSymTable, AdvAddrCnt, GetObjectCode;

   FROM Listing IMPORT
      StartListing, WriteListLine, WriteSymTab;

   FROM Srecord IMPORT
      StartSrec, WriteSrecLine, EndSrec;

   FROM ErrorX68 IMPORT
      ErrorCount, WriteErrorCount;


   TYPE
      FileName = ARRAY [0..14] OF CHAR;


   VAR
      SourceFN, ListFN, SrecFN : FileName;
      Source, List, Srec : File;
      Label, OpCode : TOKEN;
      SrcOp, DestOp : OPERAND;
      EndFile : BOOLEAN;
      NumSyms : CARDINAL;
      ObjOp, ObjSrc, ObjDest : LONG;
      nA, nO, nS, nD : CARDINAL;



   PROCEDURE MakeNames (VAR S, L, R : FileName);
   (* builds names for Source, Listing & S-Record files *)

      VAR
         T : FileName;   (* temporary work name *)
         i, l : CARDINAL;

      BEGIN
         L := '';   R := '';   (* set Listing & S-rec names to null *)

         i := 0;   l := 0;
         WHILE (S[i] # 0C) AND (S[i] # ' ') DO
            IF S[i] = '.' THEN   (* mark beginning of file extension *)
               l := i;
            END;
            S[i] := CAP (S[i]);
            INC (i);
         END;
      
         IF S[i] = ' ' THEN
            Strings.Delete (S, i, Length (S) - i);
         END;

         Assign (S, T);
         IF l = 0 THEN
            Concat (T, ".ASM", S);
         ELSE   
            Strings.Delete (T, l, i - l);
         END;

         Concat (T, ".LST", L);
         Concat (T, ".S", R);
      END MakeNames;



   PROCEDURE OpenFiles;
      BEGIN
         Lookup (Source, SourceFN, FALSE);
         IF Source.res # done THEN
            WriteLn;
            WriteString ("No Source File: ");   WriteString (SourceFN);   
            WriteLn;
            HALT;
         END;

         Delete (ListFN, List);   (* Just in case file already exists *)
         Lookup (List, ListFN, TRUE);   
         IF List.res # done THEN    (* DOS may trap this *)
            WriteLn;
            WriteString ("Cannot create disk files!");   WriteLn;
            HALT;
         END;

         Delete (SrecFN, Srec);
         Lookup (Srec, SrecFN, TRUE);
         IF Srec.res # done THEN
            WriteLn;
            WriteString ("Cannot create disk files!");   WriteLn;
            HALT;
         END;
      END OpenFiles;



   PROCEDURE StartPass2;
      BEGIN
         Reset (Source);
         IF Source.res # done THEN
            WriteString ("Unable to 'Reset' Source file for 2nd Pass.");
            WriteLn;
            HALT;
         END;
         Pass2 := TRUE;   (* Pass2 IMPORTed from CodeGenerator *)
         AddrCnt := LZero;   (* Assume ORG = 0 to start *)
         ErrorCount := 0;   (* ErrorCount IMPORTed from ErrorX68 *)
         LineCount := 0;   (* LineCount IMPORTed from Parser *)
         EndFile := FALSE;
      END StartPass2;



   PROCEDURE CloseFiles;
      BEGIN
         Close (Source);
         Close (List);
         Close (Srec);
         IF (Source.res # done) OR (List.res # done) OR (Srec.res # done) THEN
            WriteString ("Error closing files...");   WriteLn;
            HALT;
         END;
      END CloseFiles;



BEGIN   (* X68000 -- main program *)
   WriteLn; 
   WriteString ("Enter Source Filename: ");
   ReadString (SourceFN);
   WriteLn;

   MakeNames (SourceFN, ListFN, SrecFN);   

   OpenFiles;

   WriteLn;
   WriteString ("                 68000 Cross Assembler");   WriteLn;
   WriteString ("         Copyright (c) 1985 by Brian R. Anderson");
   WriteLn;   WriteLn;
   WriteString ("                 Assembling ");   WriteString (SourceFN);  
   WriteLn;   WriteLn;   WriteLn;


(*---
    Begin Pass 1 
                  ---*)
   WriteString ("PASS 1");   WriteLn;
   AddrCnt := LZero;   (* Assume ORG = 0 to start *)
   EndFile := FALSE;

   REPEAT
      LineParts (Source, EndFile, Label, OpCode, SrcOp, DestOp);
      
      BuildSymTable (AddrCnt, Label, OpCode, SrcOp, DestOp);

      AdvAddrCnt (AddrCnt);

   UNTIL EndFile OR (CompareStr (OpCode, "END") = 0);


(*---
   Begin Pass 2 
               ---*)
   WriteString ("PASS 2");   WriteLn;
   StartPass2;   (* get Source file, Parser & ErrorX68 ready for 2nd pass *)
   SortSymTab (NumSyms);
   StartListing (List);
   StartSrec (Srec, SourceFN);

   REPEAT
      LineParts (Source, EndFile, Label, OpCode, SrcOp, DestOp);

      GetObjectCode (Label, OpCode,
                     SrcOp,  DestOp, 
                     AddrCnt, ObjOp, ObjSrc, ObjDest, 
                     nA,      nO,    nS,     nD      );

      WriteListLine (List, AddrCnt, ObjOp, ObjSrc, ObjDest, nA, nO, nS, nD);

      WriteSrecLine (Srec, AddrCnt, ObjOp, ObjSrc, ObjDest, nA, nO, nS, nD); 
      
      AdvAddrCnt (AddrCnt);

   UNTIL EndFile OR (CompareStr (OpCode, "END") = 0);

   EndSrec (Srec);   (* Also: Finish off any partial line *)
   WriteErrorCount (List);   (* Error count output to Console & Listing file *)
   WriteSymTab (List, NumSyms);   (* Write Symbol Table to Listing File *)
   CloseFiles;
END X68000.


