;
;	Terra Nova Communications multi-tasking kernel
;	Initialization and task-switcher
;
;	Note: this is not intended to be a complete listing.  It's only
;	a sample of some of the techniques used in our system.
;

		PSECT	Kernel
	
;		External symbols (defined in other code segments)
		EXTERN	VecTable,	;vector table for hardware vector list
			JMPTable,	;jump table for system calls
			JMPTabLen,	;length of jump table in longwords
			KernEnd,	;end of kernel code item in heap
			IOInit,		;our private I/O initialization routine
			HeapInit,	;our private heap initialization
			SysInit,	;system variable initializer
			SysConMon,	;entry point for system console
					;monitor task
			HeapMunger,	;entry point for heap munger task
			DiskMunger	;entry point for disk munger task

;		Entry points in this module (referenced from elsewhere)
		ENTRY   Start,		;primary entry point to boot our OS
			ConSwitch,	;main context switcher
			ConSwSleep	;alternate context switcher (puts
					;calling task to sleep)

;		Include files (mostly equates)
		INCLUDE	SysEqu		;contains the low-memory absolute
					;address equates (jump table, etc)
		INCLUDE	HeapDef		;defines the heap data structure
		INCLUDE	SysIO		;contains hardware I/O equates

;		Miscellaneous storage
CodeHeap	DS.L	8		;heap header for kernel heap item
StackEnd	DS.L	40		;system stack before tasking starts
StackBegin	DS.L	0		;top of startup stack area

;		Pre-tasking initialization
;		this code works in single-task mode
;		prior to the invocation of the context switcher
Start			;Initial entry.  Calling operating system is still
 			;alive and kicking at this point.
TakeOver	LEA	ReEntry,A1	;point to re-entry instruction
		MOVE.L	A1,$20.W	;move short absolute to the vector
					;for privilege exceptions
		MOVE	USP,A0		;try a privileged instruction.  If it
			;works, then we're in priv. mode. If not, then trap to
 			;ReEntry and be in privileged mode anyway
ReEntry		LEA	StackBegin,A7	;set up initial stack

;		Turn off all interrupts in the system
;		Note: this is device-specific code.
;		The labels in the operand fields are from our own
;		SysIO include file.
		CLR.B	FDCIntMask	;clear floppy disk & system console
		CLR.B	HDIntMask	;clear hard disk completion int. mask
		CLR.B	SerIO1IntMask	;clear serial boards
		CLR.B	SerIO2IntMask

;		Initialize the vector table
;		Copy the vectors from an assembled table (in another module) 
;		into the actual hardware vector list in low RAM
		LEA	VecTable,A0	;source (in another code segment)
		LEA	$0.W,A1		;destination (begins at $00 0000)
		MOVE	#191,D7		;192 longwords to move
VecMove		MOVE.L	(A0)+,(A1)+	;move a longword
		DBRA	D7,VecMove	;repeat till done (fast loop on 68010)

;		Copy system routine JMP table from assembled object code (in
;		another module) to low memory jump table, where everyone
;		can get at them.
		LEA	JMPTable,A0	;source
		LEA	System.W,A1	;dest. (name of first system call in 
			;the jump table.  "System" is from the SysEqu include
 			;file.  It's the context switcher)
		MOVE	#JMPTabLen/4,D7	;number of longwords to move
JPTMove		MOVE.L	(A0)+,(A1)+	;move a longword
		DBRA	D7,JPTMove	;repeat till done (fast loop on 68010)

;		Clear low memory to zero (between jmp table and kernel)
		LEA	StackEnd,A1	;point to top of destination
			;and bottom of destination (end of the jump table)
		LEA	System+JMPTabLen.W,A0
		SUBA	A0,A1		;calculate the length
		MOVE.L	A1,D7		;move to D7 for counting
		LSR.L	#4,D7 		;divide by 16 for 16-byte blocks
LowClr		CLR.L	(A0)+		;clear 16 bytes, quickly
		CLR.L	(A0)+
		CLR.L	(A0)+
		CLR.L	(A0)+
		DBRA	D7,LowClr	;do it until done.

;		Clear high memory to zero (between kernel and end of RAM)
;		(RAMEnd is first byte beyond RAM, defined in SysEqu)
		LEA	RAMEnd,A1	;point to top of destination
			;and bottom of destination (end of the jump table)
		LEA	KernEnd,A0
		SUBA	A0,A1		;calc the length
		MOVE.L	A1,D7		;move to D7 for counting
		LSR.L	#4,D7 		;divide by 16 for 16-byte blocks
HiClr		CLR.L	(A0)+		;clear 16 bytes, quickly
		CLR.L	(A0)+
		CLR.L	(A0)+
		CLR.L	(A0)+
		DBRA	D7,HiClr	;do it until done.

;		Initialize all of the primary I/O devices
;		Note: this is a device specific routine not treated in the article.
		JSR	IOInit
	
;		Initialize the heap
;		Note: this is a routine in the heap manager, which creates
;		valid heap headers for the three initial heap items discussed
;		in the text: the deletion below the kernel, the kernel code
;		item, and the deletion above the kernel.
		JSR	HeapInit

;		Initialize the system zone of low memory
;		Note: this sets up the TCB and master handle arrays, as
;		discussed in the text, as well as initializing the time of day 
;		and the date and the other miscellaneous system values.
		JSR	SysInit

;		Spawn off the initial tasks
;		This will create TCBs and TData items for the tasks, but won't
;		invoke them.  They're invoked only by the context switcher.
		LEA	SysConMon,A0	;point to system console entry point
		MOVE.L	#4096,D0	;tell it how much RAM for TData
		JSR	Spawn		;jump through jump table entry
			;("Spawn" is a jump table equate in SysEqu)
		LEA	HeapMunger,A0	;spawn the heap munger
		MOVE.L	#512,D0		;heap munger's TData size
		JSR	Spawn
		LEA	DiskMunger,A0	;Spawn the disk munger
		MOVE.L	#8192,D0	;(TData includes one disk buffer)
		JSR	Spawn

		LEA	TCB1.W,A2	;get address of first TCB in array
					;(TCB1 is defined in SysEqu)
		BRA.S	ConSw1		;now start the context switcher!

;		Context Switcher: primary version
;		Simple task-switch, nothing fancy.
;		SysFlags is a low-RAM system flag byte, defined in SysEqu.
;		The data structure for the TData item is defined in SysEqu.
;		The data structure for the TCB is defined in SysEqu.
ConSwitch	BTST	#StopSys,SysFlags.W ;task switching inhibited?
		BNE.S	ConSwX		;yes, exit back to caller
		MOVE.L	OurTCB(A5),A0	;get TCB address from TData
		SUBA.L	A5,SP		;subtract TData base addr from stack
		MOVE.L	SP,TCBSP(A0)	;save relative displacement in TCB

		MOVE.L	TCBNxt(A0),A2	;get address of next TCB
ConSw1		MOVE.L	TCBA5(A2),A5	;get new TData base address
		MOVE.L	TCBSP(A2),SP	;get stack relative displacement
		ADDA.L	A5,SP		;restore absolute address
ConSwX		RTS			;return to next task

;		Context Switcher: alternate version
;		Put the calling task to sleep.
ConSwSleep	BTST	#StopSys,SysFlags.W ;task switching inhibited?
		BNE.S	ConSwX		;yes, exit back to caller
					;without going to sleep
		MOVE.L	OurTCB(A5),A0	;get TCB address from TData
		SUBA.L	A5,SP		;subtract TData base addr from stack
		MOVE.L	SP,TCBSP(A0)	;save relative displacement in TCB

		MOVE.L	TCBNxt(A0),A2	;get address of next TCB
		MOVE.L	TCBPrev(A0),A1 	;get addr of previous TCB
		MOVE.L	A2,TCBNxt(A1)	;close the pointers around the now-
		MOVE.L	A1,TCBPrev(A2) 	;sleeping task.
		MOVE.B	#Sleep,TCBState(A0) ;mark it as asleep

		MOVE.L	TCBA5(A2),A5	;get new TData base address
		MOVE.L	TCBSP(A2),SP	;get stack relative displacement
		ADDA.L	A5,SP		;restore absolute address
		RTS			;return to next task
