
/************************************************************************
 *																								*
 *										GWIZ.C												*
 *																								*
 *		Utility to globally find files matching a given template.			*
 *																								*
 *		V2.0 - 06/09/96	J.L. Post - initial release.							*
 *																								*
 *	This utility started as a program called 'wiz', which like the			*
 *	utility 'whereis', searched the current drive for a file or set		*
 *	of files. The difference was that wiz would display the DOS date		*
 *	and time, and the file size. Later, wiz was expanded to 'gwiz',		*
 *	which stands for 'global wiz'. Gwiz searched all available hard		*
 *	drives. The current version of gwiz now accepts command line			*
 *	options:																					*
 *																								*
 *						-g[drive,drive,...] (eg: -gcd) searches specified		*
 *							hard drives. If no drives are specified (-g),		*
 *							all available hard drives will be searched.			*
 *																								*
 *						-l specifies a local search of only the current			*
 *							directory and all sub directories under the			*
 *							current directory.											*
 *																								*
 *							no options functions as the original wiz and			*
 *							searches the current drive from the root down.		*
 *																								*
 *	NOTE: This utility searches for files on hard drives only.				*
 *																								*
 *	Although the default is to search only the current drive, I kept		*
 *	the name 'gwiz' because it includes the option of globally search-	*
 *	ing all hard drives, and because, to be honest, I think the name		*
 *	is really neat. My program, I can name it what I want. :)				*
 *																								*
 *	This file was created using Borland's Brief editor, version 3.1.		*
 *	It may require reformatting for use with other editors (see the		*
 *	documentation for the 'tab' program. This program and the tab			*
 *	program are distributed as freeware under the GNU General Public		*
 *	License. Both programs were compiled using Borland Turbo C v2.0.		*
 *																								*
 ************************************************************************
 *																								*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.												*
 *																								*
 *	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.									*
 *																								*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the Free Software				*
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
 *																								*
 *	The file 'COPYING' contains the GNU General Public License.				*
 *																								*
 ************************************************************************/

#include	<stdio.h>
#include	<dos.h>
#include	<dir.h>
#include	<signal.h>
#include	<alloc.h>
#include	<ctype.h>

#define	byte	unsigned char
#define	word	unsigned int
#define	VER	2
#define	REV	0

int	found;					/* match found flag		*/
int	level;					/* directory level		*/
int	start_drive;			/* drive we started on	*/
char	curr_dir[64];			/* starting directory	*/
char	temp_dir[64];			/* processing directory	*/
int	global_flag;			/* global search flag	*/
int	local_flag;				/* local search flag		*/
char	file_name[64];			/* file search template	*/
byte	drive_tbl[32];			/* drive search table	*/

static int	drive_error = 0;

/*  FYI - file info structure:

struct ffblk						* 43 bytes total *
{
	char		ff_reserved[21];
	char		ff_attrib;				file attributes:
										bit 0		read only
										bit 1		hidden
										bit 2		system
										bit 3		volume label
										bit 4		directory
										bit 5		archive
										bits 6-15	reserved
	unsigned	ff_ftime;
											time of last update:
										bits 0-4		2 sec incr (0-29)
										bits 5-10	minutes (0-59)
										bits 11-15	hours (0-23)
	unsigned	ff_fdate;				date of last update:
										bits 0-4		day (1-31)
										bits 5-8		month (1-12)
										bits 9-15	year (rel to 1980)
	long		ff_fsize;
	char		ff_name[13];
}

*/

long	div_tbl[11] = {
	1000000000,		/* 1,000,000,000	1   billion		*/
	100000000,		/*   100,000,000	100 million 	*/
	10000000,		/*    10,000,000	10  million		*/
	1000000,			/*     1,000,000	1   million		*/
	100000,			/*       100,000  100 thousand	*/
	10000,			/*        10,000  10  thousand	*/
	1000,				/*         1,000	1   thousand	*/
	100,				/*           100	1   hundred		*/
	10,				/*            10	ten				*/
	1,					/*             1	one				*/
	0
};

void	num(long val, int flag, char chr);
void	do_files(char *name);
void	ctrlc_hand(int sig);
int	err_hand(void);
int	test_drive_ready(int drive);
void	usage(void);
char	last_char(char *ptr);

void usage(void)
{
	printf("\nGWIZ V%d.%d - Global whereis utility. "
			 "(C) 1994-96 by Jeffery L. Post.\n"
			 "Searches hard drives for specified file(s).\n"
			 "Filename may include wildcards (? and *).\n\n"
			 "Usage: gwiz [options] <filename>\n\n"
			 "Options:\n"
			 "\t-g[drive,drive,...] globally search hard drives.\n"
			 "\t    eg: -gcd searches only drives C and D.\n"
			 "\t        -g with no drives specified searches all"
			 " available hard drives.\n\n"
			 "\t-l local search in current directory and"
			 " all sub directories.\n\n"
			 "\tno options searches all directories on current drive.\n",
			VER, REV);
	exit(0);
}

int main(int argc, char *argv[])
{
	int	i, j;
	char	*opt;
	char	ch;

	if (argc < 2)
		usage();
	global_flag = local_flag = 0;
	for (j=0; j<32; j++)							/* default = search all drives */
		drive_tbl[j] = 1;							/* if global flag set */
	for (i=1; i<argc; i++)						/* scan for options */
	{
		opt = argv[i];
		if (*opt == '?')
			usage();
		if (*opt == '-' || *opt == '/')
		{
			opt++;
			if (*opt == '?')
				usage();
			else if (*opt == 'g' || *opt == 'G')
			{
				global_flag = 1;
				opt++;
				ch = toupper(*opt++);			/* if individual drives specified */

				if (isalpha(ch))					/* clear drive flag table */
				{
					for (j=0; j<32; j++)
						drive_tbl[j] = 0;
				}
				while (isalpha(ch))				/* get drives to search */
				{
					ch -= 'A';
					drive_tbl[ch+1] = 1;			/* flag drive to search */
					ch = toupper(*opt++);
				}
			}
			else if (*opt == 'l' || *opt == 'L')
				local_flag = 1;
		}
		else
			strcpy(file_name, opt);				/* assume file template */
	}

	found = level = 0;
	signal(SIGINT, ctrlc_hand);				/* allow ^C to abort us */
	harderr(err_hand);
	start_drive = getdisk();
	getcwd(curr_dir, 63);
	if (global_flag)								/* search all hard drives */
	{
		for (i=3; ; i++)							/* starting with drive C */
		{
			if (test_drive_ready(i))			/* if drive exists, */
			{
				if (drive_tbl[i])					/* and is enabled, */
					do_files(file_name);			/* go search it */
			}
			else										/* done when first non-existant */
				break;								/* drive is encountered */
		}
	}
	else if (local_flag)							/* search current directory down */
		do_files(file_name);
	else												/* search current drive from root */
	{
		chdir("\\");
		do_files(file_name);
	}
	if (!found)
		printf("File not found");
	ctrlc_hand(0);
}

void ctrlc_hand(int sig)
{
	chdir(temp_dir);
	setdisk(start_drive);
	chdir(curr_dir);
	signal(SIGINT, SIG_IGN);
	exit(sig);
}

void do_files(char *name)
{
	int	i, j, k;
	int	month, day, year;
	int	hour, min;
	char	this_dir[64];
	struct ffblk *fileinfo;

	getcwd(this_dir, 63);
	level++;
	fileinfo = (struct ffblk *) malloc(sizeof(struct ffblk));
	if (!fileinfo)
	{
		printf("\nCan't allocate memory - dir level %d\n", level);
		ctrlc_hand(1);
	}
	i = findfirst(name, fileinfo, 0x16);
	while (!i)
	{
		if (!found)
			putchar('\n');
		found = 1;
		j = printf("%s", this_dir);
		if (last_char(this_dir) == '\\')
			j += printf("%s", fileinfo->ff_name);
		else
			j += printf("\\%s", fileinfo->ff_name);
		i = 48 - j;
		for (j=0; j<i; j++)
			printf(".");
		num(fileinfo->ff_fsize, 1, '.');
		month = (fileinfo->ff_fdate >> 5) & 0xf;
		day = fileinfo->ff_fdate & 0x1f;
		year = ((fileinfo->ff_fdate >> 9) & 0x3f) + 80;
		printf("  %02d-%02d-%02d", month, day, year);
		hour = (fileinfo->ff_ftime >> 11) & 0x1f;
		min = (fileinfo->ff_ftime >> 5) & 0x3f;
		printf("  %02d:%02d\n", hour, min);
		i = findnext(fileinfo);
	}
	i = findfirst("*.*", fileinfo, 0x16);
	while (!i)
	{
		if (fileinfo->ff_name[0] != '.')	/* don't care about parent dir */
		{
			if (fileinfo->ff_attrib & 0x10)		/* if sub directory... */
			{
				if (chdir(fileinfo->ff_name))		/* go there... */
				{
					printf("\nCan't change directory!\n");
					ctrlc_hand(1);
				}
				do_files(name);			/* recursively search sub directories */
				chdir("..");
			}
		}
		i = findnext(fileinfo);
	}
	free((void *) fileinfo);
	--level;
}

char last_char(char *ptr)			/* find last character in directory name */
{
	while (*ptr)
		ptr++;
	--ptr;
	return(*ptr);
}

/* output number in a rational manner (with commas at appropriate places) */

void num(long val, int flag, char chr)
{
	int	i, j, comma, p;

	if (!val)
	{
		if (flag)
		{
			for (i=0; i<12; i++)
				printf("%c", chr);
		}
		printf("0");
		return;
	}

	comma = 1;
	p = 0;
	for (i=0; i<10; i++)
	{
		j = (int) (val / div_tbl[i]);
		if (j)								/* if non-zero digit, set print flag */
			p++;
		if (p)
			putchar(j + 0x30);
		else if (flag)
			putchar(chr);
		val -= (div_tbl[i] * j);
		--comma;
		if (!comma && i < 8)
		{
			if (p)
				putchar(',');
			else if (flag)
				putchar(chr);
			comma = 3;
		}
	}
}

/* see if drive is available */

int test_drive_ready(int drive)
{
	int	retval;

	retval = 0;
	drive_error = 0;
	setdisk(drive - 1);
	if (getdisk() != drive - 1 || drive_error)
		drive_error = 0;
	else
	{
		getcwd(temp_dir, 63);
		chdir("\\");
		if (!drive_error)
			retval = 1;
	}
	return(retval);
}

int err_hand(void)
{
	drive_error = 1;
	hardretn(0);
}

/* end of gwiz.c */

