/* MORSE CODE TRAINER:

This program helps you learn morse code by sounding one or more characrers
in morse, and waiting for you to type the corresponding keyboard character(s).
Error statistics and response times are displayed.

Command line options:
 /M - Force monochrome operation even if color screen detected
 /Q - Quiet, no beeps; show morse code in window
 /? - Display help line

Setup screen (Press ESC to exit):
 Difficulty:
  0  = Hardest, code table NEVER displayed
  1  = Code table displayed only after an error
  2  = Code table always displayed
  3  = Code table displayed + error entries are removed
  4+ = As above + (n-3) extra (incorrect) entries are removed

 Number of characters:
  This sets the number of morse characters that the program will send at
  one time.

 Starting character:
  This sets the starting position in the chart, for the purposes of limiting
  the test character set. Useful values are:
   1  = Allow all possible characters
   26 = Exclude alphabetic characters
   37 = Exclude alphabetic and numeric characters
   38 = Exclude alphabetic, numeric and punctuation

 Ending character:
  This sets the ending position in the chart, for the purposes of limiting
  the test character set. Useful values are:
   26 = Alphabetic characters only
   36 = Alphabetic characters plus numbers
   41 = Alphas, numbers and punctuation
   46 = Alphas, numbers, punctuation and abbreviations

 Dot length:
  Sets the duration (in milliseconds) of a DOT character.

 Dash length:
  Sets the duration (in milliseconds) of a DASH character.

 Dot/Dash spacing:
  Sets the duration (in milliseconds) of the silence period between
  dots and/or dashes in the same letter.

 Character spacing:
  Sets the duration (in mulliseconds) of the silence period between
  characters.

 Tone frequency:
  Sets the pitch of the code tone.

Function keys (use from within program):
 F1:  Again - replay the current code
 F2:  New   - Select a new code without solving this one
 F3:  Setup - Enter the setup screen
 F4:  Save  - Save configuration file *
 ESC: Exit  - Terminate the program

If the configuration file (MORSE.CFG) is present when the program starts,
the configuration settings will be loaded from it.

Copyright 1996-1997 Dave Dunfield
All rights reserved.

Permission granted for personal (non-commercial) use only.

Compile command: cc morse -fop
*/

#include <stdio.h>
#include <window.h>

#define	NMORSE	46		// Number of morse signals
#define	CFGSIZE	18		// Number of config bytes

/*
 * Morse code character table
 */
char *morse[] = {
	"._",		// A
	"_...",		// B
	"_._.",		// C
	"_..",		// D
	".",		// E
	".._.",		// F
	"__.",		// G
	"....",		// H
	"..",		// I
	".___",		// J
	"_._",		// K
	"._..",		// L
	"__",		// M
	"_.",		// N
	"___",		// O
	".__.",		// P
	"__._",		// Q
	"._.",		// R
	"...",		// S
	"_",		// T
	".._",		// U
	"..._",		// V
	".__",		// W
	"_.._",		// X
	"_.__",		// Y
	"__..",		// Z	25

	".____",	// 1	26
	"..___",	// 2
	"...__",	// 3
	"...._",	// 4
	".....",	// 5
	"_....",	// 6
	"__...",	// 7
	"___..",	// 8
	"____.",	// 9
	"_____",	// 0	35

	"._._._",	// .	36
	"__..__",	// ,
	"..__..",	// ?
	"_.._.",	// /
	"_..._",	// --	40

	"........",	// Error
	"._...",	// Wait
	"._._.",	// End of message
	"_._",		// Invitation to transmit (Same as 'K')
	"..._._",	// End of work
	"" };

/* Special symbols used in the code display */
char *symbols[] = {
	"1", "2", "3", "4", "5", "6", "7", "8", "9", "0",
	".", ",", "?", "/",
	"--                 (-)",
	"Error              (!)",
	"Wait               (@)",
	"End of Msg         (#)",
	"Invitation to xmit (K)",
	"End of work        ($)" };

/* Practice session parameters */
unsigned diff = 3, num = 1, start = 0, end = 25;

/*  Morse characteristics */
unsigned dot = 100, dash = 300, space = 150, word=600, tone=850;

/* Statistics and maintainence variables */
unsigned correct, errors, timestamp, Last, aptr;

/* Command line arguments */
char quiet = -1;

/* Form for setting parameters */
char *form1[] = {
	50<<8|11,
	"\x01\x00\x82Difficulty           :",
	"\x01\x01\x82Number of characters :",
	"\x01\x02\x82Starting character   :",
	"\x01\x03\x82Ending character     :",
	"\x01\x04\x85Dot  length      (ms):",
	"\x01\x05\x85Dash length      (ms):",
	"\x01\x06\x85Dot/Dash spacing (ms):",
	"\x01\x07\x85Character spacing(ms):",
	"\x01\x08\x85Tone frequency   (hz):",
	0 };

/* Video attributes */
unsigned *attr;
unsigned mattr[] = {
	WSAVE|WCOPEN|WBOX1|REVERSE,
	WSAVE|WCOPEN|WBOX2|NORMAL,
	WSAVE|WCOPEN|WBOX3|NORMAL };
unsigned cattr[] = {
	WSAVE|WCOPEN|WBOX1|0x17,
	WSAVE|WCOPEN|WBOX2|0x47,
	WSAVE|WCOPEN|WBOX3|0x67 };

char test[100];			/* Buffer for test characters */

struct WINDOW *chart;	/* Pointer to chart window */

extern unsigned RAND_SEED;

/* Long accumulators, used for averaging */
unsigned laverage[2], ltemp[2], ltemp1[2];

char hello[] = {
	"Morse Code Trainer.\nCopyright 1996-1997 Dave Dunfield (VE3DRD)\nAll rights reserved.\n"
	};

char help[] = { "\nUse: MORSE [/M /Q /?]\n" };

/*
 * Main program
 */
main(int argc, char *argv[])
{
	int c;
	unsigned i, j, k, l, m;
	unsigned char array[NMORSE];
	char *ptr;
	FILE *fp;

	printf(hello);
	wopen(0, 0, 1, 1, WSAVE);
	wclose();
	attr = (W_BASE == 0xB800) ? cattr : mattr;

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case '/M' :
			case '-M' :
				attr = mattr;
				break;
			case '/Q' :
			case '-Q' :
				quiet = 0;
				break;
			case '/?' :
			case '?' :
				abort(help);
			default:
				fprintf(stderr, "\nInvalid parameter: %s\n", argv[i]);
				return; } }

	RAND_SEED = get_ticks();

	chart = wopen(0, 0, 80, 20, attr[0]);
	wprintf(hello);
	wopen(0, 20, 80, 5, attr[1]);
	wprintf(help);

	if(fp = fopen("MORSE.CFG", "rb")) {
		fget(&diff, CFGSIZE, fp);
		fclose(fp); }

	set_options();

generate:
	w_clwin(chart);

	if(diff >= 2)
		show_chart();

	for(i=0; i < num; ++i) {
		switch(c = (start + random((end+1) - start))) {
			case 35 :	c = '0';	break;
			case 36	:	c = '.';	break;
			case 37 :	c = ',';	break;
			case 38 :	c = '?';	break;
			case 39 :	c = '/';	break;
			case 40 :	c = '-';	break;
			case 41 :	c = '!';	break;
			case 42 :	c = '@';	break;
			case 43 :	c = '#';	break;
			case 44 :	c = 'K';	break;
			case 45	:	c = '$';	break;
			default:
				if(c < 26)
					c += 'A';
				else
					c += ('1'-26); }
		test[i] = c; }
	test[i] = aptr = 0;

	j = (end+1) - start;
	for(i=0; i <= end; ++i)
		array[i] = start + (i%j);

	l = RAND_SEED;
	for(i=0; i <= end; ++i) {
		k = array[j = random(end + 1)];
		array[j] = array[i];
		array[i] = k; }
	RAND_SEED = l;

	timestamp = get_ticks();

replay:
	wclwin();
	l = ltemp1[0];
	wprintf("CODE: Correct=%-4u Errors=%-4u    TIME: Last=%u.%02u   Average=%u.%02u\n\n",
		correct, errors,
		Last / 18, ((Last % 18) * 100) / 18,
		l / 18, ((l % 18) * 100) / 18);
	play_string(test);
	w_gotoxy(37, 17, chart);
	w_puts("F1:Again F2:New F3:Setup F4:Save ESC:Exit", chart);
	i = 0;
	while(i < num) {
		switch(c = toupper(wgetc())) {
			case 0x1B :		// ESC - exit
				wclose();
				wclose();
				return;
			case _K3 :		// F3 - Set options
				w_clwin(chart);
				set_options();
			case _K2 :		// F2 - New word
				goto generate;
			case _K1 :		// F1 - replay
				goto replay;
			case _K4 :		// F4 - Save config
				wclwin();
				if(fp = fopen("MORSE.CFG", "wb")) {
					fput(&diff, CFGSIZE, fp);
					fclose(fp);
					wprintf("Setup saved!"); }
				else
					wprintf("Unable to open: MORSE.CFG");
				delay(1000);
				goto replay;
			default:
				if(c == test[i]) {
					wputc(c);
					++i;
					if(diff >= 2)
						show_chart();
					else
						w_clwin(chart);
					continue; }
				if(diff >= 3) {
					erase(m = translate(c));
					l = translate(test[i]);
					j = diff - 3;
					while(j && (aptr <= end)) {
						if((k = array[aptr++]) == l)
							continue;
						if(k == m)
							continue;
						erase(k);
						--j; } }
				if(diff == 1)
					show_chart();
				++errors;
				if(quiet)
					beep(100, 500);
				delay(1000);
				goto replay; } }

	Last = get_ticks() - timestamp;

	longset(ltemp, Last);
	longadd(laverage, ltemp);
	longset(ltemp, ++correct);
	longcpy(ltemp1, laverage);
	longdiv(ltemp1, ltemp);

	wprintf(" - Time=%u.%02u", Last/18, ((Last%18)*100) / 18);
	if(quiet) {
		beep(1000, 250);
		beep(1200, 250); }
	delay(1000);
	goto generate;
}

/*
 * Display the code chart
 */
show_chart()
{
	unsigned i;
	for(i=0; i <= end; ++i) {
		if(i >= start) {
			w_gotoxy(((i/18)*20)+1, i%18, chart);
			w_printf(chart, "%-8s= ", morse[i]);
			if(i < 26)
				w_putc(i+'A', chart);
			else
				w_puts(symbols[i-26], chart); } }
}

/*
 * Erase a code from the display
 */
erase(unsigned j)
{
	w_gotoxy(((j/18)*20)+1, j%18, chart);
	w_puts("          ", chart);
}

/*
 * Set program options
 */
set_options()
{
top:
	++start;
	++end;
	wform(14, 5, attr[2], form1,
		&diff, &num, &start, &end, &dot, &dash, &space, &word, &tone);
	--start;
	--end;
	

	wclwin();

	if((num > 99) || !num) {
		wprintf("\nInvalid number of characters (1 - 99)");
		goto top; }

	if((end >= NMORSE) || !end) {
		wprintf("\nInvalid ending character (1-45)");
		goto top; }

	if(start > end) {
		wprintf("\nInvalid starting character (1-%u)", end+1);
		goto top; }

	if((tone < 100) || (tone > 15000)) {
		wprintf("\nInvalid tone frequency (100-15000)");
		goto top; }
}

/*
 * Play a string of ASCII as MORSE CODE
 */
play_string(char *ptr)
{
	int c;

	for(;;) switch(c = *ptr++) {
		case 0 :
			return;
		case ' ' :
		case '\t' :
			delay(word);
			if(!quiet)
				wputc(' ');
			continue;
		default:
			play_char(translate(c));
			if(*ptr) {
				delay(word);
				if(!quiet)
					wputc(' ');
				continue; } }
}

/*
 * Translate a character into a morse code table index
 */
translate(char c)
{
	switch(c) {
		case '0' :	return 35;
		case '.' :	return 36;
		case ',' :	return 37;
		case '?' :	return 38;
		case '/' :	return 39;
		case '-' :	return 40;
		case '!' :	return 41;
		case '@' :	return 42;
		case '#' :	return 43;
		case '$' :	return 45; }
	if((c >= 'A') && (c <= 'Z'))
		return c - 'A';
	if((c >= '1') && (c <= '9'))
		return c - ('1' - 26);

	if(quiet)
		beep(100, 500);
	return NMORSE;
}

/*
 * Play a table index as a series of dit's and dah's
 */
play_char(unsigned c)
{
	char *ptr;

	ptr = morse[c];
top:
	switch(*ptr++) {
		case '.' :
			if(quiet)
				beep(tone, dot);
			else
				wputc('.');
			goto next;
		case '_' :
			if(quiet)
				beep(tone, dash);
			else
				wputc('_');
		next:
			if(*ptr) {
				delay(space);
				goto top; } }
}

/*
 * Get the BIOS clock tick count
 */
get_ticks() asm
{
	MOV	AX,40h
	MOV	BX,6Ch
	MOV	ES,AX
	MOV	AX,ES:[BX]
}
