DEFINT A-Z

' RP.BAS : Read PIC16C84 program memory in serial mode.
'
' Usage:
'
'       qbasic/run rp.bas > hex-file
'
' Reads all 1K locations of a 16C84 and displays contents as an Intel
' Hex file (INHX8M format).  The output should be redirected to a file
' as shown above.  The program assumes the hardware is attached to LPT1
' and that it uses inverting buffers (these assumptions can be changed).
' The program should be run once without a PIC inserted to initialise the
' parallel port.  This program is based on the C source for rp.c V-0.0.
'
'
' Author:
'
'   David Tait,
'   Electrical Engineering Dept,
'   The University,
'   Manchester, M13 9PL,
'   UK.
'
'   david.tait@man.ac.uk  (Via Microchip BBS: david tait)
'
' Copyright (C) 1995 David Tait.
' This program is free software.  Permission is granted to use,
' copy, or redistribute this program so long as it is not sold
' for profit.
'
' THIS PROGRAM IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
' EXPRESSED OR IMPLIED.
'

DECLARE SUB HexWord (Word%)
DECLARE SUB PrintHex (ProgSize%)
DECLARE SUB HexByte (Byte%)
DECLARE FUNCTION LptAddr% (LptNum%)
DECLARE SUB ReadPic ()
DECLARE SUB Command (Cmd%)
DECLARE SUB IdleMode ()
DECLARE SUB VppDly ()
DECLARE FUNCTION InWord% ()
DECLARE SUB ProgMode ()
DECLARE SUB Setup ()

' LPT port bit assignments

CONST DataIn = 64

' ONLY use the next three lines if H/W uses 74xx07

' CONST DataInv = 0
' CONST VppOn = 8, VppOff = 0, VddOn = 4, VddOff = 0
' CONST ClkHi = 2, ClkLo = 0, OutHi = 1, OutLo = 0

' ONLY use the next three lines if H/W uses 74xx06

CONST DataInv = DataIn
CONST VppOn = 0, VppOff = 8, VddOn = 0, VddOff = 4
CONST ClkHi = 0, ClkLo = 2, OutHi = 0, OutLo = 1

CONST p0000 = VppOff + VddOff + ClkLo + OutLo
CONST p0100 = VppOff + VddOn + ClkLo + OutLo
CONST p0101 = VppOff + VddOn + ClkLo + OutHi
CONST p1100 = VppOn + VddOn + ClkLo + OutLo
CONST p1101 = VppOn + VddOn + ClkLo + OutHi
CONST p1110 = VppOn + VddOn + ClkHi + OutLo
CONST p1111 = VppOn + VddOn + ClkHi + OutHi

' 16C84 serial programming commands

CONST RdProg = 4, IncAdd = 6

CONST ProgSize = 1024

' Global variables

DIM SHARED ProgBuf(0 TO ProgSize - 1)
DIM SHARED LptPort, NWords, Check

Main:
  NWords = 4
  LptPort = LptAddr(1)
  Setup
  ReadPic
  PrintHex (ProgSize)
  SYSTEM
  END

SUB Command (Cmd%)
   OUT LptPort, p1100
   BitPos = 1
   DO WHILE BitPos <= 32
     IF (Cmd AND BitPos) THEN
       OUT LptPort, p1111
       OUT LptPort, p1101
     ELSE
       OUT LptPort, p1110
       OUT LptPort, p1100
     END IF
     BitPos = BitPos * 2
   LOOP
   OUT LptPort, p1101
END SUB

SUB HexByte (Byte)
  PRINT HEX$(Byte \ 16); HEX$(Byte AND 15);
  Check = Check + Byte
END SUB

SUB HexWord (Word)
  HexByte (Word \ 256)
  HexByte (Word AND 255)
END SUB

SUB IdleMode
  OUT LptPort, p0100
  VppDly
  OUT LptPort, p0000
END SUB

FUNCTION InWord%
  OUT LptPort, p1111
  OUT LptPort, p1101
  Word = 0
  BitPos = 1
  DO WHILE BitPos <= 8192
    OUT LptPort, p1111
    OUT LptPort, p1101
    Bit = (INP(LptPort + 1) AND DataIn) XOR DataInv
    IF Bit THEN Word = Word + BitPos
    BitPos = BitPos * 2
  LOOP
  OUT LptPort, p1111
  OUT LptPort, p1101
  InWord% = Word
END FUNCTION

FUNCTION LptAddr% (LptNum%)
   IF (LptNum < 0) OR (LptNum > 3) THEN LptNum = 1
   TablePtr = &H408 + 2 * (LptNum - 1)
   DEF SEG = 0
   Addr = PEEK(TablePtr) + 256 * PEEK(TablePtr + 1)
   DEF SEG
   SELECT CASE Addr
     CASE &H3BC, &H278, &H378
     CASE ELSE
       PRINT "Illegal LPT Address": SYSTEM
   END SELECT
   LptAddr% = Addr
END FUNCTION

SUB OutWord (Word%)
  OUT LptPort, p1110
  OUT LptPort, p1100
  BitPos = 1
  DO WHILE BitPos <= 8192
    IF Word AND BitPos THEN
      OUT LptPort, p1111
      OUT LptPort, p1101
    ELSE
      OUT LptPort, p1110
      OUT LptPort, p1100
    END IF
    BitPos = BitPos * 2
  LOOP
  OUT LptPort, p1110
  OUT LptPort, p1100
END SUB

SUB PrintHex (n)
    Address = 0
    DO WHILE n > 0
        Check = 0
        PRINT ":";
        HexByte (2 * NWords)
        HexWord (2 * Address)
        HexByte (0)
        IF n > NWords THEN k = NWords ELSE k = n
        FOR i = 1 TO k
            w = ProgBuf(Address)
            Address = Address + 1
            HexByte (w AND 255)
            HexByte (w \ 256)
        NEXT i
        HexByte ((-Check) AND 255)
        PRINT
        n = n - NWords
    LOOP
    PRINT ":00000001FF"
END SUB

SUB ProgMode
  OUT LptPort, p0100
  VppDly
  OUT LptPort, p1100
END SUB

SUB ReadPic
  ProgMode
  FOR i = 0 TO ProgSize - 1
    Command RdProg
    ProgBuf(i) = InWord% AND &H3FFF
    Command IncAdd
  NEXT i
  IdleMode
END SUB

SUB Setup
  OUT LptPort, p0101
  VppDly
  BitHi = (INP(LptPort + 1) AND DataIn) XOR DataInv
  OUT LptPort, p0100
  BitLo = (INP(LptPort + 1) AND DataIn) XOR DataInv
  IF (BitHi <> DataIn) OR (BitLo <> 0) THEN PRINT "Hardware Failure": SYSTEM
  OUT LptPort, p0000
END SUB

'
'  About 100ms delay
'
SUB VppDly
  t! = TIMER
  tplus! = t! + .1
  DO UNTIL t! > tplus!
    t! = TIMER
  LOOP
END SUB

