/*
 * rp.c : Read PIC16C84 program memory using serial mode.
 *
 * Usage:
 *
 *       rp [ hex-file ]
 *
 * Reads all 1K locations of a 16C84 and displays contents as an Intel
 * Hex file (INHX8M format).  If hex-file is specified the output is
 * stored there.  The program assumes the hardware is attached to LPT1
 * and uses inverting buffers (these assumptions can be changed but a
 * a recompilation is required).  The parallel port is initialised by
 * running the program without a PIC inserted.
 *
 *
 *
 * Revision history:
 *
 * Version V-0.0: 11-Mar-95; For demonstration purposes only.
 *
 *
 * Author:
 *
 *    David Tait,
 *    Electrical Engineering Dept, 
 *    The University, 
 *    Manchester M13 9PL, 
 *    UK.     
 *
 *    david.tait@man.ac.uk  (Via Microchip BBS: david tait)
 *
 *
 * Copyright (C) 1995 David Tait.
 * This program is free software.  Permission is granted to use,
 * copy, or redistribute this program so long as it is not sold
 * for profit.
 *
 * THIS PROGRAM IS PROVIDED AS IS AND WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>

#define LPTNUM  1    /* change to agree with LPT port used */
/* #define U7407 */  /* define U7407 if H/W uses non-inverting buffers */

#define PSIZE   1024
#define NWORDS   4   /* words/line in hex output (MPASM uses 4) */

#define IN      64
#define VPP     8
#define VDD     4
#define CLK     2
#define OUT     1

#define RDPROG  4
#define INCADD  6

#define VPPDLY  100     /* milliseconds */
#define MAXDLY  5       /* milliseconds */

#define TSET    60      /* machine dependent: OK for 486DX33 at least */
#define THLD    60      /* can be 0 for 386DX20 but these are OK too  */
#define TDLY    200

#ifdef U7407
#define inbit           (inportb(s_reg)&IN)
#define vppon           (d_bits |= VPP)
#define vppoff          (d_bits &= ~VPP)
#define vddon           (d_bits |= VDD)
#define vddoff          (d_bits &= ~VDD)
#define clkhi           (d_bits |= CLK)
#define clklo           (d_bits &= ~CLK)
#define outhi           (d_bits |= OUT)
#define outlo           (d_bits &= ~OUT)
#else
#define inbit           (~inportb(s_reg)&IN)
#define vppon           (d_bits &= ~VPP)
#define vppoff          (d_bits |= VPP)
#define vddon           (d_bits &= ~VDD)
#define vddoff          (d_bits |= VDD)
#define clkhi           (d_bits &= ~CLK)
#define clklo           (d_bits |= CLK)
#define outhi           (d_bits &= ~OUT)
#define outlo           (d_bits |= OUT)
#endif /* U7407 */
#define assert          (outportb(d_reg,d_bits))


int progbuf[PSIZE];

int d_bits = 0;
int d_reg;
int s_reg;
int check;
int dummy;

void tiny_delay(int ticks)
{
   while ( ticks-- )
      ++dummy;             /* stops the loop being optimized to nil */
}

void idle_mode()
{
   vppoff, clklo, outlo, assert;
   delay(VPPDLY);
   vddoff, assert;
}

void prog_mode()
{
   vppoff, vddon, clklo, outlo, assert;
   delay(VPPDLY);
   vppon, assert;
   delay(VPPDLY);
   vppoff, assert;      /* reset PIC */
   delay(VPPDLY);
   vppon, assert;
}


void test_hw()
{
    int b;

    vddon, outhi, assert;      /* toggle RB7 to check hardware is present */
    delay(MAXDLY);
    b = inbit;
    outlo, assert;
    delay(MAXDLY);             /* OK for even the most sluggish hardware */

    if ( b != IN || inbit != 0 ) {
       fprintf(stderr,"rp: No hardware found at 0x%04X\n",d_reg);
       idle_mode();
       exit(1);
    }
}


void setup()
{
   vppoff, vddoff, clklo, outlo, assert;
   d_reg = peek(0,0x408+(LPTNUM-1)*2);  /* base address of LPT port */
   s_reg = d_reg+1;

   switch ( d_reg ) {  /* check port address is valid */
       case 0x3BC:
       case 0x378:
       case 0x278: break;
       default:  fprintf(stderr,"rp: LPT at 0x%04X!\n",d_reg);
		 idle_mode();
		 exit(1);
   }

   test_hw();
}


void clock_out(int bit)
{
   bit? outhi: outlo; clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   tiny_delay(THLD);
   outlo, assert;
}

int clock_in()
{
   int b;

   outhi, clkhi, assert;
   tiny_delay(TSET);
   clklo, assert;
   b = inbit? 1: 0;
   tiny_delay(THLD);
   return b;
}

void out_word(int w)
{
   int b;

   clock_out(0);
   for ( b=0; b<14; ++b )
     clock_out(w&(1<<b));
   clock_out(0);
}

int in_word()
{
   int b, w;

   (void) clock_in();
   for ( w=0, b=0; b<14; ++b )
     w += clock_in()<<b;
   (void) clock_in();
   return w;
}

void command(int cmd)
{
   int b;

   outlo, assert;
   tiny_delay(TDLY);
   for ( b=0; b<6; ++b )
      clock_out(cmd&(1<<b));
   outhi, assert;
   tiny_delay(TDLY);
}

void readpic()
{
   int i;

   prog_mode();
   for ( i=0; i<PSIZE; ++i ) {
      command(RDPROG);
      progbuf[i] = in_word() & 0x3FFF;
      command(INCADD);
   }
   idle_mode();
}


void hexbyte(int b)
{
    printf("%02X",b);
    check += b;
}

void hexword(int w)
{
    hexbyte((w>>8)&0xFF);
    hexbyte(w&0xFF);
}

void printhex(int n)
{
    int address = 0, i, w;

    while( n > 0 ) {
	check = 0;
	printf(":");
	hexbyte(2*NWORDS);
	hexword(2*address);
	hexbyte(0);
	for ( i=0; i < ((n>NWORDS)? NWORDS: n); ++i ) {
	    w = progbuf[address++];
	    hexbyte(w&0xFF);
	    hexbyte(w>>8);
	}
	hexbyte((-check)&0xFF);
	printf("\n");
	n -= NWORDS;
    }
    printf(":00000001FF\n");
}


void main(int argc, char **argv)
{
   if ( argc > 1 )
      if ( freopen(argv[1], "w", stdout) == NULL ) {
	fprintf(stderr,"rp: Can't open file\n");
	exit(1);
      }

   setup();
   readpic();
   printhex(PSIZE);
   idle_mode();
}


