;**********************************************************************;
;*                           K E Y B U F                              *;
;*--------------------------------------------------------------------*;
;*    Task           : Installs extended keyboard reading interrupt   *;
;*                     routines and implements a virtual keyboard     *:
;*                     buffer of up to 256 bytes (128 characters).    *;
;*                     An initial call installs the program, while a  *:
;*                     second call disables the program.              *;
;*--------------------------------------------------------------------*;
;*    Assembly       : MASM KEYBUF;                                   *;
;*                     LINK KEYBUF;                                   *;
;*                     EXE2BIN KEYBUF KEYBUF.COM                      *;
;*--------------------------------------------------------------------*;
;*    Call           : KEYBUF                                         *;
;**********************************************************************;
                                                                        
;== BIOS variable segment ==============================================
                                                                        
bios      segment at 40h          ;Segment begins at 0040:0000
                                                                        
          org 1ah
                                                                        
          ;-- BIOS pointer points to the keyboard ring buffer ----------
                                                                        
b_next    dw  (?)                 ;Pointer to next character
b_last    dw  (?)                 ;Pointer to last character
                                                                        
bios      ends
                                                                        
;== Constants ==========================================================
                                                                        
KB_LEN    equ 128                 ;Keyboard buffer length must be a
                                  ;power of 2 (change this constant to 
                                  ;change the size of the keyboard buffer
                                  ;e.g., 2, 4, 8, 16, 32, etc.)
                                                                        
;== Start of program ===================================================
                                                                        
code      segment para 'CODE'     ;Definition of CODE segment
                                                                        
          org 100h
                                                                        
          assume cs:code, ds:code, es:code, ss:code
                                                                        
start:    jmp kb_ini              ;First executable instruction
                                                                        
;== Data (stays in memory) ============================================
                                                                        
keybuf_id dw "CS"                 ;Identifies the program
                                                                        
env_seg   dw (?)                  ;Segment address of environment
                                                                        
int9      equ this dword          ;Old interrupt vector 09(h)
int9_ofs  dw (?)                  ;Offset address interrupt vector 09(h)
int9_seg  dw (?)                  ;Segment address interrupt vector 09(h)
                                                                        
int16     equ this dword          ;Old interrupt vector 16(h)
int16_ofs dw (?)                  ;Offset address interrupt vector 16(h)
int16_seg dw (?)                  ;Segment address interrupt vector 16(h)
                                                                        
;-- Virtual keyboard buffer is placed in the PSP of this program, ------
;-- making the program resident until a second  call  disables it ------
                                                                        
nextkey   dw 0                    ;Offset address of next key
curkey    dw KB_LEN - 2           ;Offset address of current key
                                                                        
;== New interrupt hander ===============================================
                                                                        
new_int9  proc far                ;New INT 9(h) handler
                                                                        
          assume ds:bios          ;Assign DS the BIOS variable segment
                                                                        
          pushf                   ;Simulate interrupt call to old INT
          call cs:int9            ;9(h) handler
                                                                        
          ;-- Get character from BIOS keyboard buffer ------------------
                                                                        
          cli
          push es                 ;Push all registers which will be 
          push ds                 ;changed by this new interrupt
          push di                 ;handler onto the stack
          push bx
          push ax
                                                                        
          mov  ax,bios            ;Get segment address of BIOS variable
          mov  ds,ax              ;segment to DS
          mov  di,cs:nextkey      ;Move DI to next character in KEYBUF
          mov  bx,b_next          ;BIOS: Get address of next character
                                                                        
ni9_0:    cmp  bx,b_last          ;Still a character in BIOS kbd buffer?
          je   ni9_end            ;No more characters --> END
                                                                        
          ;-- Still a character in the BIOS keyboard buffer --
                                                                        
          mov  ax,[bx]            ;Get character from BIOS kbd. buffer
          add  bx,2               ;Set pointer to next character
          cmp  bx,3eh             ;Reached end of keyboard buffer?
          jne  ni9_1              ;NO --> NI9_1
                                                                        
          mov  bx,1eh             ;YES --> Set start of kbd. buffer
                                                                        
ni9_1:    cmp  di,cs:curkey       ;Virtual keyboard buffer full yet?
          je   ni9_0              ;YES --> Don't store any more chars
          mov  cs:[di],ax         ;Characters in virtual kbd. buffer
          add  di,2               ;Set pointer to next character
          and  di,KB_LEN-1        ;Wrap-around
          jmp  ni9_0              ;Get next character
                                                                        
ni9_end:  mov  cs:nextkey,di      ;Mark position for next character
          mov  b_next,bx          ;Set BIOS pointer to next character
          pop  ax                 ;Pop registers off of stack
          pop  bx
          pop  di
          pop  ds
          pop  es
                                                                        
          iret                    ;Return to interrupt caller
                                                                        
          assume ds:code          ;DS indicates code segment
new_int9  endp
                                                                        
;-- New handler for BIOS keyboard interrupt 16(h) ----------------------
                                                                        
new_int16 proc far                ;New interrupt 16(h) handler
                                                                        
          sti                     ;Enable interrupt
          cmp  ah,1               ;Read keyboard status?
          ja   status             ;YES --> Status
                                                                        
          ;-- Update keyboard LEDs when function 1 of the old keyboard -
          ;-- handler is called
                                                                        
          push  ax                ;Push function code on the stack
                                                                        
          pushf                   ;Push flags onto stack
          mov  ah,1               ;Funct.no.: Key ready?
          call cs:[int16]         ;Call old handler
                                                                        
          pop  ax                 ;Pop function code off of stack
          push bx                 ;Push BX onto stack
                                                                        
ni16_0:   mov  bx,cs:curkey       ;Get pointer to current key
          add  bx,2               ;Set to next word
          and  bx,KB_LEN-1        ;Wrap-around
          or   ah,ah              ;Read character?
          je   ni16_2             ;YES --> Get character from buffer
                                                                        
          ;-- Function 1: Help caller determine whether a character is -
          ;-- available                                                -

          cmp  bx,cs:nextkey      ;Found a character in KEYBUF?
          je   ni16_1             ;NO --> NI16_1
                                                                        
          mov  ax,cs:[bx]         ;YES, Get character from KEYBUF
ni16_1:   pop  bx                 ;Pop BX off of stack
          ret  2                  ;Return to caller but DO NOT remove
                                  ;flags from stack
                                                                        
          ;-- Function 0: Read character from the keyboard buffer ------
                                                                        
ni16_2:   cmp  bx,cs:nextkey      ;Character found in KEYBUF?
          je   ni16_0             ;NO --> NI16_0
                                                                        
          mov  ax,cs:[bx]         ;YES -- > Get character from KEYBUF
          mov  cs:curkey,bx       ;Store position for current character
          pop  bx                 ;Pop BX off of stack
          iret                    ;Return to caller
                                                                        
status:   jmp  cs:[int16]         ;Jump to old handler
                                                                        
new_int16 endp
                                                                        
;-----------------------------------------------------------------------
                                                                        
instend   equ this byte           ;Everything must remain resident up
                                  ;to this memory location
                                                                        
;== Data (cann be overwritten from DOS) ================================
                                                                        
installm  db 13,10,"--- KEYBUF (c) 1988 by Michael Tischer ---",13,10,13,10
          db "KEYBUF now enabled. Entering KEYBUF a second time",13,10
          db "from the  DOS prompt disables the KEYBUF program.",13,10,"$"
                                                                        
removeit  db 13,10,"--- KEYBUF (c) 1988 by Michael Tischer ---",13,10
          db "KEYBUF program now disabled.",13,10,"$"
                                                                        
;== Program (can be overwritten from DOS) ==============================
                                                                        
;-- Start and nitialization routine ------------------------------------
                                                                        
kb_ini    label near
                                                                        
          mov  ax,3509h           ;Get contents of interrupt vector 9(h)
          int  21h                ;Cal DOS function
          cmp  es:keybuf_id,"CS"  ;Program already installed?
          jne  install            ;NO --> Install
                                                                        
          ;-- If KEYBUF is already installed, remove it ----------------
                                                                        
          cli                     ;Disable interrupts
          lds  dx,es:int9         ;DS:DS = old handler address int9(h)
          mov  ax,2509h           ;Return interrupt vector for int 9(h)
          int  21h                ;to old interrupt handler
                                                                        
          lds  dx,es:int16        ;DS:DS = Old handler address int16(h)
          mov  ax,2516h           ;Return interrupt vector 16(h) to old
          int  21h                ;interrupt handler
          sti                     ;Enable interrupt
                                                                        
          mov  bx,es              ;Move segment address of program
          mov  es,es:env_seg      ;Get segment address of environment
          mov  ah,49h             ;from code segment and
          int  21h                ;release memory
                                                                        
          mov  es,bx              ;Release memory of
          mov  ah,49h             ;old KEYBUF using
          int  21h                ;DOS interrupt 49(h)
                                                                        
          push cs                 ;Push CS onto stack
          pop  ds                 ;Pop DS off of stack
                                                                        
          mov  dx,offset removeit ;Message: Program disabled
          mov  ah,9               ;Write function number for string
          int  21h                ;Call DOS function
                                                                        
          mov  ax,4C00h           ;Funct. no.: End program
          int  21h                ;Call DOS interrupt --> END
                                                                        
          ;-- Install KEYBUF -------------------------------------------
                                                                        
install   label near
                                                                        
          ;-- In order  to configure  new keyboard buffer  within the --
          ;-- PSP, the segment address must first be moved to the end --
          ;-- of the PSP, where it cannot be overwritten              --
                                                                        
          mov  ax,[2Ch]           ;Load segment address of environment
          mov  env_seg,ax         ;and place in code segment
                                                                        
          mov  ax,3509h           ;Get contents of interrupt vector 9(h)
          int  21h                ;Call DOS function
          mov  int9_seg,es        ;Mark segment and offset address of
          mov  int9_ofs,bx        ;interrupt vector 9(h)
                                                                        
          mov  ax,3516h           ;get contents of interrupt vector 16(h)
          int  21h                ;Call DOS function
          mov  int16_seg,es       ;Mark segment and offset address of
          mov  int16_ofs,bx       ;interrupt vector 16(h)
                                                                        
          cli                     ;Disable interrupt
          mov  ax,2509h           ;Funct. no.: Set interrupt vector 9(h)
          mov  dx,offset new_int9 ;Offset addr. of new int. 9(h) handler
          int  21h                ;Call DOS interrupt
                                                                        
          mov  ax,2516h           ;Funct. no.: Set interrupt vector 16(h)
          mov  dx,offset new_int16;Offset addr. of new int. 16(h) handler
          int  21h                ;Call DOS interrupt
          sti                     ;Enable interrupts
                                                                        
          mov  dx,offset installm ;Message: Install program
          mov  ah,9               ;Function number for string display
          int  21h                ;Call DOS function
                                                                        
          ;-- Just PSP, new interrupt routine and corresponding --------
          ;-- data must be resident                             --------
                                                                        
          mov  dx,offset instend  ;Get offset address of last byte 
          add  dx,15              ;Make paragraph "full"
          mov  cl,4               ;Compute number of resident
          shr  dx,cl              ;paragraphs
          mov  ax,3100h           ;Terminate but keep resident program
          int  21h                ;with end code of (0)
                                                                        
;== Ende ===============================================================
                                                                        
code      ends                    ;End of code segment
          end  start
                                                                        
                                                                        

