;**********************************************************************;
;*                             T S R P A                              *;
;*--------------------------------------------------------------------*;
;*    Description    : This is the assembler interface to a Turbo     *;
;*                     Pascal 4.0 program which can be activated      *;
;*                     via a hotkey.                                  *;
;*--------------------------------------------------------------------*;
;*    Info           : The module must be in a program and may not    *;
;*                     be bound into a UNIT.                          *;
;*--------------------------------------------------------------------*;
;*    to assemble    : MASM TSRPA;                                    *;
;*                     ... combine with a Turbo Pascal program        *;
;**********************************************************************;


DATA   segment word public        ;Turbo data segment

DATA       ends                   ;end of the data segment

;== Constants ==========================================================

MAX_ID_LEN equ 16                 ;maximum length of the ID string

;== Program ============================================================

CODE       segment byte public    ;the Turbo code segment

           assume cs:CODE, ds:DATA, es:CODE

;-- Public declarations of internal functions --------------------------

public     tsrinit                ;allows access by the Turbo program
public     isinst
public     uninst

;-- Variables for the interrupt handler --------------------------------
;-- (accessible only via the code segment ------------------------------

id_buf     db (MAX_ID_LEN + 1) dup (0)  ;buffer for the ID string
ce_ptr     equ this dword         ;points to the routine CALL_END in the
ce_ofs     dw offset call_end     ;already-installed TSR program
ce_seg     dw ?

;-- Variables neded for activation of the Turbo program ---------------

t_ss       dw 0                   ;Turbo stack segment
t_sp       dw 0                   ;Turbo stack pointer
t_ds       dw 0                   ;Turbo data segment
t_es       dw 0                   ;Turbo extra segment

t_dta_ofs  dw 0                   ;DTA address of the Turbo program
t_dta_seg  dw 0

t_psp      dw 0                   ;seg addr of the PSP of the Turbo prg.
prc_adr    dw 0                   ;address of the Turbo TSR procedure

;-- Variables for testing for the hotkey ------------------------------

key_mask   dw 0                   ;hotkey mask for BIOS keyboard flag
recur      db 0                   ;prevents recursive TSR calls
in_bios    db 0                   ;shows activity of the BIOS disk
                                  ;interrupt

daptr      equ this dword         ;pointer to the DOS INDOS flag
daptr_ofs  dw 0                   ;offset address
daptr_seg  dw 0                   ;segment address

;-- The following variables store the old addresses of the interrupt ---
;-- handlers which will be replaced by new interrupt handlers        ---

int9_ptr   equ this dword         ;old interrupt vector 9h
int9_ofs   dw 0                   ;offset address of the old handler
int9_seg   dw 0                   ;segment address of the old handler

int13_ptr  equ this dword         ;old interrupt vector 13h
int13_ofs  dw 0                   ;offset address of the old handler
int13_seg  dw 0                   ;segment address of the old handler

int28_ptr  equ this dword         ;old interrupt handler 28h
int28_ofs  dw 0                   ;offset address of the old handler
int28_seg  dw 0                   ;segment address of the old handler

;-- Variables for storing information about the interrupted ------------
;-- program                                                 ------------

u_dta_ofs  dw 0                   ;DTA address of interrupted program
u_dta_seg  dw 0

u_psp      dw 0                   ;seg addr of the PSP of the int. prg.

uprg_ss    dw 0                   ;SS and SP of the interrupted prg.
uprg_sp    dw 0

;-----------------------------------------------------------------------
;-- TSRINIT: ends the Turbo program and activates the new interrupt ----
;--          handler
;-- Call from Turbo: procedure TsrInit( PrzPtr   : word;
;--                                     KeyMask  : word;
;--                                     ResPara  : word;
;--                                     IdString : string[16] );

tsrinit    proc    near

sframe0    struc                  ;structure for accessing the stack
bp0        dw ?                   ;stores BP
ret_adr0   dw ?                   ;return address
idptr0     dd ?                   ;pointer to the ID string
respara0   dw ?                   ;number of paragraphs to be reserved
keymask0   dw ?                   ;mask for hotkey
prcptr0    dw ?                   ;pointer to the Turbo TSR procedure
sframe0    ends                   ;end of the structure

frame      equ [ bp - bp0 ]

           push bp                ;save BP on the stack
           mov  bp,sp             ;move SP to BP
           push es                ;save ES on the stack
           ;-- save the Turbo segment registers -----------------------

           mov  cs:t_ss,ss        ;save the registers in the appropriate
           mov  cs:t_sp,sp        ;variables
           mov  cs:t_es,es
           mov  cs:t_ds,ds

           ;-- copy the ID string into the internal buffer -------------

           push ds                 ;save DS on the stack
           lds  si,frame.idptr0    ;DS:SI now points to the string
           push cs                 ;put CS on the stack
           pop  es                 ;and restore as ES
           mov  di,offset id_buf   ;ES:DI now points to ID_BUF
           xor  ch,ch              ;clear high byte of the counter
           mov  cl,[si]            ;get length of the string
           inc  cl                 ;copy the length byte too
           rep  movsb              ;copy the entire string
           pop  ds                 ;restore DS

           ;-- determine PSP of the Turbo program ----------------------

           mov  bx,cs              ;transfer CS to BX
           sub  bx,10h             ;10h paragraphs = subtract 256 bytes
           mov  cs:t_psp,bx        ;save segment address

           ;-- save the parameters passed ------------------------------

           mov  ax,frame.prcptr0  ;get pointer to the TSR procedure
           mov  cs:prc_adr,ax     ;and save
           mov  ax,frame.keymask0 ;get mask for the hotkey
           mov  cs:key_mask,ax    ;and save

           ;-- determine DTA address of the Turbo program --------------

           mov  ah,2fh            ;ftn. no.: get DTA address
           int  21h               ;call DOS interrupt
           mov  cs:t_dta_ofs,bx   ;store address in the appropriate
           mov  cs:t_dta_seg,es   ;variables

           ;-- determine the address of the INDOS flag -----------------

           mov  ah,34h            ;ftn. no.: get adr of the INDOS flag
           int  21h               ;call DOS interrupt
           mov  cs:daptr_ofs,bx   ;save address in the appropriate 
           mov  cs:daptr_seg,es   ;variables

           ;-- get the addresses of the interrupt handlers to change ---

           mov  ax,3509h          ;get interrupt vector 9h
           int  21h               ;call DOS interrupt
           mov  cs:int9_ofs,bx    ;save address of the handler in the
           mov  cs:int9_seg,es    ;appropriate variables

           mov  ax,3513h          ;get interrupt vector 13h
           int  21h               ;call DOS interrupt
           mov  cs:int13_ofs,bx   ;save address of the handler in the 
           mov  cs:int13_seg,es   ;appropriate variables

           mov  ax,3528h          ;get interrupt vector 28h
           int  21h               ;call DOS interrupt
           mov  cs:int28_ofs,bx   ;save addres of the handler in the
           mov  cs:int28_seg,es   ;appropriate variables

           ;-- install the new interupt handlers -----------------------

           push ds                ;save data segment
           mov  ax,cs             ;CS to AX and then load into DS
           mov  ds,ax

           mov  ax,2509h          ;ftn. no.: set interrupt 9h
           mov  dx,offset int09   ;DS:DX stores the addr of the handler
           int  21h               ;call DOS interrupt

           mov  ax,2513h          ;ftn. no.: set interrupt 13h
           mov  dx,offset int13   ;DS:DX stores the addr of the handler
           int  21h               ;call DOS interrupt

           mov  ax,2528h          ;ftn. no.: set interrupt 28h
           mov  dx,offset int28   ;DS:DX stores the addr of the handler
           int  21h               ;call DOS interrupt

           pop  ds                ;get DS back from the stack

           ;-- End resident program ------------------------------------

           mov  ax,3100h          ;ftn. no.: end resident program
           mov  dx,frame.respara0 ;get number of reserved paragraphs
           int  21h               ;call DOS interrupt and thus end
                                  ;the program

tsrinit  endp

;-----------------------------------------------------------------------
;-- ISINST: Determines if the program is already installed -------------
;-- Call from Turbo: function IsInst( IdString : IdsType ) : boolean;
;-- Return value: 1, if the program was already installed,
;--               else 0

isinst     proc    near

sframe1    struc                  ;structure for accessing the stack
bp1        dw ?                   ;stores BP
ret_adr1   dw ?                   ;return address
idptr1     dd ?                   ;pointer to the ID string
sframe1    ends                   ;end of the structure

frame      equ [ bp - bp1 ]

           push bp                ;save BP on the stack
           mov  bp,sp             ;transfer Sp to BP
           push ds                ;save DS on the stack

           ;-- determine segment address of the current int 9 handler --

           mov  ax,3509h          ;get interrupt vbector 9h
           int  21h               ;DOS interrupt gets seg addr in ES
           mov  di,offset id_buf  ;ES:DI points to the installed ID_BUF
           lds  si,frame.idptr1   ;DS:SI points to the ID_STRING passed

           xor  dl,dl             ;return code: not installed
           mov  cl,[si]           ;get length of the string
           mov  ch,dl             ;high byte of the counter to 0
isi0:      lodsb                  ;load character from string
           cmp  al,es:[di]        ;compare with other string
           jne  not_inst          ;not equal --> NOT_INST
           inc  di                ;increment pointer to string 2
           loop isi0              ;compare the next characters

           mov  dl,1              ;the strings are identical

not_inst:  mov  al,dl             ;put return code in AL
           pop  ds                ;get DS back from stack
           pop  bp                ;get BP back from stack
           ret  4                 ;back to the caller

isinst     endp                   ;end of the procedure

;-----------------------------------------------------------------------
;-- CALL_END: calls the end function when the TSR is reinstalled -------
;-- Input  : DI = offset address of the routine to be called
;-- Info   : This function is not intended to be called by a Turbo
;--          program

call_end   proc far

           call di                ;call the end function
           ret                    ;back to the caller

call_end   endp

;-----------------------------------------------------------------------
;-- UNINST: removes the TSR program and releases the allocated ---------
;--         memory.
;-- Call from Turbo : procedure UnInst( EndPtr : word ); external;
;-- Info        :      If the value $FFFF is passed as the address,
;--                    then no end function will be called.
;-- Note        : This function should be called only if a previous
;--               call to IS_INST() returned a value of 1.

uninst     proc    near

sframe2    struc                  ;structure for accessing the stack
bp2        dw ?                   ;stores BP
ret_adr2   dw ?                   ;return address
prcptr2    dw ?                   ;pointer to the end procedure
sframe2    ends                   ;end of the structure

frame      equ [ bp - bp2 ]


           push bp                ;save BP on the stack
           mov  bp,sp             ;transfer SP to BP
           push ds                ;save DS on the stack

           ;-- determine seg addr of the current int 9h handler ---
           mov  ax,3509h          ;get interrupt vector 9h
           int  21h               ;DOS interrupt puts seg addr in ES

           mov  di,frame.prcptr2  ;get address of the end procedure
           cmp  di,0ffffh         ;no end procedure called?
           je   no_endprc         ;NO ---> NO_ENDPRC

           ;-- Perform context change to the Turbo program and ------
           ;-- execute the specified end procedure

           mov  cs:ce_seg,es      ;save ES in the jump vector

           mov  cs:uprg_ss,ss     ;save current stack segment and stack
           mov  cs:uprg_sp,sp     ;pointer

           cli                    ;disable interrupts
           mov  ss,es:t_ss        ;activate the stack of the TSR
           mov  sp,es:t_sp        ;program

           push es                ;save ES on the stack
           mov  ah,2fh            ;ftn. no.: get DTA address
           int  21h               ;call DOS interrupt
           mov  cs:u_dta_ofs,bx   ;save DTA address of the interrupted
           mov  cs:u_dta_seg,es   ;program
           pop  es                ;get ES from the stack

           mov  ah,50h            ;ftn. no.: set address of the PSP
           mov  bx,es:t_psp       ;get segment address of the PSP
           int  21h               ;call DOS interrupt

           push ds                ;save ES and DS on the stack
           push es

           mov  ah,1ah            ;ftn. no.: set DTA address
           mov  dx,es:t_dta_ofs   ;get offset address and segment
           mov  ds,es:t_dta_seg   ;address of the new DTA
           int  21h               ;call DOS interrupt

           mov  ds,es:t_ds        ;set segment register for the Turbo
           mov  es,es:t_es        ;program

           call cs:[ce_ptr]       ;call the end procedure

           ;-- context change to the Turbo program --------------------

           mov  ah,1ah            ;ftn. no.: set DTA address
           mov  dx,cs:u_dta_ofs   ;load offset and segment addresses
           mov  ds,cs:u_dta_seg   ;of the DTA of the interrupted program
           int  21h               ;call DOS interrupt

           pop  es                ;restore seg addr of the Turbo program
           pop  ds                ;from the stack

           mov  ah,50h            ;ftn. no.: set address of the PSP
           mov  bx,cs             ;put CS in BX
           sub  bx,10h            ;calculate segment address of the PSP
           int  21h               ;call DOS interrupt

           cli                    ;disable interrupts
           mov  ss,cs:uprg_ss     ;restore stack pointer and stack
           mov  sp,cs:uprg_sp     ;segment
           sti                    ;allow interrupts again

           ;-- reinstall the interrupt handler of the TSR --------------
           ;-- program again                              --------------

no_endprc: cli                    ;disable interrupts
           mov  ax,2509h          ;ftn. no.: set handler for int 9
           mov  ds,es:int9_seg    ;segment address of the old handler
           mov  dx,es:int9_ofs    ;offset address of the old handler
           int  21h               ;reinstall the old handler

           mov  ax,2513h          ;ftn. no.: set handler for int 13
           mov  ds,es:int13_seg   ;segment address of the old handler
           mov  dx,es:int13_ofs   ;offset address of the old handler
           int  21h               ;reinstall the old handler

           mov  ax,2528h          ;ftn. no. set handler for int 28
           mov  ds,es:int28_seg   ;segment address of the old handler
           mov  dx,es:int28_ofs   ;offset address of the old handler
           int  21h               ;reinstall the old handler

           sti                    ;allow interrupts again

           mov  es,es:t_psp       ;save seg addr of the PSP of the
           mov  cx,es             ;Turbo program in CX
           mov  es,es:[ 02ch ]    ;get seg addr of environ from PSP
           mov  ah,49h            ;ftn. no.: release allocated memory
           int  21h               ;call DOS interrupt
           mov  es,cx             ;restore ES from CX
           mov  ah,49h            ;ftn. no.: release allocated memory
           int  21h               ;call DOS interrupt

           pop  ds                ;restore DS and BP from stack
           pop  bp
           ret  2                 ;return to the caller

uninst     endp                   ;end of the procedure

;-----------------------------------------------------------------------
;-- The new interrupt handlers follow ----------------------------------
;-----------------------------------------------------------------------

;-- the new interrupt 09h handler --------------------------------------

int09      proc far

           pushf                  ;simulate calling the handler via the
           call cs:int9_ptr       ;INT 9h instruction

           cli                    ;suppress interrupts
           cmp  cs:recur,0        ;is the TSR program already active?
           jne  ik_end            ;Yes, back to the caller of int 9

           ;-- test to see if the BIOS disk int is being executed

           cmp  cs:in_bios,0      ;BIOS disk interrupt active?
           jne  ik_end            ;YES --> abck to caller

           ;-- BIOS disk interrupt is not active, test for hotkey -----

           push ax                ;save ES and AX on the stack
           push es
           xor  ax,ax             ;set ES to the lowest memory segment
           mov  es,ax             
           mov  ax,word ptr es:[417h] ;get BIOS keyboard flag
           and  ax,cs:key_mask    ;mask out the non-hotkey bits
           cmp  ax,cs:key_mask    ;are only the hotkey bits left?
           pop  es                ;restore ES and AX
           pop  ax
           jne  ik_end            ;hotkey discovered? NO --> return

           ;-- the hotkey was pressed, test to see if DOS is active ---

           push ds                ;save DS and BX on the stack
           push bx
           lds  bx,cs:daptr       ;DS:BX now point to the INDOS flag
           cmp  byte ptr [bx],0   ;DOS function active?
           pop  bx                ;get BX and DS from the stack
           pop  ds
           jne  ik_end            ;DOS function active --> IK_END

           ;-- DOS is not active, activate TSR program ----------------


           call  start_tsr        ;start the TSR program
ik_end:    iret                   ;back to the interrupted program

int09      endp

;-- the new interrupt 13h handler -------------------------------------

int13      proc far

           mov  cs:in_bios,1      ;set flag and show that the BIOS disk
                                  ;interrupt is active
           pushf                  ;simulate calling the old interrupt
           call cs:int13_ptr      ;handler via int 13h
           mov  cs:in_bios, 0     ;BIOS disk interrupt no longer active

           ret  2                 ;back to the caller, but don't get
                                  ;the flag reg from the stack first
int13      endp

;-- the new interrupt 28h handler -------------------------------------

int28      proc far

           pushf                  ;simulate calling the old interrupt
           call cs:int28_ptr      ;handler via int 28h

           cli                    ;suppress further interrupts
           cmp  cs:recur,0        ;is the TSR program already active?
           je   id01              ;NO ---> ID01
id_end:    iret                   ;YES   ---> back to the caller

           ;-- the TSR program is not yet active ---------------------

id01:      cmp  cs:in_bios, 0     ;is BIOS disk interrupt active?
           jne  id_end            ;YES --> back to the caller

           ;-- BIOS disk interrupt not active, test for hotkey -------

           push ax                ;save ES and AX on the stack
           push es
           xor  ax,ax             ;set ES to the lowest memory segment
           mov  es,ax             
           mov  ax,word ptr es:[417h] ;get BIOS keyboard flag
           and  ax,cs:key_mask    ;mask out the non-hotkey bits
           cmp  ax,cs:key_mask    ;are only the hotkey bits left?
           pop  es                ;restore ES and AX
           pop  ax
           jne  ik_end            ;hotkey discovered? NO --> return

           call  start_tsr        ;start the TSR program
           iret                   ;back to the interrupted program

int28      endp


;-- START_TSR: activate the TSR program -------------------------------
start_tsr  proc near

           mov  cs:recur,1        ;set the TSR recursion flag

           ;-- perform context change to the TSR program --------------

           mov  cs:uprg_ss,ss     ;save current stack segment and
           mov  cs:uprg_sp,sp     ;stack pointer

           mov  ss,cs:t_ss        ;activate the stack of the 
           mov  sp,cs:t_sp        ;Turbo program

           push ax                ;save the processor registers on the
           push bx                ;turbo stack
           push cx
           push dx
           push bp
           push si
           push di
           push ds
           push es

           ;-- save 64 words from the DOS stack -----------------------

           mov  cx,64             ;loop counter
           mov  ds,cs:uprg_ss     ;set DS:SI to the end of the DOS stack
           mov  si,cs:uprg_sp

tsrs1:     push word ptr [si]     ;save word from the DOS stack on the
           inc  si                ;C stack and set SI to the next word
           inc  si
           loop tsrs1             ;process all 64 words

           mov  ah,51h            ;ftn. no.: get addr of the PSP
           int  21h               ;call DOS interrupt
           mov  cs:u_psp,bx       ;save seg addr of the PSP

           mov  ah,2fh            ;ftn. no.: get DTA address
           int  21h               ;call DOS interrupt
           mov  cs:u_dta_ofs,bx   ;save address of the DTA of the 
           mov  cs:u_dta_seg,es   ;interrupted program

           mov  ah,50h            ;ftn. no.: set address of the PSP
           mov  bx,cs:t_psp       ;get seg addr of the Turbo prg PSP
           int  21h               ;call DOS interrupt

           mov  ah,1ah            ;ftn. no.: set DTA address
           mov  dx,cs:t_dta_ofs   ;get offset address of the new DTA
           mov  ds,cs:t_dta_seg   ;and segment address of the new DTA
           int  21h               ;call DOS interrupt

           mov  ds,cs:t_ds        ;set segment register for the
           mov  es,cs:t_es        ;Turbo program

           sti                    ;allow interrupts again

           call cs:prc_adr        ;call the start function
           cli                    ;disable interrupts

           ;-- perform context change to the interrupted program ------

           mov  ah,1ah            ;ftn. no.: set DTA address
           mov  dx,cs:u_dta_ofs   ;load offset and segment addresses
           mov  ds,cs:u_dta_seg   ;of the interrupted program's DTA
           int  21h               ;call DOS interrupt

           mov  ah,50h            ;ftn. no.: set address of the PSP
           mov  bx,cs:u_psp       ;seg addr of the interrupted prg's PSP
           int  21h               ;call DOS interrupt

           ;-- restore DOS stack again ---------------------------------

           mov  cx,64             ;loop counter
           mov  ds,cs:uprg_ss     ;load DS:SI with the end address of
           mov  si,cs:uprg_sp     ;the DOS stack
           add  si,128            ;set SI to the start of the DOS stack
tsrs2:     dec  si                ;Si to the previous stack word
           dec  si
           pop  word ptr [si]     ;words from Turbo stack to DOS stack
           loop tsrs2             ;process all 64 words

           pop  es                ;restore the saved registers from the
           pop  ds                ;Turbo stack
           pop  di
           pop  si
           pop  bp
           pop  dx
           pop  cx
           pop  bx
           pop  ax

           mov  ss,cs:uprg_ss     ;set stack pointer and segment
           mov  sp,cs:uprg_sp     ;of the interrupted program

           mov  cs:recur,0        ;resset TSR recursion flag
           ret                    ;back to the caller

start_tsr  endp

;---------------------------------------------------------

CODE       ends                   ;end of the code segment
           end                    ;end of the program

