;**********************************************************************;
;*                             T Y P M C A                            *;
;*--------------------------------------------------------------------*;
;*    Description    : Assembler routine for setting the key repeat   *;
;*                     rate on an AT keyboard. For linking with a     *;
;*                     C program.                                     *;
;*--------------------------------------------------------------------*;
;*    to assembler   : MASM TYPMATIC;                                 *;
;*                     ... link with a C program                      *;
;**********************************************************************;

;== Constants ==========================================================

KB_STATUS_P   equ 64h             ;keyboard status port
KB_DATA_P     equ 60h             ;keyboard data port

OB_FULL       equ 1               ;bit 0 in keyboard status port
                                  ;a character in the output buffer
IB_FULL       equ 2               ;bit 1 in the keyboard status port
                                  ;a character in the input buffer

ACK_SIGNAL    equ 0fah            ;keyboard acknowledge signal
SET_TYPEM     equ 0f3h            ;set-repeat-rate code

MAX_TRY       equ 3               ;number of retries allowed

;== Segment declarations for the C program =============================

IGROUP group _text                ;combination of the program segments
DGROUP group const,_bss,  _data   ;combination of the data segments
       assume CS:IGROUP, DS:DGROUP, ES:DGROUP, SS:DGROUP

CONST  segment word public 'CONST';this segment stores all of the 
CONST  ends                       ;read-only constants

_BSS   segment word public 'BSS'  ;this segment stores all of the 
_BSS   ends                       ;uninitialized static variables

_DATA  segment word public 'DATA' ;all initialized global and static
                                  ;variables are stored in this segment
_DATA  ends

;== Program ============================================================

_TEXT  segment byte public 'CODE' ;the program segment

public     _set_typm
;-----------------------------------------------------------------------
;-- SET_TYPM: sends the key repeat rate to the keyboard controller -----
;-- Call from C : bool set_typem( byte trate );
;-- Return value: TRUE, if the repeat rate was set
;--               FALSE, if an error occurred

_set_typm  proc near

sframe0    struc                  ;structure for accessing the stack
bp0        dw ?                   ;stores BP
ret_adr0   dw ?                   ;return address to caller
trate0     dw ?                   ;repeat rate to be set
sframe0    ends                   ;end of the structure

frame      equ [ bp - bp0 ]       ;addresses the elements of the structure

           push bp                ;save BP on the stack
           mov  bp,sp             ;transfer SP to BP

           xor  dx,dx             ;assume transfer fails
           mov  ah,SET_TYPEM      ;set command code for rep rate
           cli                    ;disable interrupts
           call send_kb           ;send to the controller
           jne  error             ;error? YES --> Error

           mov  ah,byte ptr frame.trate0 ;get key repeat rate
           call send_kb           ;send to the controller
           jne  error             ;error? YES --> Error

           inc  dl                ;everything OK, return TRUE

error:     sti                    ;allow interrupts again
           mov  ax,dx             ;return value to AX
           pop  bp                ;get BP back from stack 
           ret                    ;back to the C program

_set_typm  endp

;-----------------------------------------------------------------------
;-- SEND_KB: send a byte to the keyboard controller --------------------
;-- Input    : AH = the byte to be sent
;-- Output   : zero flag: 0=error, 1=OK
;-- Registers: AX and the flag register are changed
;-- Info     : This routine is to be called only within the module

send_kb   proc near

          push cx                 ;save all registers which are changed
          push bx                 ;in this routine on the stack

          mov  bl,MAX_TRY         ;maximum of MAX_TRY retries

          ;-- wait until the controller is ready to receive data ------

skb_1:    xor  cx,cx              ;maximum of 65536 loop passes
skb_2:    in   al,KB_STATUS_P     ;read contents of status port
          test al,IB_FULL         ;still a char in the input buffer?
          loopne skb_2            ;YES --> SKB_2

          ;-- send character to the controller -------------------------

          mov  al,ah              ;get character in AL
          out  KB_DATA_P,al       ;send character to the data port
skb_3:    in   al,KB_STATUS_P     ;read contents of the status port
          test al,OB_FULL         ;reply in output buffer?
          loope skb_3             ;NO --> SKB_3

          ;-- get and evaluate relpy from controller -------------------

          in   al,KB_DATA_P       ;read reply from data port
          cmp  al,ACK_SIGNAL      ;was the character accepted?
          je   skb_end            ;YES --> everything OK

          ;-- the character was not accepted ---------------------------

          dec  bl                 ;decrement error counter
          jne  skb_2              ;still retries left?
                                  ;YES --> SKB_2

          or   bl,1               ;NO, set zero flag to 0 to indicate
                                  ;the error

skb_end:  pop  bx                 ;restore the registers from the stack
          pop  cx                 
          ret                     ;return to caller

send_kb   endp

;-------------------------------------------------------------------------

_text      ends                   ;end of the code segment
           end                    ;end of the program
