;----------------------------------------------------------------
; This code demonstrates a working protect mode interrupt.
; The timer interrupt (int 8 IRQ 0) writes a char to the screen
; each time it fires off
; -Mark  73150.730@compuserve.com
;----------------------------------------------------------------
.386p
	
desc struc		;structure for a descriptor
		dw	0
        dw	0
        db	0
        db	0
        db	0
        db	0
desc	ends

prot_enable equ 01h	; protected mode enable bit in CR0
space		equ	20h	; ASCII code for space
attribute	equ	07h	; display character attribute

code_sel	equ 08h
video_sel	equ 10h
data_sel	equ 18h
seg32_sel	equ 20h
;----------------------------------------------------------------
; Global Descriptor Table
;----------------------------------------------------------------
gdt_seg segment para public use16 'gdt_seg'
gdt_tab		label qword
	desc	<,,,,,>							; first one must be null selector
    desc    <0ffffh,,,098h,,>   			; code
	desc	<0ffffh,8000h,0bh,092h,40h,>	; video
	desc	<gdata_limit,,,092h,40h,>		; 32 bit data (40h=32bit)
    desc    <0ffffh,,,09ah,40h,>   			; 32 bit code
gdt_limit	equ $-gdt_tab	
gdt_seg ends

;----------------------------------------------------------------
; Data segment
;----------------------------------------------------------------
Gdata segment para public use16 'Gdata'

gdt_ptr	label fword  
gdt_ptr_limit	dw ?	; gdt_seg limit
gdt_ptr_addr	dd ?	; gdt_seg base address

idt_ptr	label fword	 
idt_ptr_limit 	dw ?	; IDT limit
idt_ptr_addr	dd ?	; IDT base address

prot_msg		db 'in protected mode, dude ',0
screen_pos		dw 320
gstack			db 100h dup(0)	;some stack space
Gdata_limit		equ $
Gdata 			ends

;----------------------------------------------------------------
; Interrupt Descriptor Table  8E=Interrupt  8F=Trap
;----------------------------------------------------------------
int_seg		segment	para public use32 'int_seg'
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
	desc <,seg32_sel,0,08eh,,>	 
;	desc <,seg32_sel,0,08eh,,>	 
idt_limit	equ $;-int_seg
int_seg ends


;----------------------------------------------------------------
; Starting code segment
;----------------------------------------------------------------
code 	segment para public use16 'code'
		assume  cs:code, ds:gdata
		
mainb	proc far
        mov ax,gdata			; get data segment address
        mov ds,ax				; put in DS
		
		call store_gdt_address	;store the phys address of the tables
		call store_idt_address
		
		mov ax,seg32			;fix the 32bit segment gdt entry
		mov bx,0
		mov cx,seg32_sel
		call store_gdt_des
		
		mov ax,gdata			;fix the data segment gdt entry
		mov bx,0
		mov cx,data_sel
		call store_gdt_des
		
		mov ax,code				;fix the 16bit segment entry
		mov bx,0
		mov cx,code_sel
		call store_gdt_des
		
		call fix_idt
        
;----------------------------------------------------------------
; Here we go into protect mode
;----------------------------------------------------------------
		cli						; clear interrupt
		lgdt [gdt_ptr]			; load gdt_seg addr and limit into gdt_segR	
		lidt [idt_ptr]
		
		mov eax,cr0					; get cr0 register
		or al,prot_enable			; set protected mode enable
		mov cr0,eax					; restore cr0
		jmp dword ptr cs:[enter_prot]; far jump to flush instruction queue
enter_prot:
		dw offset now_in_prot		; EIP
		dw code_sel					; code segment selector

now_in_prot:
		xor ax,ax					; clear ax
		lldt ax						; load NULL selector to LDTR
        
		mov ax,data_sel				; get data segment selector
		mov ds,ax					; put in DS
		mov es,ax					; put in ES
        mov fs,ax					; put in FS
        mov gs,ax					; put in GS
        
;----------------------------------------------------------------
; Display the protect mode message
;----------------------------------------------------------------
        mov ax,video_sel			; get video segment selector
		mov es,ax					; put in ES
        mov cx,4000h                ; buffer size to clear
		xor di,di					; screen starting address
		mov ah,attribute			; character attribute
		mov al,space				; space
		rep stosw					; fill it
		mov si, offset prot_msg 	; get protected mode message address
		mov di,320
        call show_string            ; call display procedure
        
        ;----This is 16 bit jump
        ;db 0eah
        ;dw 0
        ;dw seg32_sel
        ;----------------------------------------
        
        mov ax,data_sel
        mov ss,ax
        mov eax,offset gdata_limit
        mov esp,eax
        
        ;----This is 32 bit jump
        db 66h	;32 bit operand prefix
        db 67h	;32 bit address prefix
        db 0eah
        dd 0
        dw seg32_sel
        ;-----------------------------------------
   		;Now executing at seg32     
   		
;****************************************************************
; End of main code - start of subroutines
;****************************************************************
		
;----------------------------------------------------------------
; fix_idt - put the isr code lin address into idt descriptors
;----------------------------------------------------------------
fix_idt:
		mov ax,int_seg			;load idt segment
		mov es,ax
		mov ax,0				;start at beginning of idt
		mov di,ax
		
								;calculate the no of interrupt descriptors
		mov ax,offset idt_limit	;get length of idt	
		shr ax,3				;divide by 8
		mov cx,ax				;put into cx for use with loop 
		
fi0:
		mov ax,int_seg			;load idt segment
		mov es,ax
		mov ax,0
		mov bx,offset int_code
		mov dx,0eeh
		mov dx,0eeh
		mov dx,0eeh
		mov dx,0eeh
		call phys_to_lin
		call store_idt_addr
		add di,8
		dec cx
		or cx,cx
		jnz fi0
		
		retn	
		
		
;------------------------------------------------------------------------
; store_idt_addr IN:ax:bl=lin addr (lsw:msb) es=idt segment di=idt offset
;------------------------------------------------------------------------
store_idt_addr:
	mov es:[di],ax		;store bottom two bytes of routine offset
	xor ax,ax			;clear ax
	mov al,bl			;move rest of offset into ax
	mov es:[di+6],ax	;store top two bytes
	retn
		
;------------------------------------------------------------------------
; phys_to_lin   IN:ax=segment  bx=offset OUT: ax:bl= 32bit linear (lsw:msb)
;------------------------------------------------------------------------
phys_to_lin:
		push cx
		push ax
		shl ax,4
		add ax,bx
		mov cx,ax
		pop ax 
		shr ax,12
		xor bx,bx
		mov bl,al
		mov ax,cx
		pop cx
		retn
		
		
		
;----------------------------------------------------------------
; store_gdt_address
;----------------------------------------------------------------
store_gdt_address:
		mov	ax,offset gdt_limit		; get gdt_seg segment limit
        mov	gdt_ptr_limit,ax			; put in gdt_ptr_limit
        xor	eax,eax					; clear eax
        mov ax,gdt_seg					; get gdt_seg segment address
        shl eax,4					; convert to 32 bit linear address
        mov gdt_ptr_addr,eax			; put in gdt_ptr address
        retn
		
;----------------------------------------------------------------
; store_gdt_des:
	;cx = descriptor
	;ax = seg
	;bx = ofs
;----------------------------------------------------------------
store_gdt_des:
		push ax
		mov ax,gdt_seg		;mov gdt seg into es
		mov es,ax
		
		mov ax,cx
		and ax,0fff8h	;we're only interested in mult of 8
		mov di,ax		;this is offset into gdt_seg
		
		pop ax
		call phys_to_lin
		
		
		add di,2			;move past bytes 0 and 1 of the descriptor (limit)
		mov es:[di],ax		;write the linear address
		add di,2
		mov al,bl
		mov es:[di],al
		retn

;----------------------------------------------------------------
; store_idt_address
;----------------------------------------------------------------
store_idt_address:
		mov	ax,offset idt_limit		; get gdt_seg segment limit
        mov	idt_ptr_limit,ax			; put in gdt_ptr_limit
        xor	eax,eax					; clear eax
        mov ax,int_seg				; get gdt_seg segment address
        shl eax,4					; convert to 32 bit linear address
        mov idt_ptr_addr,eax			; put in gdt_ptr address
        retn
mainb endp

;----------------------------------------------------------------
; Procedure: show_string   - Display string in protected mode
; Input: ds:si = string address   end of string must be a zero
;----------------------------------------------------------------
show_string		proc near
		mov ax,video_sel			; get video segment selector
		mov es,ax					; put in ES
		mov ah,attribute			; display attribute
show_string1:
		lodsb						; get display character
		stosw						; put it on screen
		cmp al,0					; end of display character ?
		jne	show_string1				; no, continue
		ret							; yes, return
show_string		endp


code    ends
;end mainb


;----------------------------------------------------------------
; 32 bit code segment
;----------------------------------------------------------------
seg32 	segment para public use32 'seg32'
		assume  cs:seg32, ds:gdata
sss		proc near
		
seg32_begin:
		mov eax, offset gdata_limit
		mov esp,eax
		mov ax,video_sel
		mov es,ax
		mov di,320
		mov byte ptr es:[di],'*'
		sti							;Let the interrupts begin
		
sss0:								;Loop here and let timer interrupt us
		jmp sss0
sss		endp


;----------------------------------------------------------------
; Interrupt will land here
;----------------------------------------------------------------
int_code: 					
		;nop
		;nop
		;nop
		;nop
		cli
		;add esp,12
		mov ax,video_sel
		mov es,ax
		mov di,screen_pos
		mov al,'0'
		mov byte ptr es:[di],al
		add di,2
		mov screen_pos,di
		call end_of_int
		sti
		iretd
		
;------------------------------------------------------------		
; end_of_int - signal the 8259 PIC that its over
;------------------------------------------------------------		
end_of_int:
		push ax
		push dx
		mov dx,20h
		mov al,20h
		out dx,al
		pop dx
		pop ax
		ret
				
		
		
seg32 ends

end mainb


