/*****************************************************************************
 *                                  B J
 *****************************************************************************
 *
 *   PROGRAM ID:        BJ.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      Here are several alternatives method of calculating the Bessel fxns
 *      Jp(x).
 *
 *      The first is from JCP Miller, pick a large, arbitrary starting
 *      point and recurse downwards.  It should be good for all (p,z).
 *      It uses the little known Neumann sum for normalization:
 *
 *        (x/2)^p = SUMMATION{ gamma(p+i)/gamma(1+i) * (p+2*i) * J[p+2*i](x) }
 *                   i=0..N
 *
 *
 *      The second is from the following article: E Chalbaud & P Martin,
 *      J Math Phys 33(7), p.2483 [July 92].  It's an approximation based
 *      on matched power series with the asymptotic series.  The absolute
 *      error is less than 0.005 for all z (!) for |p|<= 1,  and just 0.001
 *      for |p|<= 0.8.  Now here's the idea: Use downward recursion into the
 *      interval [-0.8 < p < 0.2] and calculate this approximation to normalize
 *      your starting value.  It is only good for real (p,z) values.
 *
 *      The third is the rational polynomial method of Hart. It is only valid for
 *      Bessel functions of integer order (whereas the other two routines are valid
 *      for all values of p). If you only need bessel functions of integer order,
 *      it is much faster to use the built-in bessel functions available in most C
 *      compiler math libraries.
 *
 *      Finally, I should mention that Numerical Recipes 2nd edition has a very
 *      good routine based on continued fractions (ie  Steeds method) that
 *      simultaneously calculates J, Y, J', & Y' for a given (p,x).
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *                      When using MS C compilers, the Bessel fxns Jn(x) & Yn(x)
 *                      are built-in to the math library:
 *                           _j0, _j1, _jn, _y0, _y1, _yn
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>
#define nint(a)  ( ((a)-floor(a)<0.5)?floor(a):ceil(a) )





/*
// Here's the first idea for Jp(x) from JCP Miller.......
*/
double bj( double p, double z )
{
// This algorithm holds for COMPLEX p,z  as well.  But you need to
// replace FLOAT declarations with COMPLEX, use a COMPLEX gamma fxn,
// and come up with some tests to detect "near integer" values of p
// for complex p.

double A[3], t, b, c, F, y;
double big= 1e10, tiny= 1e-10; // tiny= 1/big
int i, j, k, np;

  // Negative integer orders make the gamma fxn blow up. So fix that here:
  np= nint(p);
  if( fabs(p-np) < tiny  && np<0 )
  {
      np= -np;
      t= bj( np, z );
      if( np%2 )
          t= -t;
      return( t );
  }


  // Must start downward recursion from a high enough value...
  //k= p +sqrt( 40.0*p );       // For non-integers, need to take z
  k= max(fabs(p),fabs(z)) + 15; // into account.
  k= 2*(k/2) +1;                // ....rounded to nearest odd integer


  y= 2.0 / z;
  b= gamma(p+1.) * pow( y, p );
  c= 1;

  A[(k+2)%3]= 0.;
  A[(k+1)%3]= 1e-50;
  for( F=0, i=k ; i>0 ; i-- )
  {
      t= A[i%3]= y*(p+i)*A[(i+1)%3] -A[(i+2)%3];

      // check for overflow
      if( fabs(A[i%3]) > big )
      {
          F          *= tiny;
          A[i%3]     *= tiny;
          A[(i+1)%3] *= tiny;
          A[(i+2)%3] *= tiny;
      }

      // accumulate odd terms
      if( i>2 && i%2 == 1 )
      {
          j= i/2;
          F += c*t;
          // c /= (p+2*i)/(p+2*i-2) * (p+i-1)/i;
          c *= (p+2*j-2)/(p+2*j) * j/(p+j-1);
      }
  }
  // account for scale of c-factors, the last should be  b
  F *= b/c;
  // add the last term by hand
  F += b*A[1];

  // return A[1] scaled by F
  return( A[1] / F );
}



/*
// Here's another idea for Jp(x). It from the following article:
//    E Chalbaud & P Martin, J Math Phys 33(7), p.2483 [July 92]
//
// This function is still under construction....it still needs:
//    - a check for overflow during recursion
//    - a good starting point for recursion
//    - get rid of the checks at the top of the function (these
//       checks WILL be needed in a complex version)
*/

/*
double bj( double p, double z )
{
     double *A, c, F;
     int i, k, nz, np;
     double az, ap;

     nz= fabs(z); np= fabs(p);

     az= real(z); ap= real(p);

      rz= abs(1-az/z);           //real z gives rz <1e-7
      rp= abs(1-ap/p)
      rip= abs(1- nint(ap)/ap ); //integer p gives rip <1e-7

        if( (rz+rp+rip  <1.0e-7) && (ap >=0) )
            BJ= bjn( nint(ap), az);
        else
        {
            k= 1 + ceil( p );
            A= (double *)malloc( (k+1)*sizeof(double) );

            A[k]= 0;
            A[k-1]= 1.0e-37

            for( i=k-2 ; i>=0 ; i-- )
                 A[i]= 2*(i+1)*A[i+1]/z -A[i+2];

            {
                double Cz, Sz, zz;
                double l, a, m, Ca, Sa, G2;
                double P1, P0, Q1, Q0, p1, p0, q1, q0;
                double Jp;

                Cz= cos(z);  Sz= sin(z);
                zz= z*z;

                l= 0.0083 +p*(0.73 +p*( 1.46 +p*1.16));
                a= M_PI_4 *(1 +p +p);  Sa= sin(a); Ca= cos(a);
                m= 4*p*p;
                G2= pow( 2, -p ) /gamma( p+1 );

                P1=  l *M_1_SQRTPI *M_SQRT2 *Ca;
                p1=  l *M_1_SQRTPI *M_SQRT2 *Sa;
                Q0= -l *M_1_SQRTPI *M_SQRT2 *Ca *(m-1)/8.0;
                q0=  l *M_1_SQRTPI *M_SQRT2 *Sa *(m-1)/8.0;

                p0= 1.2 *G2 *(l+a/M_PI) +
                    0.6*( G2*(p+p+1)/(p+p+2) -2*(P1+p1+Q0) -q0 );

                P0= G2 - q0 -p0;

                Jp= pow( 1+zz, -p/2-0.25 ) *pow( z, p ) /(1 +l*zz);
                Jp *= (P0+P1*zz)*Cz +(p0+p1*zz)*sqrt(1+zz)*Sz/z
                       +Q0*z*Sz +q0*sqrt(1+zz)*Cz;

                F= Jp/A[0];
            }
            F= F * A[k-1];
        }
        return( F );
}
*/


/*
// The remaining routines are based on rational polynomials and
// only accept integer values of the order.
*/
double bjn( int n, double x )
{
int j, jsum, m;
double sum, rc, xabs, y;
double b0, b1, b;
double big= 1e10, tiny= 1e-10; // tiny= 1 / big


  if( n < 0 )
  {
      n= -n;
      rc= bjn( n, x );
      if( n%2 )
          rc= -rc;
  }
  else if( n==0 )
      rc= bj0( x );
  else if( n==1 )
      rc= bj1( x );
  else
  {
      xabs= fabs( x );
      if( xabs < tiny )
         rc= 0;
      else if( xabs > (float)n )
      {
            y= 2.0 / xabs;
            b0= bj0( xabs );
            b1= bj1( xabs );
            for( j=1 ; j<n ; j++ )
            {
                 b= j*y*b1 -b0;
                 b0= b1;
                 b1= b;
            }
            rc= b1;
      }
      else
      {
          // use Miller's algorithm
          rc= bj( n, x );
      }

      if( x<0  &&  n%2 )
          rc= -rc;
  }

  return( rc );
}




double bj1( double x )
{
double rc, xabs, xx, z;
double y,p1,p2,p3,p4,p5,q1,q2,q3,q4,q5,r1,r2,r3,r4,r5,r6;
double s1,s2,s3,s4,s5,s6;

  r1=72362614232.e0, r2=-7895059235.e0, r3=242396853.1e0,
    r4=-2972611.439e0, r5=15704.48260e0, r6=-30.16036606e0;

  s1=144725228442.e0, s2=2300535178.e0, s3=18583304.74e0,
    s4=99447.43394e0, s5=376.9991397e0, s6=1.0;

  p1=1.e0, p2=0.183105e-2, p3=-0.3516396496e-4, p4=0.2457520174e-5,
    p5=-0.240337019e-6;

  q1=0.04687499995e0, q2=-0.2002690873e-3, q3=0.8449199096e-5,
    q4=-0.88228987e-6, q5=0.105787412e-6;

  if( (xabs= fabs(x)) < 8.0 )
  {
      y= x*x;
      rc= x*(r1+y*(r2+y*(r3+y*(r4+y*(r5+y*r6)))))/
                  (s1+y*(s2+y*(s3+y*(s4+y*(s5+y*s6)))));
  }
  else
  {
      z= 8.0 / xabs;
      y= z*z;
      xx= xabs - 2.356194491;
      rc= sqrt(0.636619772/xabs)*(cos(xx)*(p1+y*(p2+y*(p3+y*(p4+y*p5
             ))))-z*sin(xx)*(q1+y*(q2+y*(q3+y*(q4+y*q5)))));
      if( x<0 )
          rc= -rc;
  }

  return( rc );

}





double bj0( double x )
{
double  rc, xabs, xx, z;
double  y,p1,p2,p3,p4,p5,q1,q2,q3,q4,q5,r1,r2,r3,r4,r5,r6;
double  s1,s2,s3,s4,s5,s6;

  p1=1.0, p2=-0.1098628627e-2, p3=0.2734510407e-4,
    p4=-.2073370639e-5, p5=0.2093887211e-6;

  q1=-0.1562499995e-1, q2=0.1430488765e-3, q3=-0.6911147651e-5,
    q4=0.7621095161e-6,q5=-0.934945152e-7;

  r1=57568490574.e0, r2=-13362590354.e0, r3=651619640.7e0,
    r4=-11214424.18e0, r5=77392.33017e0, r6=-184.9052456e0;

  s1=57568490411.e0, s2=1029532985.e0, s3=9494680.718e0,
    s4=59272.64853e0, s5=267.8532712e0, s6=1.0;


  if( (xabs= fabs(x)) < 8.0 )
  {
      y= x*x;
      rc= (r1+y*(r2+y*(r3+y*(r4+y*(r5+y*r6)))))
             /(s1+y*(s2+y*(s3+y*(s4+y*(s5+y*s6)))));
  }
  else
  {
      z = 8.0 / xabs;
      y = z*z;
      xx= xabs - 0.785398164;
      rc= sqrt(0.636619772/xabs)*(cos(xx)*(p1+y*(p2+y*(p3+y*(p4+
           y*p5))))-z*sin(xx)*(q1+y*(q2+y*(q3+y*(q4+y*q5)))));
  }

  return( rc );
}

