/*****************************************************************************
 *                                  B J Y
 *****************************************************************************
 *
 *   PROGRAM ID:        BJY.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      These routines are from a series of articles by two of the
 *      authors of Numerical Recipes - WH Press & SA Teukolsky -
 *      which appeared in Computers in Physics, March thru June 1991.
 *
 *      The theory behind them is Steed's method for calculating the
 *      Coulomb wave functions. J, J', Y, & Y' values are generated
 *      simultaneously.
 *
 *      The utility of these methods are the routines which calculate the
 *      modified/variations of the plain old bessel functions.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *   SPECIAL NOTES:     This program is placed in the public domain and can be
 *                      used freely by anyone.
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

#if defined( DOUBLE_PRECISION )
    static const double eps= 1.0e-16;
    static const double fpmin= 1.0e-100;
#else
    static const double eps= 1.0e-10;
    static const double fpmin= 1.0e-30;
#endif


/* const double pi= 3.141592653589793; */
static const double pi= M_PI;
static const int maxit= 10000;
static const double xmin= 2.0;


static double chebev( double a, double b,
                      double *c, int m, double x )
{
int j;
double d, dd, y, y2, sv;

  if( (x-a)*(x-b) > 0 )
      nrerror( "x not in range in chebev" );

  d= dd= 0;
  y= (x + x -a -b ) / (b-a);
  y2= y + y;

  for( j=m-1 ; j>=1 ; j-- )
  {
      sv= d;
      d= y2*d - dd +c[j];
      dd= sv;
  }

  return( y*d - dd + c[0]/2 );
}


static void cheb( double x,
                  double *gam1, double *gam2,
                  double *gampl, double *gammi )
{
const int nuse1= 7;
const int nuse2= 8;

double xx;
static double c1[]= { -1.142022680371172, 6.516511267076e-3,
                       3.08709017308e-4, -3.470626964e-6,
                       6.943764e-9, 3.678e-11, -1.36e-13 };

static double c2[]= { 1.843740587300906, -0.076852840844786,
                      1.271927136655e-3, -4.971736704e-6,
                     -3.312612e-8, 2.4231e-10, -1.7e-13, -1.0e-15 };


  xx= 8.0 *x*x -1;
  gam1[0]= chebev( -1, 1, c1, nuse1, xx );
  gam2[0]= chebev( -1, 1, c2, nuse2, xx );

  gampl[0]= (*gam2) - x*(*gam1);
  gammi[0]= (*gam2) + x*(*gam1);
}





void bjy( double x, double xnu,
          double *rj, double *ry,
          double *rjp, double *ryp )
{
double  a, b, br, bi, c , cr, ci, d, del, del1, den, di,
        dlr, dli, dr, e, f, fact, fact2, fact3, ff, gam,
        gam1, gam2, gammi, gampl, h, p, pimu, pimu2, q,
        r, rjl, rjl1, rjmu, rjp1, rjpl, rjtemp, ry1, rymu,
        rymup, rytemp, sum, sum1, temp, w, x2, xi, xi2, xmu, xmu2;

int isign, nl, i, l, bDone;


  if( x<0.0 || xnu<0.0 )
      nrerror( "Bad arguments in bessjy" );

  if( x<xmin )
      nl= (int)(xnu+0.5);
  else
  {
      nl= (int)(xnu-x+1.5);
      if( nl<0 )    nl= 0;
  }

  xmu= xnu - nl;
  xmu2= xmu * xmu;
  xi= 1/x;
  xi2= xi + xi;
  w= xi2 / pi;
  isign= 1;
  h= xnu * xi;
  if( h<fpmin )   h= fpmin;
  b= xi2 * xnu;
  d= 0;
  c= h;

  for( i=1, bDone=0 ; !bDone && i<=maxit ; i++ )
  {
      b += xi2;
      d = b-d;
      if( fabs(d)<fpmin )   d= fpmin;
      c= b - 1/c;
      if( fabs(c)<fpmin )   c= fpmin;
      d= 1/d;
      del= c*d;
      h *= del;
      if( d<0 )   isign= -isign;
      bDone= fabs(del-1) < eps;
  }
  if( i>maxit )
      nrerror( "x too large in bessjy, try asmptotic expansion" );


  rjl= isign * fpmin;
  rjpl= h * rjl;
  rjl1= rjl;
  rjp1= rjpl;
  fact= xnu * xi;

  for( l=nl ; l>=1 ; l-- )
  {
      rjtemp= fact * rjl + rjpl;
      fact -= xi;
      rjpl= fact * rjtemp - rjl;
      rjl= rjtemp;
  }

  if( rjl == 0 )   rjl= eps;
  f= rjpl / rjl;

  if( x<xmin )
  {
      x2= x/2;
      pimu= pi*xmu;

      if( fabs(pimu)<eps )
          fact= 1;
      else
          fact= pimu / sin( pimu );

      d= -log( x2 );
      e= xmu * d;
      if( fabs(e)<eps )
          fact2= 1;
      else
          fact2= sinh( e ) /e;

      cheb( xmu, &gam1, &gam2, &gampl, &gammi );

      ff= 2/pi * fact * (gam1*cosh(e) + gam2*fact2*d);
      e= exp( e );
      p= e / (gampl*pi);
      q= 1 / (e*pi*gammi);
      pimu2= pimu / 2;

      if( fabs( pimu2 )<eps )
          fact3= 1;
      else
          fact3= sin( pimu2 ) /pimu2;

      r= pi * pimu2 * fact3 * fact3;
      c= 1;
      d= -x2 * x2;
      sum= ff + r*q;
      sum1= p;

      for( i=1, bDone=0 ; !bDone && i<=maxit ; i++ )
      {
          ff= (i*ff+p+q) / (i*i-xmu2);
          c *= d/i;
          p /= i-xmu;
          q /= i+xmu;
          del= c*(ff+r*q);
          sum += del;
          del1= c*p - i*del;
          sum1 += del1;
          bDone= fabs(del) < (1+fabs(sum))*eps;
      }
      if( i>maxit )
          nrerror( "bessy series failed to converge" );

      rymu= -sum;
      ry1= -sum1 * xi2;
      rymup= xmu * xi * rymu - ry1;
      rjmu= w / (rymup-f*rymu);
  }
  else
  {
      a= 0.25 - xmu2;
      p= -xi/2;
      q= 1.0;
      br= x + x;
      bi= 2;
      fact= a * xi / (p*p+q*q);
      cr= br + q*fact;
      ci= bi + p*fact;
      den= br*br + bi*bi;
      dr= br/den;
      di= -bi/den;
      dlr= cr*dr - ci*di;
      dli= cr*di + ci*dr;
      temp= p*dlr - q*dli;
      q= p*dli + q*dlr;
      p= temp;

      for( i=2, bDone=0 ; !bDone && i<=maxit ; i++ )
      {
          a += 2*(i-1);
          bi += 2;
          dr= a*dr + br;
          di= a*di + bi;
          if( fabs(dr)+fabs(di) < fpmin )   dr= fpmin;
          fact= a / (cr*cr+ci*ci);
          cr= br + cr*fact;
          ci= bi - ci*fact;
          if( fabs(cr)+fabs(ci) < fpmin )   cr= fpmin;
          den= dr*dr + di*di;
          dr /= den;
          di /= -den;
          dlr= cr*dr - ci*di;
          dli= cr*di + ci*dr;
          temp= p*dlr - q*dli;
          q= p*dli + q*dlr;
          p= temp;
          bDone= fabs(dlr-1)+fabs(dli) < eps;
      }
      if( i>maxit )
          nrerror( "df2 failed in bessjy" );

          gam= (p-f)/q;
          rjmu= sqrt( w / ((p-f)*gam+q) );
          if( rjl<0 )   rjmu= -rjmu;
          rymu= rjmu * gam;
          rymup= rymu * (p + q/gam);
          ry1= xmu*xi*rymu - rymup;
  }

  fact= rjmu/rjl;
  rj[0]= rjl1*fact;
  rjp[0]= rjp1*fact;

  for( i=1 ; i<=nl ; i++ )
  {
      rytemp= (xmu+i) * xi2 * ry1 - rymu;
      rymu= ry1;
      ry1= rytemp;
  }

  ry[0]= rymu;
  ryp[0]= xnu * xi * rymu - ry1;
}




void sphbes( int n, double x, double *sj, double *sy, double *sjp, double *syp )
{
const double rtpid2= M_1_SQRT2/M_1_SQRTPI; // sqrt(pi/2)
double factor, order, rj, rjp, ry, ryp;

  if( n<0 || x<=0 )
      nrerror( "bad arguments in sphbes" );

  order= n + 0.5;
  bjy( x, order, &rj, &ry, &rjp, &ryp );
  factor= rtpid2 / sqrt(x);
  sj[0]= factor * rj;
  sy[0]= factor * ry;
  sjp[0]= factor * rjp - sj[0]/(x+x);
  syp[0]= factor * ryp - sy[0]/(x+x);
}




void bik( double x, double xnu,
          double *ri, double *rk,
          double *rip, double *rkp )
{
int i, l, nl, bDone;
double a, a1, b, c, d, del, del1, delh, dels, e, f, fact, fact2,
       ff, gam1, gam2, gammi, gampl, h, p, pimu, q, q1, q2, qnew,
       ril, ril1, rimu, rip1, ripl, ritemp, rk1, rkmu, rkmup,
       rktemp, s, sum, sum1, x2, xi, xi2, xmu, xmu2;


  if( x<0 || xnu<0 )
      nrerror( "bad arguments in bik" );

  nl= (int)(xnu + 0.5);
  xmu= xnu - nl;
  xmu2= xmu * xmu;
  xi= 1/x;
  xi2= xi + xi;
  h= xnu * xi;
  if( h<fpmin )   h= fpmin;
  b= xi2 * xnu;
  d= 0;
  c= h;

  for( i=1, bDone=0 ; !bDone && i<=maxit ; i++ )
  {
      b += xi2;
      d= 1 / (b+d);
      c= b + 1/c;
      del= c*d;
      h *= del;
      bDone= fabs(del-1) < eps;
  }
  if( i>maxit )
      nrerror( "x too large in bik; try asymptotic expansion" );

  ril= fpmin;
  ripl= h*ril;
  rip1= ripl;
  fact= xnu * xi;

  for( l=nl ; i>=1 ; l-- )
  {
      ritemp= fact * ril + ripl;
      fact -= xi;
      ripl= fact * ritemp + ril;
      ril= ritemp;
  }

  f= ripl / ril;
  if( x<xmin )
  {
      x2= x/2;
      pimu= pi * xmu;
      if( fabs(pimu)<eps )
          fact= 1;
      else
          fact= pimu / sin( pimu );
      d= -log( x2 );
      e= xmu * d;
      if( fabs(e)<eps )
          fact2= 1;
      else
          fact2= sinh( e ) /e;

      cheb( xmu, &gam1, &gam2, &gampl, &gammi );

      ff= fact * (gam1*cosh(e) + gam2*fact2*d);
      sum= ff;
      e= exp( e );
      p= 0.5 * e / gampl;
      q= 0.5 / (e*gammi);
      c= 1;
      d= x2*x2;
      sum1= p;
      for( i=1, bDone=0 ; !bDone && i<=maxit ; i++ )
      {
          ff= (i*ff+p+q) / (i*i-xmu2);
          c *= d/i;
          p /= i - xmu;
          q /= i + xmu;
          del= c*ff;
          sum += del;
          del1= c * (p-i*ff);
          sum1 += del1;
          bDone= fabs(del) < fabs(sum)*eps;
      }
      if( i>maxit )
          nrerror( "bessk series failed to converge" );
      rkmu= sum;
      rk1= sum1 * xi2;
  }
  else
  {
      b= 2 * (x+1);
      d= 1/b;
      h= delh= d;
      q1= 0;
      q2= 1;
      a1= 0.25 - xmu2;
      q= c= a1;
      a= -a1;
      s= 1 + q*delh;
      for( i=2, bDone=0 ; !bDone && i<=maxit ; i++ )
      {
          a -= 2*(i-1);
          c *= -a/i;
          qnew= (q1 - b*q2) / a;
          q1= q2;
          q2= qnew;
          q += c*qnew;
          b += 2;
          d= 1 / (b + a*d);
          delh= (b*d - 1) * delh;
          h += delh;
          dels= q*delh;
          s += dels;
          bDone= fabs(dels/s) < eps;
      }
      if( i>maxit )
          nrerror( "bessik: failure to converge in cf2" );

      h *= a1;
      rkmu= sqrt( pi / (x+x)) * exp(-x)/s;
      rk1= rkmu * (xmu+x+0.5-h) * xi;
  }
  rkmup= xmu *xi * rkmu - rk1;
  rimu= xi / (f*rkmu - rkmup);
  fact= rimu / ril;
  ri[0]= ril1 * fact;
  rip[0]= rip1 * fact;
  for( i=1 ; i<=nl ; i++ )
  {
      rktemp= (xmu+i) * xi2 * rk1 + rkmu;
      rkmu = rk1;
      rk1= rktemp;
  }
  rk[0]= rkmu;
  rkp[0]= xnu*xi*rkmu - rk1;
}




void airy( double x,
           double *ai, double *bi,
           double *aip, double *bip )
{
double absx, ri, rip, rj, rjp, rk, rkp, rootx, ry, ryp, z;

  absx= fabs(x);
  rootx= sqrt(absx);
  z= absx * 2 * rootx / 3;

  if( x>0 )
  {
      bik( z, 1/3.0, &ri, &rk, &rip, &rkp );
      ai[0]= rootx*M_1_SQRT3*rk/pi;
      bi[0]= rootx * (rk/pi + 2*M_1_SQRT3*ri);
      bik( z, 2/3.0, &ri, &rk, &rip, &rkp );
      aip[0]= -x * M_1_SQRT3 * rk / pi;
      bip[0]= x * (rk/pi + 2*M_1_SQRT3*ri);
  }
  else if( x<0 )
  {
      bjy( z, 1/3.0, &rj, &ry, &rjp, &ryp );
      ai[0]= rootx * (rj-M_1_SQRT3*ry) / 2;
      bi[0]= -rootx * (ry+M_1_SQRT3*rj) /2;
      bjy( z, 2/3.0, &rj, &ry, &rjp, &ryp );
      aip[0]= absx * (M_1_SQRT3*ry+rj) / 2;
      bip[0]= absx * (M_1_SQRT3*rj-ry) /2;
  }
  else
  {
      ai[0]= 0.35502805;
      bi[0]= ai[0] / M_1_SQRT3;
      aip[0]= -0.2588194;
      bip[0]= -aip[0] / M_1_SQRT3;
  }
}
