/*****************************************************************************
 *                                  B Y
 *****************************************************************************
 *
 *   PROGRAM ID:        BY.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      These routines use the rational polynomial method of Hart to calculate
 *      the Bessel functions Yp(x). They are only valid for Bessel functions of
 *      integer order.  To calculate Yp(x) for non-integral values of p, use the
 *      defining formula:
 *
 *           Yp(x).sin(p.pi) =  Jp(x).cos(p.pi) - J_p(x)
 *
 *
 *      Unlike Jp(x), recursion is always stable for Yp(x). Therefore, we just
 *      need values for Y(x) for p=0,1   and we can recurse to any other integer
 *      value.
 *
 *      Finally, I should mention that Numerical Recipes 2nd edition has a very
 *      good routine based on continued fractions (ie  Steeds method) that
 *      simultaneously calculates J, Y, J', & Y' for a given (p,x).
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *                      When using MS C compilers, the Bessel fxns Jn(x) & Yn(x)
 *                      are built-in to the math library:
 *                           _j0, _j1, _jn, _y0, _y1, _yn
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>

#define nint(a)  ( ((a)-floor(a)<0.5)?floor(a):ceil(a) )



double by( double p, double z )
{
double pi= M_PI, rc;
double big= 1e10, tiny= 1e-10; // tiny= 1/big
int np;

  np= nint(p);
  if( fabs(p-np) > tiny )
  {
      // use defining relation
      rc= cos(p*pi)*bj(p,z) - bj(-p,z);
      rc /= sin(p*pi);
  }
  else
  {
      // use rational polynomials
      rc= byn( np, z );
  }

  return( rc );
}


double byn( int n, double x )
{
int j;
double rc, y, b0, b1, bp;

  if( n < 0 )
  {
      n= -n;
      rc= byn( n, x );
      if( n%2 )
          rc= -rc;
  }
  else if( n==0 )
      rc= by0( x );
  else if( n==1 )
      rc= by1( x );
  else
  {
      y= 2.0 / x;
      b1= by1( x );
      b0= by0( x );

      for( j=1 ; j<n ; j++ )
      {
          bp= j*y*b1 -b0;
          b0= b1;
          b1= bp;
      }

      rc= b1;
  }

  return( rc );
}






double by1( double x )
{
double rc, xx, z;
double y,p1,p2,p3,p4,p5,q1,q2,q3,q4,q5,r1,r2,r3,r4,r5,r6;
double s1,s2,s3,s4,s5,s6,s7;

  p1=1.0, p2=0.183105e-2, p3=-0.3516396496e-4, p4=0.2457520174e-5,
    p5=-0.240337019e-6;

  q1=0.04687499995e0, q2=-0.2002690873e-3, q3=0.8449199096e-5,
    q4=-0.88228987e-6, q5=0.105787412e-6;

  r1=-0.4900604943e13, r2=0.1275274390e13, r3=-0.5153438139e11,
    r4=0.7349264551e9, r5=-0.4237922726e7, r6=0.8511937935e4;

  s1=0.2499580570e14, s2=0.4244419664e12, s3=0.3733650367e10,
    s4=0.2245904002e8, s5=0.1020426050e6, s6= 0.3549632885e3, s7=1.0;


  if( x < 8.0 )
  {
      y= x*x;
      rc= x*(r1+y*(r2+y*(r3+y*(r4+y*(r5+y*r6)))))/(s1+y*(s2+y*(s3+
            y*(s4+y*(s5+y*(s6+y*s7))))))+0.636619772
            *(bj1(x)*log(x)-1./x);
  }
  else
  {
      z= 8.0 / x;
      y= z*z;
      xx= x -2.356194491;
      rc= sqrt(0.636619772/x)*(sin(xx)*(p1+y*(p2+y*(p3+y*(p4+y*p5
              ))))+z*cos(xx)*(q1+y*(q2+y*(q3+y*(q4+y*q5)))));
  }

  return( rc );
}





double by0( double x )
{
double rc, xx, z;
double y,p1,p2,p3,p4,p5,q1,q2,q3,q4,q5,r1,r2,r3,r4,r5,r6;
double s1,s2,s3,s4,s5,s6;


  p1=1.0, p2=-0.1098628627e-2, p3=0.2734510407e-4,
    p4=-0.2073370639e-5, p5=0.2093887211e-6;

  q1=-0.1562499995e-1, q2=0.1430488765e-3, q3=-0.6911147651e-5,
    q4=0.7621095161e-6, q5=-0.934945152e-7;

  r1=-2957821389.e0, r2=7062834065.e0, r3=-512359803.6e0,
    r4=10879881.29e0, r5=-86327.92757e0, r6=228.4622733e0;

  s1=40076544269.e0, s2=745249964.8e0, s3=7189466.438e0,
    s4=47447.2647e0, s5=226.1030244e0, s6=1.0;


  if( x < 8.0 )
  {
      y= x*x;
      rc= (r1+y*(r2+y*(r3+y*(r4+y*(r5+y*r6)))))/(s1+y*(s2+y*(s3
              +y*(s4+y*(s5+y*s6)))))+0.636619772*bj0(x)*log(x);
  }
  else
  {
      z= 8.0 / x;
      y= z*z;
      xx= x -0.785398164;
      rc= sqrt(0.636619772/x)*(sin(xx)*(p1+y*(p2+y*(p3+y*(p4+y*p5
             ))))+z*cos(xx)*(q1+y*(q2+y*(q3+y*(q4+y*q5)))));
  }

  return( rc );
}

