/*****************************************************************************
 *                                  G A M M A
 *****************************************************************************
 *
 *   PROGRAM ID:        GAMMA.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      These routines compute the gamma function.  This routine is one by
 *      Lanczos as reported in Numerical Recipes.  The most puzzling part of
 *      Lanczos' method (to me) was the calculation of the "magic" c[]
 *      coefficients.  Then I ran accross the following article:
 *          JL Spouge, SIAM J of Num Anal 31, (June 94), p.931
 *
 *      This article is an update to Lanczos' method, with a straightforward,
 *      explicit formula for calculating as many c[] as you need.  While
 *      Spouge's c[] aren't quite as accurate as Lanczos', you can get the same
 *      accuracy by using an extra 1 or 2 coefficients.
 *
 *      Spouge formula for c[] is:
 *          c[0]= 1
 *                 pow(-1,k-1) * exp(a-k) * pow(a-k,k-0.5)
 *          c[k]= -----------------------------------------   , where 0 < k < ceil(a)
 *                 sqrt(2*pi) * (k-1)!
 *
 *     Lanczos' formula for the gamma fxn is:
 *          N= ceil(a) -1
 *          SUM= c[0] + c[k]/(z+k), where k=1..N
 *          gamma(z+1)= pow(z+a,z+0.5) * exp(-z-a) * sqrt(2*pi) * SUM
 *          gamma(z)=   pow(z+a,z-0.5) * exp(-z-a) * sqrt(2*pi) * SUM * (1 + a/z)
 *
 *    The relative error in gamma(z+1), according to Spouge is:
 *
 *                sqrt(a) * pow(2*pi,-a-0.5)
 *          Er < ----------------------------
 *                      Real(a+z)
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>





double gammln( double xx )
{
/*
      If you have xx<1, then GAMMLN  may be complex or undefined.
      In these cases gamma uses the 'reflection formula' for Re(x)<1:

                                        pi*(1-x)
                  gamma(x)= --------------------------
                                 gamma(2-x) * sin(pi*x)
*/

int j;
double x, tmp, sum;
double c[6]= { 76.18009173e0,-86.50532033e0,24.01409822e0,
                  -1.231739516e0,0.120858003e-2,-0.536382e-5 };

//double stp= 2.50662827465e0;
double stp= M_SQRT2 / M_1_SQRTPI;
double rc= 0.0;

  x= xx - 1.0;

  if( xx > 1.0 )
  {
      tmp= x + 5.5;
      tmp -= (x+0.5) * log( tmp );
      sum= 1.0;

      for( j=0 ; j<6 ; j++ )
      {
          x += 1.0;
          sum += c[j]/x;
      }

      rc= log( stp*sum ) - tmp;
   }

   return( rc );
}





double gamma( double x )
{
double pi= M_PI;
double logDBL_MAX= DBL_MAX_EXP * M_LN2;
double logDBL_MIN= DBL_MIN_EXP * M_LN2;
double rc, spx, sgn= 1.0;

  if( x < 1.0 )
  {
      spx= sin( pi*x );
      if( spx < 0.0 )
          sgn= -1.0;

      if( sgn*spx > 1.0e-10 )
          rc= pi*(1.-x)/(spx*gamma( 2.0-x ));
      else
          rc= sgn * DBL_MAX;
  }
  else
  {
      spx= gammln(x);
      rc= DBL_MAX;
      if( spx < logDBL_MAX )
          rc= DBL_MIN;
      if( spx > logDBL_MIN )
          rc= exp( spx );
  }

  return( rc );
}


