/*****************************************************************************
 *                   C O L L I N S   L I B R A R Y   H E A D E R
 *****************************************************************************
 *
 *   PROGRAM ID:        COLLINS.H
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:       This header file is used in conjunction with the C
 *                      source in the other modules of the Collins Math
 *                      Library.
 *
 *
 *   COMPILE/LINK:      A Microsoft C 6.0 compatable compiler is needed
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



#if !defined( COLLINS_INCLUDED )
#define  COLLINS_INCLUDED   1



#include <math.h>
#include <float.h>
#include <stdio.h>
#include <stdlib.h>

/*
   You may wish to add the following macros to your <math.h>
   or just leave them here and un-comment them.

// Constants rounded for 21 decimals
#define M_E         2.71828182845904523536
#define M_LOG2E     1.44269504088896340736
#define M_LOG10E    0.434294481903251827651
#define M_LN2       0.693147180559945309417
#define M_LN10      2.30258509299404568402
#define M_PI        3.14159265358979323846
#define M_PI_2      1.57079632679489661923
#define M_PI_4      0.785398163397448309616
#define M_1_PI      0.318309886183790671538
#define M_2_PI      0.636619772367581343076
#define M_1_SQRTPI  0.564189583547756286948
#define M_2_SQRTPI  1.12837916709551257390
#define M_SQRT2     1.41421356237309504880
#define M_1_SQRT2   0.707106781186547524401
#define M_SQRT3     1.73205080756887729352
#define M_1_SQRT3   0.57735026918962576451
#define M_EULER     0.57721566490153286061
*/


//#define  max(a,b)   ((a)>(b) ? (a):(b))
//#define  min(a,b)   ((a)<(b) ? (a):(b))
#if defined( DOUBLE_PRECISION )
    #define float double
    #define G_EPS 1e-9
#else
    #define G_EPS 1e-5
#endif

#if defined( COLLINS_EXTERNS )
    extern unsigned long enStep;  // Signals progress to external application
#else
    unsigned long enStep;  // Signals progress to external application
#endif



// typedefs
#if !defined( _INC_WINDOWS )
   typedef unsigned char  BYTE;
   typedef unsigned int   WORD;
   typedef unsigned long  DWORD;
#endif
// I'd really like to replace all my WORDs with DWORDs to allow really big
// data sets to be used in the FFT & FHT algorithms....unfortunately, the
// DOS malloc functions use a parameter of "size_t" which is just a single
// WORD long.

typedef  void (*DERIV)(int n, float x, float y[], float dy_dx[]);
typedef  float (*OPTFXN)(int n, float x[]);
typedef  void (*GRADF)(int n, float x[], float F0, float g[]);
typedef  float (*FUNC1d)(float);
typedef  void (*NLFXNS)(int n, float x[], float f[]);
typedef  struct {float r,i;} COMPLEX;



/*****************************************************************************
 * MISC subdir
 *****************************************************************************/

// <nrutil.c>
/*
    Matrix and vector allocation routines from Numerical Recipes.
    I have made (what I think are) improvements to these routines.
*/
void nrerror( char *szErrorMsg );
float **matrix( WORD low1, WORD hi1, WORD low2, WORD hi2 );
double **dmatrix( WORD low1, WORD hi1, WORD low2, WORD hi2 );
WORD **imatrix( WORD low1, WORD hi1, WORD low2, WORD hi2 );
float *vector( WORD low, WORD hi );
double *dvector( WORD low, WORD hi );
WORD *ivector( WORD low, WORD hi );
BYTE *bvector( WORD low, WORD hi );
void free_matrix( float **matrix, WORD low1, WORD hi1, WORD low2, WORD hi2 );
void free_dmatrix( double **matrix, WORD low1, WORD hi1, WORD low2, WORD hi2 );
void free_imatrix( WORD **matrix, WORD low1, WORD hi1, WORD low2, WORD hi2 );
void free_vector( float *vector, WORD low, WORD hi );
void free_dvector( double *vector, WORD low, WORD hi );
void free_ivector( WORD *vector, WORD low, WORD hi );
void free_bvector( BYTE *vector, WORD low, WORD hi );

// <random.c>
/*
    rand1() is a linear congruential pseudorandom number generator
    proposed by BA Wichmann & ID Hill in BYTE, March 1987, p. 127.
    It is very well tested by the authors.  It has a period of
    6.95e12.  It uses only integer math, so it's fairly fast.
    Another interesting point, it was developed especially for
    16 bit computers, ie 2^15 = 32000 or so
*/
void seed1( float i, float j, float k );
float rand1( void );
/*
    rand2() is another LC generator. This one has a period of 2.31e18
    and was adopted by Bratley, Fox, & Schrage -- or so I hear.
    This one was developed for 32 bit computers, ie 2^31 = 2147483000
    or thereabouts.  It is 3 times faster than rand1().
*/
void seed2( float i, float j, float dummy );
float rand2( void );
/*
    The idea for rand3() came from a book by Alan Miller.
    The only difference is that he uses an exponent of 5.04, whereas
    I just use 5 -- it runs faster than using pow().  I use it inline
    to start Knuth's generator rand3()!

    It is the slowest generator due to the f.p. multiplication.
*/
void seed3( float z, float dummy1, float dummy2 );
float rand3( void );
/*
    rand4() has the best statistics of my random number generators.
    It's not an LC generator; it's a "Fibinacci" generator.  It
    requires just one subtraction to generate a number.  It does
    require the storage of a vector of 55 floats.  Not much is known
    about the theoretical properties (other than the fact that it has
    an extremely long period), but empirical results are quite good...
    at least according to DE Knuth.  Besides (24,55) other index pairs
    have been suggested in the literature, most notably (5,17).
*/
void seed4( float z, float dummy1, float dummy2 );
float rand4( void );
/*
    rand5() is called a combined Tausworthe generator.  I found it
    described in an article, ACM Transactions on Modelling and
    Computer Simulation, Vol. 1, No. 2, April 1991.  I didn't study
    the article much, except to see the claim that this is a fast
    generator with good stastical properties.
*/
void seed5( float i, float j, float dummy );
float rand5( void );
/*
    This is another LC generator.  It uses the built in wrap-around
    modulus in the UNSIGNED LONG type.  It is the fastest generator
    I've implemented to date ( about 30% faster than rand2()! ).

    This generator was in an article by WH Press & SA Teukolsky in
    Computers in Physics, Sept/Oct 1992, p. 522.
*/
void seed6( float i, float dummy1, float dummy2 );
float rand6( void );
/*
    rand7() combines the UNSIGNED LONG trick with Knuth's Fibonacci
    method.  It is the fastest generator, slightly edging out rand6().
    This time I used the pair (33,97).

    Knuth says that the period of these generators is  (2^k -1)2^m
    where k= the larger of the index pair
    and   m= #bits in the array's base data type, eg 32-bits for longs.

    In the present case this is 2^129 = 6.8e38

    Another index pair that is particularly well-known in physics
    circles is (103,250).  They need an index pair this large because
    they use a trick to speed up the generator even more!! Instead of
    integer addition, they XOR the two numbers. This is 3 or 4 times
    faster because you don't have to do the binary carries, but at the
    same time it reduces the period of the generator.

    Also, if you're out for pure speed, eliminate the square brackets
    and use pointer notation: l7[i] --> *(l7+i)
    However, I find that the pointer version of a complex expression
    is nearly unreadable.
*/
void seed7( float i, float dummy1, float dummy2 );
float rand7( void );




/*****************************************************************************
 * ODE subdir
 *****************************************************************************/

/*
   The subroutines in this subdirectory include:
               1) Runge-Kutta-Gill solvers
               2) Runge-Kutta-Fehlberg solvers
               3) Runge-Kutta-Merson solvers
               4) a Bulirsch-Stoer (modified midpoint) solver
               5) a stiff Rosenbrock (ROW) solver

   All of them will solve a system of first order ordinary
   differential equations.  The RK solvers are good, all-
   purpose routines.  The BS solver is supposed to be faster & more
   accurate, but I have found it to be neither robust nor reliable.

   The difference between the Runge-Kutta methods is the
   way that the local error is estimated.  Gill uses a full-,
   half-step technique and makes (3n-1) derivative evaluations
   per step. Fehlberg uses two methods of different orders
   which just happen to have the same intermediate vectors;
   it only uses (n+2) derivative evaluations per step.  Merson
   uses two methods of identical order which have common inter-
   mediates and uses (n+1) derivative evals per step.  These
   error estimates are critical to adaptive step-size strategies.
   The modules are named "rkNM" where N,M are the orders of the
   rk estimates used in the method. (Since the Gill methods use
   only one rk estimate, they're named "rkN").

void rkNM( int n, void (*fxn)( int, float, float *, float * ),
           float *ystart, float x1, float x2, float eps,
           float h1, float hmin, int nPts, char *szDataFile );

   ystart is the vector of dependent variables, on input it
          contains the y-values evaluated at x= x1.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h1 is the estimate of the step size
   hmin is the minimum allowed step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   The subroutine also writes nPts intermediate (x,y) values to a
   datafile called szDataFile.


****  EXTERNAL int enStep;  ******
   This value is declared in each ode solver. It is used to signal
   the calling routine about the main for-loop progress. This info
   is occassionally required by the external DERIV fxn. To access
   the variable, the calling routine declares it as:

        extern unsigned long enStep;

*/

/*
   <bsode.c>

   The following subroutine is mainly from Numerical Recipes
   as translated/modified by myself.  It is a Bulirsch-Stoer
   ode solver (thus the name BS_ode, get it?).  The driver
   is exactly the same one used in the Runge-Kutta solvers.
*/
void bsode( int n, DERIV fxn, float ystart[], float x1, float x2,
             float eps, float h, int nPts, char *szDataFile );
/*
   rk23(), is the smallest and fastest of the solvers. It
   should be used when size is the most important factor in
   the selection of the ode solver.
*/

// <rk23.c>
void rk23( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szDataFile );
// <rk4.c>
void rk4( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szDataFile );
// <rk44.c>
void rk44( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szDataFile );
// <rk45.c>
void rk45( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szDataFile );
// <euler.c> ....the old stand-by
void euler( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szFile );
/*
   <stiff.c>

   This is an ODE solver for stiff systems of equations.  It is
   mainly due to an article in Computers in Physics, May/Jun 1991,
   p.89.  The article was by two of the authors of Numerical
   Recipes....
*/
void stiff( int n, DERIV fxn, float ystart[], float x1, float x2,
             float eps, float h, int nPts, char *szDataFile );
/*
   <tainmin.c>

   And here is a stiff ODE solver from H Tain-Min, BIT (1983), p.118.
   Unlike other stiff methods, this one uses no Jacobian!! Instead
   he uses a basic trapezoidal scheme plus iteration. The article is
   rather vague about how the iteration parameters are controlled, so
   the routine is my best shot at it.

   The routine also incorporates a novel stepsize scheme that I read about
   in Simeon Ola Fatunla's Numerical Methods for Initial-Value Problems
   for Ordinary Differential Equations (Academic Press 1988). Instead of
   double-stepping, you step across the interval forwards, then backwards.
   The difference between your starting value and the value you calculate
   is just twice the local truncation error (this method only works for
   even-order methods -- like the 2nd order trapezoidal scheme).

   <tainbak.c>, <taindoub.c> are implementations that differ in their
   step-size control algorithms.
*/
void tainmin( int n, DERIV fxn, float ystart[], float x1, float x2,
              float eps, float h, int nPts, char *szFile );
/*
   <inveuler.c>

   Here's a weird method. It was in Fatunla's book. It's called the
   Inverse Euler Method, not to be confused with the Backward Euler
   Method. The algorithm (which according to Fatunla is strictly for
   scalar eqns) is:  y(+)= y + [y/(y-hf)]hf=  y*y / (y-hf).

   Fatunla says it has some interesting stability characteristics, and
   he has applied it to vector ODEs on a component-by-component basis,
   but this destroys some of the stability.

   What I have done is to vectorize the equation. I have no idea if it
   is any good or not, but we'll see....
                      y(+)= y + [ <y|y-hf>/<y-hf|y-hf> ]hf

   Sure enough, it doesn't do well on stiff problems, but it appears
   to be stable.
 */
void inveuler( int n, DERIV fxn, float ystart[], float x1, float x2,
               float eps, float h, int nPts, char *szFile );
/*
   <usmani.c>

   Next up, a method due to
     RA Usmani & RP Agarwal, Comp and Math with Appls (1985), p.1183
   as described in
     DM Kondrat & DIB Jacques, Internat J Comp Math (1992), p.117

   This method is A-stable, 3rd order, 1-step. It is actually 2-step, but
   uses an interation scheme like Tain-Min's.

   In addition, Kondrat & Jacques present a method that is A-stable, 4th
   order, 2-step (3-step with iteration).
*/
void usmani( int n, DERIV fxn, float ystart[], float x1, float x2,
             float eps, float h, int nPts, char *szFile );




/*****************************************************************************
 * FUNCS subdir
 *****************************************************************************/

// <bj.c>  Bessel function J for integer orders
double bj( double p, double z );
double bjn( int n, double x );
double bj1( double x );
double bj0( double x );

// <by.c>  Bessel function Y for integer orders
double by( double p, double z );
double byn( int n, double x );
double by0( double x );
double by1( double x );

// <bjy.c>
/*
   returns both J & Y plus their derivatives
*/
void bjy( double x, double xnu, double *rj, double *ry, double *rjp, double *ryp );
/*
   spherical bessel functions j & y plus derivatives --actually bessel
   functions of order 1/2
*/
void sphbes( int n, double x, double *sj, double *sy, double *sjp, double *syp );
           /* modified bessel functions i & k plus derivatives  */
void bik( double x, double xnu, double *ri, double *rk, double *rip, double *rkp );
           /* the airy functions ai & bi plus derivatives       */
void airy( double x, double *ai, double *bi, double *aip, double *bip );

// <erf.c>
double erfc( double x );
double erf( double x );

// <gamma.c>
double gammln( double x );
double gamma( double x );




/*****************************************************************************
 * LE subdir
 *****************************************************************************/

// <linsys.c>
/*
   Warning, ludcmp() & decomp() destroy the input matrix, A, and
   replace it with the LU decomposition.

   ludcmp() returns the determinant.
   decomp() returns the condition number.
*/
void decomp( int n, float *a[], int ipvt[], float *cond );
void ludcmp( int n, float *A[], int indx[], float *d );
/*
   Warning, lubksb() & solve() destroy b and replace it with x, the
   solution vector of A.x=b.   On input, A is LU decomposition of A.
*/
void solve(  int n, float *a[], int ipvt[], float b[]  );
void lubksb( int n, float *A[], int indx[], float b[] );
/*
   MINV() returns the matrix inverse of the input matrix A.
   The matrix A is NOT altered by the process.
*/
float **minv( int n, float *A[] );




/*****************************************************************************
 * OPTIM subdir
 *****************************************************************************/

/*
   Each function in this section that has a parameter of
   type GRADF can be passed a NULL for that parameter.  In
   which case, the function reverts to an internal finite
   differencing scheme to calculate gradient information.

   The returned value is the number of iterations. If the
   return value is less than zero, the routine failed to
   converge to the required tolerance.
*/

/*
   Line Search Algorithms

   These subroutines start at the point x and move in
   the search direction s so as to minimize the
   objective function F.  On return, x is updated to
   x+dx which minimizes F, s is set to the step taken
   (ie, dx) and F0 is set to F(x+dx).
*/

// <linmin.c>  uses Brent's method as described in Numerical Recipes
void linmin( int n, float x[], OPTFXN FXN, float s[], float *F0 );
/*
   <linmin2.c>

   linmin2() attempts to use derivative information to aid in
   the line search.

   Please note, this is NOT the routine from Numerical Recipes.
   I found this in a conference review book, and I can't recall
   the name of the publication....
*/
void linmin2( int n, float x[], OPTFXN FXN, GRADF DFXN, float s[], float *F0 );
// <linmin3.c>  linmin3() was outlined in an article by JE Dennis & RB Schnabel.
void linmin3( int n, float x[], OPTFXN FXN, float g[], float s[], float *F0 );
/*
   <linmin4.c>

   linmin4() is very simple search. It was described in an article by
     J Rohn, Computing (1993), p.373
*/
void linmin4( int n, float x[], OPTFXN FXN, float g[], float s[], float *F0 );

/*
   Optimization Algorithms

   The following routines do multi-dimensional optimization.
   The input variables are always the same:

   n is the dimension of the vectors.
   x is the solution vector.  On input it contains your
        initial guess at the solution.
   FXN(n,x) is a subroutine which calculates the value
        of the objective function for a given x.
   DFXN(n,x,F,g) is a subroutine which calculates the
       gradient vector g, given the values of x and F.
   tol is the tolerance, ie the relative error you will
       accept in the solution vector.
*/

/*
   <bandb.c>

   First is the "nearly" Conjugate Gradient method of Barzilia & Barwein.
   It is described in:  IMA Journal of Numerical Analysis (1988) p. 141
   Its strong point is that is uses no line search at all!!  What's more,
   it is very stingy on storage (2n), which means it can handle very large
   systems of equations.
*/
int bandb( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <bfgs.c>

   This is the Broyden-Fletcher-Goldfarb-Shanno quasi-newton
   optimization method.  It was discovered by all four authors
   more or less simultaneously in 1970.  It is recognized as
   the best of the multivariable optimization methods.

   It has hefty storage requirements n(n+5)
*/
int bfgs( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <cgpr.c>

   This is the the Polak-Ribeire variant of the Conjugate
   Gradient method.  It too has modest storage requirements (3n)
*/
int cgpr( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <nandm.c>

   This is the simplex method of Nelder and Mead (N and M--get it?)
   See Comp J (1965) p308 for details.

   Although its prototype suggests that it uses derivative info,
   that is merely cosmetic -- it makes all of the optimization
   routines look the same.

   It requires lots of storage n(n+3).
*/
int nandm( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <powell.c>

   This subroutine is my translation of Powell's direction set
   search from Numerical Recipes.  I feel that it has better,
   more flexible memory management than the original recipe.

   Although its prototype suggests that it uses derivative info,
   that is merely cosmetic -- it makes all of the optimization
   routines look the same.

   It too requires lots of storage n(n+3).
*/
int powell( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <qndav.c>

   This is my interpretation of WC Davidon's 1975 optimization
   algorithm.  It was revolutionary because it did away with the
   burdensome line searches of earlier quasi-newton methods.

   It requires more storage than any of the other methods n(n+12).
*/
int qndav( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <ssr1.c>

   This is Sun Lin-ping's Scaled SR1 method as described in:
        J Comp Math, Vol 12 No 4 (1994), p.380

   This article extends the ideas developed in a previous paper by
   Sun Lin-ping & M Osborne.

   As I understand it, this is basically Davidon's 1975 algorithm
   within the context of Oren's Self-Scaling update framework.

   The paper introduces a number of really neat tricks, including
   a Cholesky factorization of the update matrix, which is much
   cleaner than anything I've read before....due to the fact that
   he's working with a rank one update.

   Storage requirements are of the order n(n+7).
*/
int ssr1( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <marqsr1.c>

   The Levenberg-Marquardt-Broyden NLE solver was successful, so I
   decided to try a Levenberg-Marquardt-Optimization method.  It
   uses a unique strategy to switch between the (Quasi-)Newton and
   the Gradient direction, which requires no line searches.

   My modifications to the algorithm are as follows:
   Instead of solving  (J~.J + uI).dx = -J~.f
                 for   dx at each step,

   I picked a rational polynomial function of u that retains the
   asymptotic dx-values at u=0 and  u=infinity.

       dx= -(ug + H.f)/(1+uu)
       at u=0,   dx= -H.f ---> Newton-Raphson step
       at u=inf, dx= -g/u ---> tiny Gradient step

   My second trick is to use the Symmetric Rank 1 (SR1) method
   to update the inverse Jacobian (H.J = I):

       H= H +(dx-H.dg)(dx-H.dg)/(dx.dg-dg.H.dg)

*/
int marqsr1( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <marqdfp.c>
   Same tricks with Davidon-Fletcher-Powell
*/
int marqdfp( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <trustbfgs.c>

   My Levenberg-Marquardt-BFGS routine was never very successful.
   Instead, I tried a Trust Region method using BFGS to predict the
   Newton step.

   It interpolates between the Newton(-BFGS) direction, N, and the
   Gradient direction, G.  For an interpolation formula, I use:

       s= bN + (1-b)G,   where  b= min( 1, u/N )
                                u= trust radius
                         so that  0 < b < 1


   Then I take the actual dx to be in the direction of s, with
   length of min(u,N).

   The end result is that when the trust radius gets >= unity, the
   method starts taking pure Newton steps.  When the trust radius
   is tiny, the method takes scaled gradient steps.

*/
int trustbfgs( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <branin.c>

   The following is a homotopy method based on the soln of an ODE.
   The ODE is essentially one suggested by FH Branin in 1972:

              dx/dt =  -g


   This is basically a continuous version of Cauchy's gradient
   method.  My contribution is to add a small inertial term to
   the equation:

              dx/dt =  -g + b.xlast


   Where b < 1, is a small constant that controls how much inertia
   the equations carry from step to step.  What we then have is
   a continous version of the Conjugate Gradient method.

   The x-vector is of length 2n+1.  x[0] contains the value for b.
   x[1] thru x[n] contain the initial guess at the soln.
   x[n+1] thru x[n+n] contain the initial velocity.

   On exit the velocity is zero, and x[1] thru x[n] is the soln.
*/
int branin( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <zirilli.c>

   The following is a homotopy method based on the soln of an ODE.
   The ODE is the one suggested by Zirilli et al in 1982:

              m.d(dx/dt)/dt + b.dx/dt =  -g,   b= viscosity parameter
                                               m= inertia (mass) parameter

   Which is the analog of Newton's law of motion.  Unfortunately, this
   method is second order, which doubles the size of the vectors
   required to specify the problem.  And if matrices are required by
   your solution method, they are quadrupoled in size!  (Zirilli uses
   an implicit method that DOES make use of matrices).

   The x-vector is of length 2n+3:
   x[1] thru x[n] contain the initial guess at the soln.
   x[n+1] thru x[n+n] contain the initial velocity.
   x[2n+1] contains the viscosity parameter.
   x[2n+2] contains the inertial parameter.

   On exit the velocity is zero, and x[1] thru x[n] is the soln.
*/
int zirilli( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <homopt.c>

   The following is a continuation method. It works by replacing the
   gradient function with:
            gradHomo= grad + (u-1)grad0

   In other words, the gradient at the initial point is subtracted
   from the true gradient. This subtractive vector is attenuated as
   the method progresses, until the true soln is achieved at u=1.

   Within homopt, another optimization routine is called at each
   u-step. Right now it's bandb(), but it can be any method that uses
   its GRADF parameter [eg qndav(), but NOT nandm() or powell() !].
*/
int homopt( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );
/*
   <mukai.c>

   This is the the Conjugate Gradient method of:
         H Mukai, Math Prog (1978), p.298

   It is a model based method, ie it relies on a model rather than
   doing line searches.
*/
int mukai( int n, float x[], OPTFXN FXN, GRADF DFXN, float tol );




/*****************************************************************************
 * NLE subdir
 *****************************************************************************/

/*
   The subroutines in this section that deal with multivariable
   problems return an integer value.  This return value indicates
   the number of iterations. If the return value is less than zero,
   the routine failed to converge to the required tolerance.
*/

/*
   <ridder.c>

   My favorite routine for a single nle is from Ridders. The logic
   is easy to follow.  It is a scaled version of false position.

   x1<x2 must bracket a root on input.
*/
float ridder( FUNC1d f, float x1, float x2, float tol );
/*
   <zbrent.c>

   The following is the Van Wijngaarden-Dekker-Brent method of solving
   a single nonlinear equation as reported in Numerical Recipes.

   It is a combination of root bracketing, bisection, and inverse
   quadratic interpolation.

   x1<x2 must bracket a root on input.
*/
float zbrent( FUNC1d f, float x1, float x2, float tol );

/*
   <nleb1.c>

   This is my interpretation of Broyden's (First) nonlinear equation
   solver ( see Math Comp (1967) p.368 ).  It is to nonlinear equations
   what Fletcher-Powell's 1963 algorithm is to optimization-- the
   first, original quasi-newton method.  Unlike Fletcher-Powell, it is
   STILL (one of) the best method(s) in it's field.

   The name nleb1= Non Linear Equations, Broyden's 1st
   the name qnb1=  Quasi-Newton, Broyden's 1st
*/
int nleb1( int n, float x[], NLFXNS f, float tol );
/*
   <nleit.c>

   This is my interpretation of Ip and Todd's nonlinear equation
   solver ( see SIAM J of Num Analysis (1988) p.206 ).  It is to
   nonlinear equations what Davidon's 1975 algorithm is to optim-
    ization-- a quasi-newton method sans line searches!  It avoids
   line searches by keeping the condition of the iteration matrix
   (in some sense) optimal.

   The name nleit= Non Linear Equations, Ip and Todd
   the name qnocx= Quasi-Newton, Optimally ConveX
                   (a term from the SIAM article)
*/
int nleit( int n, float x[], NLFXNS f, float tol );
/*
   <nlecg.c>

   This is the only Conjugate Gradient I have ever seen that
   directly solves a system of NLEs.  Not only that, but it
   has no line search!!  I found it in the following article:

               AT Chronopoulos, J Comp Appl Math 40(1992) p.73
                                                [June 92]

   The name nlecg= Non Linear Equations, Conjugate Gradient
   the name cgchr=  Conjugate Gradient, Chronopoulos
*/
int nlecg( int n, float x[], NLFXNS fUser, float tol );
/*
   <nlebb.c>

   According to Barzilia & Barwein, their gradient method can
   be applied directly to NLEs. See the BANDB.C module in the
   OPTIM subdirectory.....
*/
int nlebb( int n, float x[], NLFXNS fUser, float tol );
/*
   <nleweg.c>

   Here is a unique vectorization of the Wegstein-Secant method in
   two slightly differently implementations of the same algorithm.
*/
int nleweg( int n, float x[], NLFXNS fUser, float tol );
int nleweg2( int n, float x[], NLFXNS fUser, float tol );

/*
   <nlemarq.c>

   This is my interpretation of the Levenberg-Marquardt NLE solver
   (1963).  It uses a unique strategy to switch between the Newton
   and the Gradient direction, which requires no line searches.

   My modifications to the algorithm are as follows:
   Instead of solving  (J~.J + uI).dx = -J~.f   for dx at each step,
   as the original Marqardt method, I use the following tricks:

   I picked a rational polynomial function of u that retains the
   asymptotic dx-values at u=0 and  u=infinity.

       dx= -(ug + H.f)/(1+uu)
       at u=0,   dx= -H.f ---> Newton-Raphson step
       at u=inf, dx= -g/u ---> tiny Gradient step

   My second trick is to use Broyden's method to update the
   inverse Jacobian (H.J = I):

       H= H +(dx-H.df)(dx.H)/(dx.H.df)


   The name nlemarq= Non Linear Equations, Marquardt's method
   the name qnmarq=  Quasi-Newton, Marquardt's method
*/
int nlemarq( int n, float x[], NLFXNS f, float tol );
/*
   <homo.c>

   This module contains a couple of homotopy (Newton & Fixed Point )
   NLE solvers.  Both methods are based on Ip&Todd and use a simple,
   discrete stepping algorithm.


   The name HomoNewt= Homotopy, Newton
   the name HomoFixP= Homotopy, Fixed Point
*/
int homonewt( int n, float x[], NLFXNS f, float tol );
int homofixp( int n, float x[], NLFXNS f, float tol );



/*****************************************************************************
 * SPLINES subdir
 *****************************************************************************/

/* Routines for interpolating functions & splining function values */
/*
   <fht.c>

   This subroutine computes the Fast Hartley Transform very much
   like an FFT subroutine, but it uses only real quantities since
   in place of EXP(i*w*t)= COS(w*t) +i*SIN(w*t)
   it uses     CAS(w*t)=   COS(w*t) +  SIN(w*t)

         n =  length of input vector (must be integer power of 2)
         f =  real vector to be transformed
         dir= direction of transform ( forward = +1,
                                       reverse = -1 )

   To reconstruct the function, do the following sum over 0<= k <=n:
        F(t)= ff[k].cas(2.pi.t.k/L)
*/
void fht( WORD n, float f[], int dir );


// <fft.c>  ...And of course, the obligatory Fast Fourier Transform.
void fft( int n, COMPLEX f[], int dir );




#endif

