/*****************************************************************************
 *                                  L I N S Y S
 *****************************************************************************
 *
 *   PROGRAM ID:        LINSYS.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      These routines solve linear systems of equations and are mainly my
 *      tranlation of Forsythe, Malcolm & Moler's FORTRAN routines.
 *      It is a naive translation, ie it uses FORTRAN's row-major order not C's
 *      column-major order for matrices.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


void decomp( int n, float *a[], int ipvt[], float *cond )
{
/*
     subroutine decomp(n,a,cond,ipvt,work)

     decomposes a double precision matrix by gaussian elimination
     and estimates the condition of the matrix.

     use solve to compute solutions to linear systems.

     input..

        ndim= declared row dimension of the array containing  a.

        n= order of the matrix.

        a= matrix to be triangularized.

     output..

        a  contains an upper triangular matrix  u  and a permuted
          version of a lower triangular matrix  i-l  so that
          (permutation matrix)*a= l*u .

        cond= an estimate of the condition of  a .
           for the linear system  a*x= b, changes in  a  and  b
           may cause changes  cond  times as large in  x .
           if  cond+1.0 == cond , a is singular to working
           precision.  cond  is set to  1.0d+32  if exact
           singularity is detected.

        ipvt= the pivot vector.
           ipvt[k]= the index of the k-th pivot row
           ipvt[n]= (-1)**(number of interchanges)

     work space..  the vector  work  must be declared and included
                   in the call.  its input contents are ignored.
                   its output contents are usually unimportant.

     the determinant of a can be obtained on output by
        det(a)= ipvt[n] * a[1][1] * a[2,2) * ... * a[n,n).

*/

int    nm1, i, j, k, kp1, kb, km1, m;
float ek, t, anorm, ynorm, znorm;
float dsign, *work;


  ipvt[n]= 1;
  if( n == 1 )
  {
      if( a[1][1] != 0.0 )
          cond[0]= 1.0;
      else
          cond[0]= 1e32;

      return;
  }

  work= vector( 1, n );
  nm1= n - 1;


  // compute 1-norm of a
  anorm= 0.0;
  for( j=1 ; j<=n ; j++ )
  {
      t= 0.0;
      for( i=1 ; i<=n ; i++ )
          t= t + fabs(a[i][j]);

      if( t > anorm )
          anorm= t;
  }

  // gaussian elimination with partial pivoting
  for( k=1 ; k<=nm1 ; k++ )
  {
      kp1= k+1;

      // find pivot
      m= k;
      for( i=kp1 ; i<=n ; i++ )
          if( fabs(a[i][k]) > fabs(a[m][k]) )
              m= i;

      ipvt[k]= m;
      if( m != k )
          ipvt[n]= -ipvt[n];
      t= a[m][k];
      a[m][k]= a[k][k];
      a[k][k]= t;

      // skip step if pivot is zero
      if( t == 0.0 )
          continue;

      // compute multipliers
      for( i=kp1 ; i<=n ; i++ )
          a[i][k]= -a[i][k]/t;

      // interchange and eliminate by columns
      for( j=kp1 ; j<=n ; j++ )
      {
          t= a[m][j];
          a[m][j]= a[k][j];
          a[k][j]= t;

          if( t == 0.0 )
              continue;

          for( i=kp1 ; i<=n ; i++ )
              a[i][j]= a[i][j] + a[i][k]*t;
      }
  }

//   cond= (1-norm of a)*(an estimate of 1-norm of a-inverse)
//   estimate obtained by one step of inverse iteration for the
//   small singular vector.  this involves solving two systems
//   of equations, (a-transpose)*y= e  and  a*z= y  where  e
//   is a vector of +1 or -1 chosen to cause growth in y.
//   estimate= (1-norm of z)/(1-norm of y)

//   solve (a-transpose)*y= e

  for( k=1 ; k<=n ; k++ )
  {
      t= 0.0;
      if( k != 1 )
      {
          km1= k-1;
          for( i=1 ; i<=km1 ; i++ )
              t= t + a[i][k]*work[i];
      }

      ek= 1.0;
      if( t < 0.0 )
          ek= -1.0;

      if( a[k][k] == 0.0 )
      {
         // stop, singularity!
         cond[0]= 1e32;
         free_vector( work, 1, n );
         nrerror( "singular matrix was fed to decomp()" );
         return;
      }

      work[k]= -(ek + t)/a[k][k];
  }

  for( kb=1 ; kb<=nm1 ; kb++ )
  {
      k= n - kb;
      t= 0.0;
      kp1= k+1;
      for( i=kp1 ; i<=n ; i++ )
          t= t + a[i][k]*work[i];

      work[k]= t + work[k];
      m= ipvt[k];

      if( m == k )
          continue;

      t= work[m];
      work[m]= work[k];
      work[k]= t;
  }

  ynorm= 0.0;
  for( i=1 ; i<=n ; i++ )
      ynorm= ynorm + fabs(work[i]);

  // solve a*z= y
  solve( n, a, ipvt, work );

  znorm= 0.0;
  for( i=1 ; i<=n ; i++ )
      znorm= znorm + fabs(work[i]);

  // estimate condition
  cond[0]= anorm * znorm/ynorm;
  if( cond[0] < 1.0 )
      cond[0]= 1.0;

  // done
  free_vector( work, 1, n );
  return;
}


void solve(  int n, float *a[], int ipvt[], float b[]  )
{
/*
      subroutine solve(ndim, n, a, b, ipvt)
c
c
c   solution of linear system, a*x = b .
c   do not use if decomp has detected singularity.
c
c   input..
c
c     ndim= declared row dimension of array containing a .
c
c     n= order of matrix.
c
c     a= triangularized matrix obtained from decomp .
c
c     b= right hand side vector.
c
c     ipvt= pivot vector obtained from decomp .
c
c   output..
c
c     b= solution vector, x .
c

*/
int    kb, km1, nm1, kp1, i, k, m;
float  t;

  // forward elimination
  if( n == 1 )
  {
      b[1]= b[1]/a[1][1];
      return;
  }

  nm1= n-1;
  for( k=1 ; k<=nm1 ; k++ )
  {
      kp1= k+1;
      m= ipvt[k];
      t= b[m];
      b[m]= b[k];
      b[k]= t;
      for( i=kp1 ; i<=n ; i++ )
          b[i]= b[i] + a[i][k]*t;
  }

  // back substitution
  for( kb=1 ; kb<=nm1 ; kb++ )
  {
     km1= n-kb;
     k= km1+1;
     b[k]= b[k]/a[k][k];
     t= -b[k];
     for( i=1 ; i<=km1 ; i++ )
         b[i]= b[i] + a[i][k]*t;
  }
  b[1]= b[1]/a[1][1];

  return;
}

float **minv( int n, float *A[] )
{
/*
   This subroutine returns the inverse of the matrix A.
*/

    float **A_lu, **Ai, d;
    int *indx;
    int i, j;

    A_lu= matrix( 1, n, 1, n );
    Ai= matrix( 1, n, 1, n );
    indx= ivector( 1, n );


    for ( i=1 ; i<=n ; i++ )
    {
        for ( j=1 ; j<=n ; j++ )
        {
            A_lu[i][j]= A[j][i];
            Ai[i][j]= 0.0;
        }
        Ai[i][i]= 1.0;
    }

    decomp( n, A_lu, indx, &d );

    for ( j=1 ; j<=n ; j++ )
        solve( n, A_lu, indx, Ai[j] );

    for ( i=1 ; i<n ; i++ )
    {
        for ( j=i+1 ; j<=n ; j++ )
        {
            d= Ai[i][j];
            A[i][j]= A[j][i];
            A[j][i]= d;
        }
    }

    free_matrix( A_lu, 1, n, 1, n );
    free_ivector( indx, 1, n );
    return ( Ai );
}


