/*****************************************************************************
 *                              L U D E C O M P
 *****************************************************************************
 *
 *   PROGRAM ID:        LUDECOMP.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      These routines solve linear systems of equations and are mainly from
 *      Numerical Recipes as translated/modified by myself.  It uses the LU
 *      decomposition method.
 *
 *      Another good library for linear systems are the routines described in
 *      Forsythe, Malcolm, & Moler's book.  Those also use LU decomposition.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/




/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


void ludcmp( int n, float *A[], int indx[], float *d )
{
/*
   Warning, this subroutine destroys the input matrix, A, and
   replaces it with its LU decomposition and  d=det A.
*/

    const float tiny= 1.0e-20;

    int imax, i, j, k;
    float *vv, aamax, temp, sum, dum;

    (*d)= 1.0;
    for ( i=1 ; i<=n ; i++ )
    {
        aamax=0.0;
        for ( j=1 ; j<=n ; j++ )
        {
            temp= fabs( A[i][j] );
            if ( temp>aamax )     aamax= temp;
        }
        if ( aamax==0.0 )
            nrerror( "singular matrix was fed to ludcmp()" );

        vv[i]= 1.0 / aamax;
    }

    for ( j=1 ; j<=n ; j++ )
    {
        for ( i=1 ; i<=j-1 ; i++ )
        {
            sum= A[i][j];
            for ( k=1 ; k<=i-1 ; k++ )
                sum -= A[i][k] * A[k][j];
            A[i][j]= sum;
        }

        aamax= 0.0;
        for ( i=j ; i<=n ; i++ )
        {
            sum= A[i][j];
            for ( k=1 ; k<=j-1 ; k++ )
                sum -= A[i][k] * A[k][j];
            A[i][j]= sum;
            dum= vv[i] * fabs( sum );

            if ( dum>= aamax )
            {
                imax= i;
                aamax= dum;
            }
        }

        if ( j!=imax )
        {
            for ( k=1 ; k<=n ; k++ )
            {
                dum= A[imax][k];
                A[imax][k]= A[j][k];
                A[j][k]= dum;
            }
            (*d)= -(*d);
            vv[imax]= vv[j];
        }

        indx[j]= imax;
        if ( A[j][j]==0.0 )    A[j][j]= tiny;
        if ( j!=n )
        {
            dum= 1.0 / A[j][j];
            for ( i=j+1 ; i<=n ; i++ )
                A[i][j] *= dum;
        }
    }


    for ( j=1 ; j<=n ; j++ )
        (*d) *= A[j][j];
}




void lubksb( int n, float *A_lu[], int indx[], float b[] )
{
/*
   Warning, this subroutine destroys b and replaces it with x, the
   solution vector of A.x=b    A_lu is LU decomposition of A.
*/
    float sum;
    int ii, ll, i, j;

    ii=0;
    for ( i=1 ; i<=n ; i++ )
    {
        ll= indx[i];
        sum= b[ll];
        b[ll]= b[i];

        if ( ii!=0 )
        {
            for ( j=ii ; j<=i-1 ; j++ )
                sum -= A_lu[i][j] * b[j];
        }
        else
        {
            if ( sum!=0.0 )
                ii= i;
        }

        b[i]= sum;
    }

    for ( i=n ; i>=1 ; i-- )
    {
          sum= b[i];
          if ( i<n )
          {
              for ( j=i+1 ; j<=n ; j++ )
                  sum -= A_lu[i][j] * b[j];
          }

          b[i]= sum / A_lu[i][i];
    }
}


float **minv( int n, float *A[] )
{
/*
   This subroutine returns the inverse of the matrix A.
*/

    float **A_lu, **Ai, d;
    int *indx;
    int i, j;

    A_lu= matrix( 1, n, 1, n );
    Ai= matrix( 1, n, 1, n );
    indx= ivector( 1, n );


    for ( i=1 ; i<=n ; i++ )
    {
        for ( j=1 ; j<=n ; j++ )
        {
            A_lu[i][j]= A[j][i];
            Ai[i][j]= 0.0;
        }
        Ai[i][i]= 1.0;
    }

    ludcmp( n, A_lu, indx, &d );

    for ( j=1 ; j<=n ; j++ )
        lubksb( n, A_lu, indx, Ai[j] );

    for ( i=1 ; i<n ; i++ )
    {
        for ( j=i+1 ; j<=n ; j++ )
        {
            d= Ai[i][j];
            A[i][j]= A[j][i];
            A[j][i]= d;
        }
    }

    free_matrix( A_lu, 1, n, 1, n );
    free_ivector( indx, 1, n );
    return ( Ai );
}
