/*****************************************************************************
 *                             T S T F X N S
 *****************************************************************************
 *
 *   PROGRAM ID:        TSTFXNS.c
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:       This source file is used in conjunction with the C
 *                      source files <NLETEST.C> & <OPTTEST.C>.
 *
 *                      It is a collection of "classic", non-linear systems
 *                      of equations from the literature, mainly:
 *
 *                          JJ More', BS Garbow, & KE Hillstrom,
 *                          ACM Trans. on Math. Software,
 *                          Vol. 7, No. 1, March 1981.
 *
 *
 *   COMPILE/LINK:      A Microsoft C 6.0 compatable compiler is needed
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



#define DEF_EL_TSTCNT  1
#include "tstfxns.h"

/* All these test functions are declared to conform to  */
/* the typedef NLFXNS in <collins.n>. However, some of  */
/* the test functions in this file are fixed dimension. */
/* These functions ignore the parameter n, as well as   */
/* any components of x and f beyond the intrinsic dim.  */






void  Freud_Roth(int n, float x[], float f[])
{
// This fxn is due to Freudenstein and Roth.
// It is a fixed dim fxn, n=2.
// The solns are:
//       f.f= 0             at x= (5, 4)
//       f.f= 48.9842....   at x= (11.41..., -0.8968...)
//
// The recommended starting point is:
//       x= (0.5, -2)

elTstCnt++;
   f[1]= x[1] -13 + ((5-x[2])*x[2] - 2)*x[2];
   f[2]= x[1] -29 + ((1+x[2])*x[2] -14)*x[2];
}


void  Powell2d(int n, float x[], float f[])
{
// This fxn is due to MJD Powell.
// It is a fixed dim fxn, n=2.
// The solns are:
//       f.f= 0             at x= (1.098...e-5, 9.106...)
//
// The recommended starting point is:
//       x= (0, 1)

elTstCnt++;
   f[1]= x[1]*x[2]*1e4 -1;
   f[2]= exp(-x[1]) +exp(-x[2]) -1.0001;
}


static float theta( float x1, float x2 )
{
   if( x1 > 0 )
   {
       return( 0.5*atan(x2/x1)/M_PI );
   }
   else
   {
       return( 0.5*atan(x2/x1)/M_PI +0.5 );
   }
}
void  HelicalValley(int n, float x[], float f[])
{
// I do not know who suggested this fxn.
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (1, 0, 0)
//
// The recommended starting point is:
//       x= (-1, 0, 0)

elTstCnt++;
   f[1]= 10 * ( x[3] -theta(x[1],x[2])*10 );
   f[2]= 10 * ( sqrt(x[1]*x[1]+x[2]*x[2]) -1);
   f[3]= x[3];
}


static float MyPow( float x, float y )
{
float ylx;

   if( fabs(y)<1e-30 )
       return( 1.0 );
   else if( x<1e-30 )
       return( 0.0 );
   else if( x>1e20 &&  y>1.0 )
       return( 1e37 );
   else if( x>1e20 &&  y<-1.0 )
       return( 1e-37 );
   else if( ylx=y*log(x)  &&  fabs(ylx) > 85 )
   {
       if( ylx<0.0 )
          return( 1e-37 );
       else
          return( 1e37 );
   }
   else
       return pow( x, y );

}
void  GulfRnD(int n, float x[], float f[])
{
// This fxn is due to the Research and Development Dept at Gulf Oil.
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (50, 25, 1.5)
//
// The recommended starting point is:
//       x= (5, 2.5, 0.15)


int i;
float t, y;

elTstCnt++;
   for( i=1 ; i<=3 ; i++ )
   {
      t= i*0.01;
      y= 25 +pow( -50*log(t), 2.0/3 );
      f[i]= exp( -pow( fabs(3*i*y*x[2]), x[3] )/x[1] ) -t;
   }
}


void  Box3d(int n, float x[], float f[])
{
// I do not know who suggested this fxn. (Box of Box-Keller fame?)
// It is a fixed dim fxn, n=3.
// The solns are:
//       f.f= 0             at x= (1, 10, 1)
//                     and  at x= (10, 1, -1)
//                     and  at x= (b, b, 0 ), b= arbitrary
//
// The recommended starting point is:
//       x= (0, 10, 20)

int i;
float t;

elTstCnt++;
   for( i=1 ; i<=3 ; i++ )
   {
      t= i*0.1;
      f[i]= exp(-t*x[1]) -exp(-t*x[2]) -x[3]*( exp(-t)-exp(-10*t) );
   }
}


void  Powell4d(int n, float x[], float f[])
{
// This fxn is due to MJD Powell.
// It is a fixed dim fxn, n=4.
// The solns are:
//       f.f= 0             at x= (0, 0, 0, 0)
//
// The recommended starting point is:
//       x= (3, -1, 0, 1)

float sr5= 2.2360679774997896964;

elTstCnt++;
   f[1]= x[1] +10*x[2];
   f[2]= sr5 * (x[3]-x[4]);
   f[3]= x[2] -x[3]-x[3];  f[3] *= f[3];
   f[4]= sr5*M_SQRT2 * (x[1]-x[4]) * (x[1]-x[4]);
}





void  Rosenbrock(int n, float x[], float f[])
{
// This fxn is due to HH Rosenbrock.
// n must be a multiple of 2 (ie even).
// The solns are:
//       f.f= 0             at x= (1, 1, ... , 1)
//
// The recommended starting point is:
//       x= (-1.2, 1, -1.2, 1, ... , -1.2, 1)

int i;

elTstCnt++;
   for( i=2 ; i<=n ; i+=2 )
   {
       f[i-1]= 10.0 * (x[i]-x[i-1]*x[i-1]);
       f[i]  = 1.0 -x[i-1];
   }

}


void  MJDPowell(int n, float x[], float f[])
{
// This fxn is due to MJD Powell.
// n must be a multiple of 4.
// The solns are:
//       f.f= 0             at x= (0, 0, ... , 0)
//
// The recommended starting point is:
//       x= (3, -1, 0, 1, ... , 3, -1, 0, 1)

int i;
float sr5= 2.2360679774997896964;

elTstCnt++;
   for( i=4 ; i<=n ; i+=4 )
   {
       f[i-3]= x[i-3] + 10*x[i-2];
       f[i-2]= sr5 * (x[i-1]-x[i]);
       f[i-1]= x[i-2] -x[i-1] -x[i-1];  f[i-1] *= f[i-1];
       f[i]  = sr5*M_SQRT2 * (x[i-3]-x[i]) * (x[i-3]-x[i]);
   }

}


void  TrigFxn(int n, float x[], float f[])
{
// I do not know who suggested this fxn.
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (0, 0, ... , 0)
//
// The recommended starting point is:
//       x= (1/n, 1/n, ... , 1/n)

int i, j;
float y;

elTstCnt++;
   for( y=0, j=1 ; j<=n ; j++ )
       y += cos(x[j]);

   for( i=1 ; i<=n ; i++ )
       f[i]= n -y +i*( 1-cos(x[i]) ) -sin(x[i]);
}


void  Brown(int n, float x[], float f[])
{
// I do not know who suggested this fxn. (Brown is a pretty common name.)
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (a, a, ... , a), where a satisfies:  n*a^n -(n+1)*a^(n-1) +1= 0
//                                                                        a= 1 in particular
//       f.f= 1             at x= (0, 0, ... , 0, n+1)
//
// The recommended starting point is:
//       x= (1/2, 1/2, ... , 1/2)


int i, j;
float y, z;

elTstCnt++;
   for( y=0, j=1 ; j<=n ; j++ )
       y += x[j];

   for( z= x[n], i=1 ; i<n ; i++ )
   {
       f[i]= x[i] +y -(n+1);
       z *= x[i];
   }
   f[n]= z -1;
}


void  More(int n, float x[], float f[])
{
// This fxn is due to JJ More', BS Garbow, & KE Hillstrom. It was in an
// article in ACM Trans. on Math. Software, Vol. 7, No. 1, March 1981.
// This article was the source for most of the functions in this module.
//
// There are no restrictions on n.
// The solns are:
//       f.f= 0             at x= (-1, -1, ... , -1))
//
// The recommended starting point is:
//       x= (1, 1, ... , 1)

int i, j;
float y;

elTstCnt++;
   for( y=0, j=1 ; j<=n ; j++ )
       y += x[j];

   for( i=1 ; i<=n ; i++ )
       f[i]= x[i] - 2*y/n -1;
}


