/*****************************************************************************
 *                                  H O M O
 *****************************************************************************
 *
 *   PROGRAM ID:        HOMO.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This module contains a couple of homotopy (Newton & Fixed Point )
 *      NLE solvers.  Both methods are based on Ip&Todd and use a simple,
 *      discrete stepping algorithm.
 *
 *
 *      The name HomoNewt= Homotopy, Newton
 *      the name HomoFixP= Homotopy, Fixed Point
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


// locally global variables
static NLFXNS fxn, hfxn;
static int nLoops;
static float u;   // homotopy parameter
static float *s;

// local functions and macros
#define TINY  1.0e-10
static void opt( int n, float x[], float tol );

static void hNewt( int n, float x[], float f[] );
static void hFixP( int n, float x[], float f[] );



/*
   ********* HERE ARE THE ENTRY POINTS TO THIS MODULE *************
*/
int homonewt( int n, float x[], NLFXNS f, float tol )
{
//  This subroutine sets the Newton homotopy as the function to
//  be solved.


    s= vector( 1, n );
    fxn= f;
    hfxn= hNewt;
    fxn(n,x,s);

    opt( n, x, tol );
    free_vector( s, 1, n );

    return( nLoops );
}


int homofixp( int n, float x[], NLFXNS f, float tol )
{
//  This subroutine sets the Fixed Point homotopy as the function to
//  be solved.

int i;

    s= vector( 1, n );
    fxn= f;
    hfxn= hFixP;
    for( i=1 ; i<=n ; i++ )
        s[i]= x[i];

    opt( n, x, tol );
    free_vector( s, 1, n );

    return( nLoops );
}








static void opt( int n, float x[], float tol )
{
int   i, imax, k, iRc;
float du, *xold;

/*
------------------------VARIABLE KEY--------------------------------
   n       length of independent (vector) variable
   x       independent variable-- on input it contains
           the initial guess;  on output, the soln
   s       stores initial vector needed for homotopy fxn
   u       scalar needed for homotopy fxn
  du       change in u for each step

---------------------------------------------------------------------
*/
  xold= vector( 1, n );


// Extract stepping criteria from x[0]
  imax= (int)x[0];
  if( imax < 1 )
      imax= 1;
  du= 1.0/imax;

// /****/printf( "homo: start x= (%e %e)\n", x[1], x[2] );
// /****/printf( "\t s= (%e %e)\n", s[1], s[2] );

// main loop
  for( u=du, i=1; u<1.0 && i<=imax; i++ )
  {
      for( k=1 ; k<=n ; k++ )
          xold[k]= x[k];

      // You can use any of the NLE solvers,
      // Ip&Todd seems to be the most robust
      iRc= nleit( n, x, hfxn, tol );
//      iRc= nlebb( n, x, hfxn, tol );

      if( iRc > 0 )
      {
          du *= 2;
          u += du;
      }
      else
      {
          u -= du;
          du /= 2;
          u += du;
          for( k=1 ; k<=n ; k++ )
              x[k]= xold[k];
      }
// /****/printf( "\t step=%d, u=%e, x= (%e %e)\n", i, u, x[1], x[2] );
  }

// final correction before returning
  nleit( n, x, fxn, tol );
//  nlebb( n, x, fxn, tol );

  if( i > imax )
      nLoops= -imax;
  else
      nLoops= i;

  free_vector( xold, 1, n );
}


static void hNewt( int n, float x[], float f[] )
{
int i;

  fxn(n,x,f);
  for( i=1 ; i<=n ; i++ )
  {
      f[i]= f[i] +(u-1)*s[i];
  }
// /****/printf( "hNewt: x= (%e %e)\n", x[1], x[2] );
// /****/printf( "\t     f= (%e %e)\n", f[1], f[2] );
}

static void hFixP( int n, float x[], float f[] )
{
int i;

  fxn(n,x,f);
  for( i=1 ; i<=n ; i++ )
  {
      f[i]= u*f[i] +(u-1)*(s[i]-x[i]);
  }
}

