/*****************************************************************************
 *                                  N L E B 1
 *****************************************************************************
 *
 *   PROGRAM ID:        NLEB1.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is my interpretation of Broyden's (First) nonlinear equation
 *      solver ( see Math Comp (1967) p.368 ).  It is to nonlinear equations
 *      what Fletcher-Powell's 1963 algorithm is to optimization-- the
 *      first, original quasi-newton method.  Unlike Fletcher-Powell, it is
 *      STILL (one of) the best method(s) in it's field.
 *
 *      The name nleb1= Non Linear Equations, Broyden's 1st
 *      the name qnb1=  Quasi-Newton, Broyden's 1st
 *
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


// locally global variables
static NLFXNS fxn;
static float *fSf;
static int nLoops;

// local functions and macros
#define TINY  1.0e-10
static void qnb1( int n, float x[], float tol, float f[], float df[],
                  float a[], float p[], float pH[], float Hf[], float *H[] );

static float fSquared( int n, float x[] )
/*
   This function exists so that the we can use the line minimization
   routine from the optimization library.... which requires a scalar-,
   not a vector- valued function!
*/
{
    int i;
    float ff= 0;

    fxn( n, x, fSf );
    for( i=1 ; i<=n ; i++ )
        ff += fSf[i]*fSf[i];

    return ( ff );
}






/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/

int nleb1( int n, float x[], NLFXNS f, float tol )
/*
   This subroutine is just a dummy to allocate the n(n+6)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
    float **w;
    w= matrix( 1, n+6, 1, n );
    fSf= vector( 1, n );
    fxn= f;

    if ( tol <= 0.0 )  tol= 1.0e-6;
    qnb1( n, x, tol, w[1], w[2], w[3], w[4], w[5], w[6], &w[6] );
    free_matrix( w, 1, n+6, 1, n );
    free_vector( fSf, 1, n );

    return( nLoops );
}









static void qnb1( int n, float x[], float tol, float f[], float df[],
                  float Hdf[], float dx[], float dxH[], float Hf[], float *H[] )
{
    float ff, ff0, dxHf, dxHdf, u;
    int   i, k, nMain, bRestart, bConverged;


/* ----------------------VARIABLE KEY--------------------------------
   n       length of independent (vector) variable
   x       independent variable-- on input it contains
               the initial guess;  on output, the soln
   f       stores value of previous function (fxn) call
   ff      the scalar dot product f.f
   dx      the estimated correction vector
   df      change in function values at current step
   H       "iteration matrix" --approx inverse of the Jacobian
   dxHf    the scalar dot product   dx.H.f
   dxHdf   the scalar dot product   dx.H.df
   dxH     the vector dot product   dx.H
   Hf      the vector dot product H.f
   Hdf     the vector dot product H.df

---------------------------------------------------------------------
*/


// initialize  f, ff
    fxn( n, x, f );
    ff= fSquared( n, x );
    ff0= 0.0;
    nLoops= 0;
    nMain= 0;
    bRestart= 1;
    bConverged= 0;


// main loop
    while( bConverged<2  &&  nMain< 5 )
    {
        if( bRestart )
        {
            bRestart= 0;
            nMain++;

            u= n*sqrt(ff) +1;
            for( k=1 ; k<=n ; k++ )
            {
                for( i=1 ; i<k ; i++ )
                {
                    H[i][k]= 0.0;
                    H[k][i]= 0.0;
                }
                H[k][k]= 1.0;
//                dx[k]= -f[k];
                dx[k]= -f[k]/u;
            }
        }


        for( k=1 ; k<=n ; k++ )
        {
            df[k] = -f[k];    // dump f into df
        }
        ff0= ff;




        // Do the line search; linmin internally updates x= x +dx, dx, & ff
        // The fxn call is to update f
        linmin( n, x, fSquared, dx, &ff );
        fxn( n, x, f );



// /****/ printf( "Broyden: F= %e, u= %e\n", ff, u );
// /****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );
// /****/ printf( "\t  dx= (%e %e)\n", dx[1], dx[2] );



        if( fabs(ff-ff0)<= 0.5*tol*(ff+ff0+TINY) )
            bConverged++;
        else if( fabs(ff-ff0)<= 0.5*tol*(2+ff+ff0+TINY) )
            bConverged++;
        else if( ff <= tol )
            bConverged= 2;
        else
        {
            bConverged= 0;

// calc new df
            for( i=1 ; i<=n ; i++ )
                df[i] += f[i];

// calc dot products
            dxHf= 0.;
            dxHdf= 0.;
            for( i=1 ; i<=n ; i++ )
            {
                dxH[i]= 0.;
                Hf[i]= 0.;
                Hdf[i]= 0.;
                for( k=1 ; k<=n ; k++ )
                {
                    Hdf[i] += H[i][k]*df[k];
                    dxH[i] += dx[k]*H[k][i];
                    Hf[i]  += H[i][k]*f[k];
                }
                dxHdf += dxH[i]*df[i];
                dxHf  += dxH[i]*f[i];
            }


// update H and calc new dx
            for( i=1 ; i<=n ; i++ )
            {
                for( k=1 ; k<=n ; k++ )
                    H[i][k] += (dx[i]-Hdf[i])*dxH[k]/dxHdf;
                dx[i]= -Hf[i] -(dx[i]-Hdf[i])*dxHf/dxHdf;
            }



            nLoops++;
            if( nLoops%(100*n) == 0 )
                bRestart= 1;

        } // endif-elseif-else

    } // end while(not converged)



    if( bConverged<2 )
        nLoops= -nLoops;
}
