/*****************************************************************************
 *                                  N L E I T
 *****************************************************************************
 *
 *   PROGRAM ID:        NLEIT.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is my interpretation of Ip and Todd's nonlinear equation
 *      solver ( see SIAM J of Num Analysis (1988) p.206 ).  It is to
 *      nonlinear equations what Davidon's 1975 algorithm is to optim-
 *       ization-- a quasi-newton method sans line searches!  It avoids
 *      line searches by keeping the condition of the iteration matrix
 *      (in some sense) optimal.
 *
 *      The name nleit= Non Linear Equations, Ip and Todd
 *      the name qnocx= Quasi-Newton, Optimally ConveX
 *                      (a term from the SIAM article)
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


// locally global variables
static NLFXNS fxn;
static int nLoops;

// local functions and macros
#define TINY  1.0e-10
static void qnocx( int n, float x[], float tol, float f[], float df[],
                   float q[], float s[], float qH[], float Hdf[],
                   float Hf[], float *H[] );





/*
   ********* HERE IS THE ENTRY POINT TO THIS MODULE *************
*/
int nleit( int n, float x[], NLFXNS f, float tol )
{
/*
   This subroutine is just a dummy to allocate the n(n+7)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/

    float **w;
    w= matrix( 1, n+7, 1, n );
    fxn= f;


    qnocx( n, x, tol, w[1], w[2], w[3], w[4], w[5], w[6], w[7], &w[7] );
    free_matrix( w, 1, n+7, 1, n );

    return( nLoops );
}









static void qnocx( int n, float x[], float tol, float f[], float df[],
                   float q[], float s[], float qH[], float Hdf[],
                   float Hf[], float *H[] )
{
    float ff, ff0, ss, sHdf, HdfHdf, qHdf, qHf, a;
    int   i, k, nmain, bRestart, bConverged;


/*
------------------------VARIABLE KEY--------------------------------
   n       length of independent (vector) variable
   x       independent variable-- on input it contains
           the initial guess;  on output, the soln
   f       stores value of call  to the functions, fxn
   df      change in function values at current step
   q       a vector needed in update formula for H
   s       the correction vector
   H       the iteration matrix --approx inverse of the Jacobian

   SCALAR DOT PRODUCTS
   qHdf= q.H.df               HdfHdf= H.df.H.df
   qHf = q.H.f                ss = s.s
   sHdf= s.H.df               ff = f.f

   VECTOR DOT PRODUCTS
   qH = q.H
   Hdf= H.df
   Hf = H.f

---------------------------------------------------------------------
*/

// initialize  f, ff
    fxn( n, x, f );
    ff0= ff= 0.0;
    for( k=1 ; k<=n ; k++ )
        ff= ff + f[k]*f[k];
    nLoops= 0;
    nmain= 0;
    bRestart= 1;
    bConverged= 0;


// main loop
    while( bConverged<2 && (nmain< 5) )
    {
        printf( "%5d\r", nLoops );

        ff0= ff;
        if( bRestart )
        {
            bRestart= 0;
            nmain++;
            for( k=1 ; k<=n ; k++ )
            {
                for( i=1 ; i<k ; i++ )
                {
                    H[i][k]= 0.0;
                    H[k][i]= 0.0;
                }
                H[k][k]= 1.0;
                s[k]= -f[k] / (sqrt(ff)*n);
                q[k]= 0.0;
            }
        }
// /****/ printf( "Ip&Todd: F= %e\n", ff );
// /****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );

// calc new x and dump-off f values into unused storage
        for( i=1 ; i<=n ; i++ )
        {
            x[i]= x[i] +s[i];
            qH[i]= f[i];
        }

// calc new f,ff and df vectors
        fxn( n, x, f );
        ff= 0.0;
        for( i=1 ; i<=n ; i++ )
        {
            ff= ff +f[i]*f[i];
            df[i]= f[i] -qH[i];
        }

        if( fabs(ff-ff0) <= 0.5*tol*(ff+ff0+TINY) )
            bConverged++;
        else if( fabs(ff-ff0) <= 0.5*tol*(2+ff+ff0+TINY) )
            bConverged++;
        else if( ff <= tol )
            bConverged= 2;
        else
        {
            bConverged= 0;

// dot products not involving q
            ss= 0.0;
            sHdf= 0.0;
            HdfHdf= 0.0;
            for( i=1 ; i<=n ; i++ )
            {
                Hf[i]= 0.0;
                Hdf[i]= 0.0;
                for( k=1 ; k<=n ; k++ )
                {
                    Hf[i]= Hf[i] +H[i][k]*f[k];
                    Hdf[i]=Hdf[i]+H[i][k]*df[k];
                }
                ss= ss +s[i]*s[i];
                sHdf= sHdf +s[i]*Hdf[i];
                HdfHdf= HdfHdf +Hdf[i]*Hdf[i];
            }

// new q
            a= sqrt( fabs( HdfHdf/ss ) );
            if( sHdf> 0.0 )  a= -a;
            for( i=1 ; i<=n ; i++ )
                q[i]= a*s[i] -Hdf[i];

// dot products containing q
            qHdf=0.0;
            qHf= 0.0;
            for( i=1 ; i<=n ; i++ )
            {
                qH[i]= 0.0;
                for( k=1 ; k<=n ; k++ )
                    qH[i]= qH[i] +q[k]*H[k][i];
                qHdf= qHdf +q[i]*Hdf[i];
                qHf=  qHf  +q[i]*Hf[i];
            }

// Update H and s
            for( i=1 ; i<=n ; i++ )
            {
                for( k=1 ; k<=n ; k++ )
                    H[i][k]= H[i][k] +(s[i]-Hdf[i])*qH[k]/qHdf;
                s[i]= -Hf[i] -qHf*(s[i]-Hdf[i])/qHdf;
            }

// Check for restart
            nLoops++;
            if( nLoops%(20*n*n) == 0 )   bRestart= 1;

        } // endif-elseif-else
    } // end while(not converged)


    if( bConverged < 2 )
        nLoops= -nLoops;
}

