/*****************************************************************************
 *                                  N L E M A R Q
 *****************************************************************************
 *
 *   PROGRAM ID:        NLEMARQ.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is my interpretation of the Levenberg-Marquardt NLE solver
 *      (1963).  It uses a unique strategy to switch between the Newton
 *      and the Gradient direction, which requires no line searches.
 *
 *      My modifications to the algorithm are as follows:
 *      Instead of solving  (J~.J + uI).dx = -J~.f
 *                    for   dx at each step,
 *
 *      I picked a rational polynomial function of u that retains the
 *      asymptotic dx-values at u=0 and  u=infinity.
 *
 *          dx= -(ug + H.f)/(1+uu)
 *          at u=0,   dx= -H.f ---> Newton-Raphson step
 *          at u=inf, dx= -g/u ---> tiny Gradient step
 *
 *      My second trick is to use Broyden's method to update the
 *      inverse Jacobian (H.J = I):
 *
 *          H= H +(dx-H.df)(dx.H)/(dx.H.df)
 *
 *
 *      The name nlemarq= Non Linear Equations, Marquardt's method
 *      the name qnmarq=  Quasi-Newton, Marquardt's method
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


// locally global variables
static NLFXNS fxn;
static float *fSf;
static int nLoops;

// local functions and macros
#define TINY  1.0e-10
static void qnmarq( int n, float x[], float tol, float f[], float df[], float g[],
                    float dx[], float dxH[], float Hf[], float Hdf[], float *H[] );

// This is f.f/2. It is used to calc the gradient direction.
static float F( int n, float x[] )
{
    int i;
    float ff= 0;

    fxn( n, x, fSf );

    for( i=1 ; i<=n ; i++ )
        ff += fSf[i]*fSf[i];

    return ( ff * 0.5 );
}


static float gradF( int n, float x[], float F0, float g[] )
{
    int i;
    float gabs, a, e= G_EPS;

    gabs= 0;
    for( i=1 ; i<=n ; i++ )
    {
        a= fabs( x[i] );
        if( a< 1.0 )  a= 1.0;
        a= a*e;
        x[i] += a;
        g[i]= ( F(n,x) -F0 ) / a;
        x[i] -= a;
        gabs += g[i]*g[i];
    }

    return( sqrt(gabs) );
}






/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/

int nlemarq( int n, float x[], NLFXNS f, float tol )
/*
   This subroutine is just a dummy to allocate the n(n+7)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
    float **w;
    w= matrix( 1, n+7, 1, n );
    fSf= vector( 1, n );
    fxn= f;

// /****/ printf( "START of MARQUARDT:  x= (%e %e)\n", x[1], x[2] );

    if( tol <= 0.0 )
        tol= 1.0e-6;
    qnmarq( n, x, tol, w[1], w[2], w[3], w[4], w[5], w[6], w[7], &w[7] );
    free_matrix( w, 1, n+7, 1, n );
    free_vector( fSf, 1, n );

    return( nLoops );
}









static void qnmarq( int n, float x[], float tol, float f[], float df[], float g[],
                    float dx[], float dxH[], float Hf[], float Hdf[], float *H[] )
{
    float Fo, Fn, dxHdf, u, Hfabs, gabs;
    int   i, k, nMain, bRestart, bConverged;


/* ----------------------VARIABLE KEY--------------------------------

   u       Marquardt parameter; controls step length & direction
   n       length of independent (vector) variable
   x       independent variable; on input it contains the
               initial guess; on output, the soln
   f       stores value of previous function (fxn) call
   g       gradient of objective fxn:  F= f.f /2
   Fn      the newest value of objective fxn
   Fo      the old value of objective fxn
   dx      the correction vector
   df      change in function values at current step
   H       "iteration matrix" -->approx inverse of the Jacobian
   Hf      Newton correction;  vector dot product H.f

   Hdf     the vector dot product   H.df
   dxH     the vector dot product   dx.H
   dxHdf   the scalar dot product   dx.H.df

---------------------------------------------------------------------
*/


// initialize
    nLoops= 0;
    nMain= 0;
    bRestart= 1;
    bConverged= 0;


// main loop
    while ( bConverged<2  &&  nMain< 50 )
    {
        if( bRestart )
        {
            bRestart= 0;
            nMain++;

            u= n * 1000;
            fxn( n, x, f );

            for( Fn=0, k=1 ; k<=n ; k++ )
                Fn += f[k]*f[k];
            Fn *= 0.5;
            gabs= gradF( n, x, Fn, g );

            Hfabs= 0;
            for( k=1 ; k<=n ; k++ )
            {
                for( i=1 ; i<k ; i++ )
                {
                    H[i][k]= H[k][i]= 0;
                }
                H[k][k]= 1;
                Hf[k]= f[k];
                Hfabs += Hf[k]*Hf[k];
            }
            Hfabs= sqrt( Hfabs );
        }


// calc Marquardt correction
        for( k=1 ; k<=n ; k++ )
        {
            dx[k]= -(Hfabs/gabs*u*g[k] +Hf[k]) / (u*u +1);
            x[k] += dx[k];

            // temporarily dump old f into df
            df[k]= -f[k];
        }
// /****/ printf( "Marquardt: F= %e, u= %e\n", Fn, u );
// /****/ printf( "\t  g= (%e %e)\n", g[1], g[2] );
// /****/ printf( "\t  n= (%e %e)\n", Hf[1], Hf[2] );
// /****/ printf( "\t dx= (%e %e)\n", dx[1], dx[2] );
// /****/ printf( "\t  x= (%e %e)\n", x[1], x[2] );
        Fo= Fn;



        // obtain new function values
        fxn( n, x, f );
        for( Fn= 0, k=1 ; k<=n ; k++ )
        {
            df[k] += f[k];
            Fn += f[k]*f[k];
        }
        Fn *= 0.5;


// *** Restart test ***
        nLoops++;

        if( nLoops%(20*n*n) == 0 )
        {
            bRestart= 1;
        }
///        else if( nLoops%(2*n*n) == 0   &&   u > 0.01  )
///        {
///            u= 0.01;   // try (nearly) pure Newton every 2n^2 steps
///                       // should have good approx to H by then
///        }

// *** Exit test ***
        if( 2.0*fabs(Fn-Fo) <= tol*(fabs(Fn)+fabs(Fo)+TINY) )
            bConverged++;
        if( 2.0*fabs(Fn-Fo) <= tol*(2+fabs(Fn)+fabs(Fo)+TINY) )
            bConverged++;
        else if( Fn <= tol )
            bConverged= 2;
        else if( Fn > Fo )
        {
            bConverged= 0;

            // redo last step with larger u-value
            u *= 5;   // notice that I don't increase on a bad step as much as
                      // I decrease on a good step. This ensures we drift to
                      // the Newton direction over time.
                      //
                      // In addition to the [5, 0.1] used here, I've tried
                      // other step scalings, eg [2, 0.5], that don't work
                      // as well.

            for( k=1 ; k<=n ; k++ )
            {
                x[k] -= dx[k];
                f[k] -= df[k];
                Fn= Fo;
            }
        }
        else
        {
            // shorten u-value; calc new g, H, Hf
            u *= 0.1;

            // calc dot products
            dxHdf= 0;
            for( i=1 ; i<=n ; i++ )
            {
                dxH[i]= 0;
                Hdf[i]= 0;
                for( k=1 ; k<=n ; k++ )
                {
                    dxH[i] += dx[k]*H[k][i];
                    Hdf[i] += H[i][k]*df[k];
                }
                dxHdf += dxH[i]*df[i];
            }

            // calc new g, update H and Hf
            gradF( n, x, Fn, g );

            Hfabs= 0;
            for( i=1 ; i<=n ; i++ )
            {
                Hf[i]= 0;
                for( k=1 ; k<=n ; k++ )
                {
                    H[i][k] += (dx[i]-Hdf[i]) * dxH[k] / dxHdf;
                    Hf[i] += H[i][k]*f[k];
                }
                Hfabs += Hf[i]*Hf[i];
            }
            Hfabs= sqrt( Hfabs );

        }

    } // end while(not converged)


    if( bConverged < 2 )
        nLoops= -nLoops;
}
