/*****************************************************************************
 *                                  N L E W E G
 *****************************************************************************
 *
 *   PROGRAM ID:        NLEWEG.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              November 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This is the vectorized form of the Secant-Wegstein-Newton method for
 *      scalar equations. All three of these method degenerate into the same
 *      procedure if finite differences are used instead of derivatives and
 *      if the equation to be solved is in standard Newton form:  f(x)= 0.
 *
 *      I use an interesting vectorization formula that I ran across while
 *      studying ODEs. It was mentioned in the following article:
 *          E Hairer, Num Math 35, (1980), p.57
 *
 *      I forget exactly who he credits with the following formula:
 *
 *           ab     a Re<b|c> + b Re<a|c> - c Re<a|b>
 *          ---- = ---------------------------------------
 *            c                  <c|c>
 *
 *      Notice three properties of this equation:
 *      (1) ab/c = ba/c,  the formula is symmetric
 *      (2) ab/b = a,     the multiplication & division embodied in the
 *                         formula are true inverses of one another
 *      (3) The scalar equation is an identity relation
 *
 *
 *      It was interesting to me that when I worked thru all of the algebra,
 *      I ended up with the Borwein-Barzilai formula with two additional terms:
 *
 *          x := x -f<df|dx>/<df|df> -dx<f|df>/<df|df> +df<f|dx>/<df|df>
 *
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


// locally global variables
static NLFXNS fxn;

static int nLoops;
static float *f;

// local functions and macros
#define TINY  1.0e-10
static void nle( int n, float x[], float tol, float df[], float dx[], float x_[] );




/*
   ******* HERE IS THE ENTRY POINT INTO THIS MODULE *********
*/

int nleweg( int n, float x[], NLFXNS fUser, float tol )
/*
   This subroutine is just a dummy to allocate the (3*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
{
float **w;

  w= matrix( 1, 3, 1, n );
  f= vector( 1, n );
  fxn= fUser;

  if( tol <= 0.0 )
      tol= 1.0e-6;

  nle( n, x, tol, w[1], w[2], w[3] );

  free_matrix( w, 1, 3, 1, n );
  free_vector( f, 1, n );

  return( nLoops );
}








static void nle( int n, float x[], float tol, float df[], float dx[], float x_[] )
{
int i, nLoopMax;
int bConverged;
float dfdf, dxdf, fdf, fdx, dxdx, ff, ffLast, safe;
float Cf, Cdf, Cdx;
float *x__= dx; // this is an alias for dx

// initializations
    nLoopMax= max( 4*n, 1000 );

    fxn( n, x, f );
    for( ff=0, i=1 ; i<=n ; i++ )
    {
        ff += f[i]*f[i];
        x_[i]= 0.9 * (x[i]+tol);
    }
    ffLast= ff;
    fxn( n, x_, df );
    for( ff=0, i=1 ; i<=n ; i++ )
    {
        ff += df[i]*df[i];
        df[i] -= f[i];
        dx[i]= x_[i] -x[i];
    }

    fdf= fdx= dfdf= dxdx= dxdf= 0;
    for( i=1 ; i<=n ; i++ )
    {
        fdf  += f[i]*df[i];
        fdx  += f[i]*dx[i];
        dfdf += df[i]*df[i];
        dxdx += dx[i]*dx[i];
        dxdf += dx[i]*df[i];
    }
    Cf= dxdf / dfdf;
    Cdf= fdx / dfdf;
    Cdx= fdf / dfdf;



    bConverged= 0;
    if( ff<= TINY )   bConverged= 2;



// main loop
    for( nLoops=1 ; bConverged<2 && nLoops<=nLoopMax ; nLoops++ )
    {
        printf( "%5d\r", nLoops );
        for( i=1 ; i<=n ; i++ )
        {
            x__[i]= x[i] +Cdf*df[i] -Cdx*dx[i] -Cf*f[i];
            x[i]= x_[i];
            x_[i]= x__[i];
            f[i] += df[i];
        }
        fxn( n, x_, df );
        for( i=1 ; i<=n ; i++ )
        {
            df[i] -= f[i];
            dx[i]= x_[i] -x[i];
        }

        // Calc dot products
        ffLast= ff;
        ff= fdf= fdx= dfdf= dxdx= dxdf= 0;
        for( i=1 ; i<=n ; i++ )
        {
            ff   += f[i]*f[i];
            fdf  += f[i]*df[i];
            fdx  += f[i]*dx[i];
            dfdf += df[i]*df[i];
            dxdx += dx[i]*dx[i];
            dxdf += dx[i]*df[i];
        }

        // Calc scalar coeffs for the vector update formula
        Cf= dxdf / dfdf;
        Cdf= fdx / dfdf;
        Cdx= fdf / dfdf;
        // Since values can change wildly, put safeguards on the C-coeffs
        //safe= sqrt(1+ff)*sqrt(1+dxdx)*sqrt(1+dfdf);
        //Cf /= safe;
        //Cdf /= safe;
        //Cdx /= safe;



// exit test(s)
        if( 2*fabs(ff-ffLast) <= tol*(fabs(ff)+fabs(ffLast)+TINY) )
            bConverged++;
        else if( 2*fabs(ff-ffLast) <= tol*(2+fabs(ff)+fabs(ffLast)+TINY) )
            bConverged++;
        else if( ff<= TINY )
            bConverged= 2;
        else
            bConverged= 0;


    } // end for(not converged)


  // return best x-value
  for( i=1 ; i<=n ; i++ )
      x[i]= x_[i];

  if( bConverged < 2 )
      nLoops= -nLoops;
}
