/*****************************************************************************
 *                                  B S O D E
 *****************************************************************************
 *
 *   PROGRAM ID:        BSODE.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      The following subroutine is mainly from Numerical Recipes
 *      as translated/modified by myself.  It is a Bulirsch-Stoer
 *      ode solver (thus the name BS_ode, get it?).  The driver
 *      is exactly the same one used in the Runge-Kutta solvers.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: Number of iterations required for solution
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void RZExtr( int n, int iest, float xest, float yest[],
                     float yz[], float dy[], float *D[] );

static void MMid( int n, float y[], float dy[], float xs, float htot,
                   int nstep, float yout[], float yn[], float ym[] );

static void BSStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ysav[], float dysav[],
                     float yseq[], float yerr[], float *w[] );

static void ODEBS( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h, float *w[] );

#define imax 11
#define nuse  7




/* Here's the entry point for this module */
void bsode( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (16*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
int i;
float **w;

  w= matrix( 1, 16, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
       gszDataFile[i]= szFile[i];

  ODEBS( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 16, 1, n );
}


static void ODEBS( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h, float *w[] )
{
/*
   This subroutine also drives several other ode solvers:
                1) Runge-Kutta-Gill solvers
                2) Runge-Kutta-Merson solvers
                3) Runge-Kutta-Fehlberg solvers


   All will solve a system of first order ordinary differential
   equations.  The RK solver is a good, all-purpose routine.
   The BS solver can be faster and more accurate, but is less
   robust.

   ystart is the vector of dependent variables, on input it
          contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h1 is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   which must look like f(n,x,y,dy)  where the derivatives are
   returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/

const int maxstp= 10000;
const float tiny= 1.e-30;

float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i, nstp;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          yscal[i]= fabs(y[i]) +fabs(h*dy[i]) +tiny;

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }
      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )
          h= x2-x;

      BSStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], w[4], &w[4] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
             ystart[i]= y[i];

          if( fp && gkmax )
          {
             kount= kount + 1;
             fprintf( fp, "\n%e  %e", x, y[1] );
             for( i=2 ; i<=n ; i++ )
                 fprintf( fp, "  %e", y[i] );
          }
          if( fp )
              fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Step size underflow in bsode()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Max iterations in bsode()" );
  }
}



static void BSStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ysav[], float dysav[],
                     float yseq[], float yerr[], float *w[] )
{
const float grow= 1.2;
const float shrink= 0.95;

int nseq[]= { 0,2,4,6,8,12,16,24,32,48,64,96 };

int i, j;
int bDone= 0;
float xest, temp, errmax;
float h= htry;
float xsav= x[0];


  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }

  while( !bDone )
  {
      for( i=1 ; i<=imax ; i++ )
      {
          MMid( n, ysav, dysav, xsav, h, nseq[i], yseq,
                w[1], w[2] );
          xest= h / nseq[i];
          xest *= xest;
          RZExtr( n, i, xest, yseq, y, yerr, &w[2] );

          if( i > 3 )  // guards against spurious early convergence
          {
              errmax= 0.0;
              for( j=1 ; j<=n ; j++ )
              {
                  temp= fabs( yerr[j] / yscal[j] );
                  errmax= (errmax > temp )? errmax : temp;
              }
              errmax /= eps;


              if( errmax<1.0 )
              {
                  x[0] += h;
                  hdid[0]= h;
                  if( i==nuse )
                      hnext[0]= h * shrink;
                  else if( i==nuse-1 )
                      hnext[0]= h * grow;
                  else
                      hnext[0]= ( h*nseq[nuse-1] ) / nseq[i];

                  bDone= 1;
              }
          }

      }

      // step has failed!  Flag user, reduce step size, and repeat
      if( !bDone )
      {
          h= 0.25*h/pow( 2, (imax-nuse)/2 );
          if( x[0]+h==x[0] )
              nrerror( "Step size underflow in bsode()" );
      }

  }

}



static void MMid( int n, float y[], float dy[], float xs, float htot,
                   int nstep, float yout[], float yn[], float ym[] )
{
int i, j;
float h= htot/nstep;
float x, h2, swap;

  /* first step */
  for( i=1 ; i<=n ; i++ )
  {
      ym[i]= y[i];
      yn[i]= y[i] + h*dy[i];
  }

  x= xs + h;
  f( n, x, yn, yout );

  h2= h + h;

  /* normal step */
  for( j=2 ; j<=nstep ; j++ )
  {
      for( i=1 ; i<=n ; i++ )
      {
          swap= ym[i] + h2*yout[i];
          ym[i]= yn[i];
          yn[i]= swap;
      }
      x += h;

      f( n, x, yn, yout );
  }

  /* last step */
  for( i=1 ; i<=n ; i++ )
      yout[i]= 0.5 * ( ym[i] + yn[i] + h*yout[i] );

}



static void RZExtr( int n, int iest, float xest, float yest[],
                     float yz[], float dy[], float *D[] )
{
int m1, j, k;
float yy, v, ddy, c, b, b1;
float x[imax];
float fx[nuse];

  x[iest]= xest;
  if( iest==1 )
  {
      for( j=1 ; j<=n ; j++ )
      {
          yz[j]= yest[j];
          D[j][1]= yest[j];
          dy[j]= yest[j];
      }
  }
  else
  {
      m1= (iest<nuse)? iest : nuse;
      for( k=1 ; k<=m1-1 ; k++ )
          fx[k+1]= x[iest-k] / xest;

      for( j=1 ; j<=n ; j++ )
      {
          v= D[j][1];
          D[j][1]= c= yy= yest[j];
          for( k=2 ; k<=m1 ; k++ )
          {
              b1= fx[k] * v;
              b= b1 - c;
              if( b )
              {
                  b= ( c-v )/b;
                  ddy= c*b;
                  c= b*b1;
              }
              else
                  ddy=v;

              if( k!=m1 )
                  v= D[j][k];

              D[j][k]= ddy;
              yy += ddy;
          }

          dy[j]= ddy;
          yz[j]= yy;
      }
  }

}
