/*****************************************************************************
 *                                  I N V E U L E R
 *****************************************************************************
 *
 *   PROGRAM ID:        INVEULER.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      Here's a weird method. It was in Fatunla's book. It's called the
 *      Inverse Euler Method, not to be confused with the Backward Euler
 *      Method. The algorithm (which according to Fatunla is strictly for
 *      scalar eqns) is:  y(+)= y + [y/(y-hf)]hf=  y*y / (y-hf).
 *
 *      Fatunla says it has some interesting stability characteristics, and
 *      he has applied it to vector ODEs on a component-by-component basis,
 *      but this destroys some of the stability.
 *
 *      What I have done is to vectorize the equation. I have no idea if it
 *      is any good or not, but we'll see....
 *                         y(+)= y + [ <y|y-hf>/<y-hf|y-hf> ]hf
 *
 *      Sure enough, it doesn't do well on stiff problems, but it appears
 *      to be stable.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>



/* Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE23( int n, float ystart[], float y[], float yscal[],
                    float dy[], float x1, float x2, float eps,
                    float h1, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float k0[], float kn[], float ktemp[] );




/* Here's the entry point for this module */
void inveuler( int n, DERIV fxn, float ystart[], float x1, float x2,
               float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (9*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/
      int i;
      float **w;

      w= matrix( 1, 9, 1, n );
      f= fxn;
      gkmax= nPts +1;
      for( i=0 ; szFile && szFile[i] ; i++ )
          gszDataFile[i]= szFile[i];

      xODE23( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
      free_matrix( w, 1, 9, 1, n );
}


static void xODE23( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] )
{
/*
   ystart is the vector of dependent variables, on input it
         contains the initial values.
   n is the length of y
   x1 and x2 are the beginning and end of the interval
   eps is the error control
   h is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace array

   The user must supply a subroutine to calculate the derivatives
   like f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/
      const unsigned long maxstp= 123456;
      unsigned long nstp;
      const float tiny= 1.e-30;

/****/float xlast= -1;
      float x= x1;
      float dxsav= (x2-x1)/gkmax;
      float xsav= x -dxsav -dxsav;
      float hdid, hnext;
      int kount= 0;
      int i;
      int bDone= 0;
      FILE *fp= NULL;

      //Initialize
      h= (x2>x1) ? fabs(h) : -fabs(h);
      if( gszDataFile[0] )
          fp= fopen( gszDataFile, "w" );



      for( i= 1 ; i<=n ; i++ )
          y[i]= ystart[i];

      for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
      {
         enStep= nstp;  /* signal next step to external process */
         f( n, x, y, dy);

         for( i=1 ; i<=n ; i++ )
             // yscal[i]= fabs(y[i]) +fabs(h*dy_dx[i]) +tiny;   // non-stiff scaling
             yscal[i]= max( fabs(y[i]), 1.0 );                // stiff scaling

         if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
         {
               if( kount < gkmax-1 )
               {
                    kount= kount +1;
                    fprintf( fp, "\n%e  %e", x, y[1] );
                    for( i=2 ; i<=n ; i++ )
                        fprintf( fp, "  %e", y[i] );
                    xsav= x;
               }

         }

         if( (x+h-x2)*(x+h-x1) > 0.0 )    h= x2-x;
         xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                    w[1], w[2], w[3], w[4], w[5], w[6] );

         if( (x-x2)*(x2-x1) >= 0.0 )
         {
             bDone= 1;

             for( i=1 ; i<=n ; i++ )
                 ystart[i]= y[i];

             if( fp && gkmax )
             {
                 kount= kount + 1;
                 fprintf( fp, "\n%e  %e", x, y[1] );
                 for( i=2 ; i<=n ; i++ )
                     fprintf( fp, "  %e", y[i] );
             }
             if( fp )
                fclose( fp );
         }
         else
         {
             if( fabs(hnext) < tiny )
                 nrerror( "Stepsize underflow in inveuler()" );

             h= hnext;
         }
      }



      if( !bDone )
      {
          if( fp )
              fclose( fp );
          nrerror( "Maximum iterations in inveuler()" );
      }
}



static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ytemp[], float ysav[],
                    float dysav[], float k0[], float kn[], float ytemp2[] )
{
    const float pgrow= -0.5;
    const float pshrnk= -1.0;
    const float safety= 0.95;

    int bDone= 0;
    int i;
    float yy,    yf,    ff;
    float yysav, yfsav, ffsav;
    float htemp, h= htry;
    float xtemp, xsav= x[0];
    float errmax, temp;


//    printf( "Start xrkstep with h= %e \n", h ); // *********
    for( i=1, yysav=yfsav=ffsav=0 ; i<=n ; i++ )
    {
        ysav[i]= y[i];
        dysav[i]= dy[i];

        yysav += ysav[i]*ysav[i];
        ffsav += dysav[i]*dysav[i];
        yfsav += dysav[i]*ysav[i];
    }



    while( !bDone )
    {

// Use a forward step....
        x[0]= xsav +h;
        if( x[0]==xsav )
            nrerror( "Stepsize underflow in inveuler()" );


        for( i=1 ; i<=n ; i++ )
            y[i]= ysav[i] +(yysav-h*yfsav)/(yysav-2*h*yfsav+h*h*ffsav)
                           *h*dysav[i];

//  ....and a double step.
        htemp= h/2;
        xtemp= xsav +htemp;

      // #1
        for( i=1 ; i<=n ; i++ )
            ytemp[i]= ysav[i]
                   +(yysav-htemp*yfsav)/(yysav-2*htemp*yfsav+htemp*htemp*ffsav)
                   *htemp*dysav[i];

        f( n, xtemp, ytemp, dy );


      // #2
        for( i=1, yy=yf=ff=0 ; i<=n ; i++ )
        {
            yy += y[i]*y[i];
            ff += dy[i]*dy[i];
            yf += dy[i]*y[i];
        }
        for( i=1 ; i<=n ; i++ )
            ytemp[i]= ytemp[i] + (yy-htemp*yf)/(yy-2*htemp*yf+htemp*htemp*ff)
                      *htemp*dy[i];


// We now have the estimate for the full step in y[],
// and the estimate to the double step in ytemp[].

// Error control is handled by comparing ysav to ytemp.

// Put error estimate ---> ytemp
        errmax= 0.0;
        for( i=1 ; i<=n ; i++ )
        {
            kn[i]= ytemp[i] -y[i];
            temp= fabs( kn[i] / yscal[i] );
            errmax= (errmax > temp)? errmax : temp;

            // form 2nd order approx
            y[i]= ytemp[i] +kn[i];
        }
        errmax= errmax / eps;

//printf( "  y= (%e, %e, %e) \n", y[1], y[2], y[3] ); // *********
//printf( "      err= %e \n", errmax ); // *********


// Try new step-size if we didn't meet eps
        if( errmax>1 )
            h= safety * h * pow(errmax,pshrnk);
        else
            bDone= 1;
    }

    hdid[0]= h;

    hnext[0]= safety * h * pow(errmax,pgrow);
    if( hnext[0] > 4*h )
        hnext[0]= 4*h;
    else if( hnext[0] < h/4 )
        hnext[0]= h/4;


//    printf( "Exit xrkstep with h= %e \n\n", h ); // *********
}

