/*****************************************************************************
 *                                  R K 4
 *****************************************************************************
 *
 *   PROGRAM ID:        RK4.C
 *
 *   AUTHOR:            Glynne Casteel
 *                      GlynneC@ix.netcom.com
 *
 *                      This software IS copyrighted, BUT you may use it freely
 *                      PROVIDED THAT you send me a copy of any commercial or
 *                      shareware product that incorporates this code.
 *
 *
 *   DATE:              June 11, 1994
 *
 *   DESCRIPTION:
 *
 *      This subroutine is a Runge-Kutta-Gill ODE solver of order 4.
 *
 *
 *   INPUT PARAMETERS:  None
 *
 *   RETURN/EXIT VALUE: None
 *
 *   INPUT FILES:       None
 *
 *   OUTPUT FILES:      None
 *
 *   COMPILE/LINK:      Microsoft C 6.0 compatable compiler
 *
 *
 *   SPECIAL NOTES:     None
 *
 *****************************************************************************
 *                           MODIFICATION LOG
 *
 *   DATE          NAME                DESCRIPTION
 *   ------------  ------------------  ----------------------------------
 *
 ******************************************************************************/



/*
   When the COLLINS_EXTERNS flag is turned on it puts the following
   types of definitions into the source:

    extern unsigned long enStep;

   In the application the flag isn't on so declaration occurs.
   These variables are used to communicate progress of the subroutine
   to the application.
*/
#define COLLINS_EXTERNS   1
#include <collins.h>


/*Local prototypes & globals */
static int gkmax= 1;
static char gszDataFile[40]= {0};
static DERIV f;

static void xODE4( int n, float ystart[], float y[], float yscal[],
                   float dy[], float x1, float x2, float eps,
                   float h, float *w[] );

static void xRKStep( int n, float y[], float dy[], float yscal[],
                     float *x, float htry, float eps, float *hdid,
                     float *hnext, float ytemp[], float ysav[],
                     float dysav[], float *w[] );

static void xRK( int n, float y[], float dy[], float x, float h,
                 float yout[], float yt[], float dyt[], float dym[] );

/*
    This subroutine is Gill's version of Runge-Kutta,
    but this is usually the method people are referring to
    when they talk about "the" Runge-Kutta method.  Combined
    with a half-step/full-step strategy, this method is less
    efficient than Fehlberg's, but the logic is simpler. The
    step driver is the same one used with other ode routines
    in this library.
*/


/* Here's the entry point for this module */
void  rk4( int n, DERIV fxn, float ystart[], float x1, float x2,
            float eps, float h, int nPts, char *szFile )
{
/*
   This subroutine is just a dummy to allocate the (16*n)
   dimensional workspace w which is needed by the following
   subroutines. This is a holdover from the FORTRAN roots of
   this library. In FORTRAN, you can't dynamically allocate
   an array, so the calling program must create a "work array"
   of a certain size. Then you use clever function parameter
   mapping to divvy up this array into all of the vectors and
   matrices needed in the library module.

   This subroutine also assigns user defined functions to
   (locally) global pointers, so we don't have to keep passing
   them to each lower level.
*/

int i;
float **w;

  w= matrix( 1, 16, 1, n );
  f= fxn;
  gkmax= nPts +1;
  for( i=0 ; szFile && szFile[i] ; i++ )
      gszDataFile[i]= szFile[i];

  xODE4( n, ystart, w[1], w[2], w[3], x1, x2, eps, h, &w[3] );
  free_matrix( w, 1, 16, 1, n );
}


static void xODE4( int n, float ystart[], float y[], float yscal[],
                  float dy[], float x1, float x2, float eps,
                  float h, float *w[] )
{
/*
   this subroutine also drives these other ode solvers:
               1) a Runge-Kutta-Fehlberg solver
               2) Runge-Kutta-Merson solver
               3) Bulirsch-Stoer (modified midpoint) solver

   All of them will solve a system of first order ordinary
   differential equations.  The RK solvers are good, all-
   purpose routines.  The BS solver can be faster and more
   accurate, but is less robust.

   The difference between the RK methods lies in the way
   they estimate the local truncation error.  Such estimates
   are crucial to adaptive step-sizing strategies.  Gill
   compares a full-step and two half-steps over the interval,
   a strategy that requires (3n+1) derivative evaluations at
   each step.  Fehlberg compares two methods of different
   orders.  By using common intermediates, this strategy
   uses only (n+2) derivative evals.  Merson uses two methods
   of the same order with common intermediates, and makes
   (n+1) derivative evaluations per step.

   ystart is the vector of dependent variables, on input it
         contains the y-values at x=x1; on output it contains
         the y-values propagated to x=x2.
   n is the length of y
   x1 and x2 are the beginning and end of the interval of
         integration
   eps is the error control
   h1 is the estimate of the step size
   f is the user supplied subroutine to calculate derivatives
   w is a dummy workspace matrix

   The user must supply a subroutine to calculate the derivatives
   f(n,x,y,dy)  where the derivatives are returned in dy.

   the subroutine also writes nDataPts (x,y) values to a data
   file called gszDataFile.
*/

const int maxstp= 10000;
const float tiny= 1.0e-30;

float x= x1;
float dxsav= (x2-x1)/gkmax;
float xsav= x -dxsav -dxsav;
float hdid, hnext;
int kount= 0;
int i, nstp;
int bDone= 0;
FILE *fp= NULL;

  //Initialize
  h= (x2>x1) ? fabs(h) : -fabs(h);
  if( gszDataFile[0] )
      fp= fopen( gszDataFile, "w" );



  for( i= 1 ; i<=n ; i++ )
      y[i]= ystart[i];

  for( nstp=1 ; nstp<=maxstp && !bDone ; nstp++ )
  {
      enStep= nstp;  /* signal next step to external process */
      f( n, x, y, dy);

      for( i=1 ; i<=n ; i++ )
          yscal[i]= fabs(y[i]) +fabs(h*dy[i]) +tiny;

      if( fp  &&  fabs(x-xsav) > fabs(dxsav) )
      {
          if( kount < gkmax-1 )
          {
              kount= kount +1;
              fprintf( fp, "\n%e  %e", x, y[1] );
              for( i=2 ; i<=n ; i++ )
                  fprintf( fp, "  %e", y[i] );
              xsav= x;
          }

      }

      if( (x+h-x2)*(x+h-x1) > 0.0 )    h= x2-x;

      xRKStep( n, y, dy, yscal, &x, h, eps, &hdid, &hnext,
                w[1], w[2], w[3], &w[3] );

      if( (x-x2)*(x2-x1) >= 0.0 )
      {
          bDone= 1;

          for( i=1 ; i<=n ; i++ )
             ystart[i]= y[i];

          if( fp && gkmax )
          {
             kount= kount + 1;
             fprintf( fp, "\n%e  %e", x, y[1] );
             for( i=2 ; i<=n ; i++ )
                 fprintf( fp, "  %e", y[i] );
          }
          if( fp )
             fclose( fp );
      }
      else
      {
          if( fabs(hnext) < tiny )
              nrerror( "Stepsize underflow in RK4()" );

          h= hnext;
      }
  }



  if( !bDone )
  {
      if( fp )
          fclose( fp );
      nrerror( "Maximum iterations in RK4()" );
  }
}






static void xRKStep( int n, float y[], float dy[], float yscal[],
                    float *x, float htry, float eps, float *hdid,
                    float *hnext, float ytemp[], float ysav[],
                    float dysav[], float *w[] )
{
const float pgrow= -0.2;
const float pshrnk= -0.25;
const float safety= 0.9;
const float errcon= 5.8e-4;

int bDone= 0;
int i;
float fcor= 1.0/15.0;
float h= htry;
float xsav= x[0];
float errmax;


  for( i=1 ; i<=n ; i++ )
  {
      ysav[i]= y[i];
      dysav[i]= dy[i];
  }


  while ( !bDone )
  {
      float temp;
      float hh= h /2.0;

      // Take two half steps
      xRK( n, ysav, dysav, xsav, hh, ytemp, w[1], w[2], w[3] );
      x[0]= xsav +hh;
      f( n, x[0], ytemp, dy);
      xRK( n, ytemp, dy, x[0], hh, y, w[1], w[2], w[3] );
      x[0]= xsav +h;

      if( x[0]==xsav )
          nrerror( "Stepsize underflow in RK4()" );

      // Take the full step
      xRK( n, ysav, dysav, xsav, h, ytemp, w[1], w[2], w[3] );


      errmax= 0.0;
      for( i=1 ; i<=n ; i++ )
      {
          ytemp[i]= y[i] -ytemp[i];
          temp= fabs( ytemp[i] / yscal[i] );
          errmax= (errmax > temp)? errmax : temp;
      }
      errmax= errmax / eps;

      // Try new step-size if we didn't meet eps
      if( errmax>1 )
          h= safety * h * pow(errmax,pshrnk);
      else
          bDone= 1;
  }

  hdid[0]= h;

  if( errmax > errcon )
      hnext[0]= safety * h * pow(errmax,pgrow);
  else
      hnext[0]= 4.0 * h;

  // Make correction to get fifth order accuracy
  for( i=1 ; i<=n ; i++ )
      y[i]= y[i] + fcor*ytemp[i];
}



static void xRK( int n, float y[], float dy[], float x, float h,
                float yout[], float yt[], float dyt[], float dym[] )
{
int i;
float hh= h/2.0;
float h6= h/6.0;
float xh= x +hh;


  for( i=1 ; i<=n ; i++ )
      yt[i]= y[i] + hh*dy[i];

  f( n, xh, yt, dyt );

  for( i=1 ; i<=n ; i++ )
      yt[i]= y[i] + hh*dyt[i];

  f( n, xh, yt, dym );

  for( i=1 ; i<=n ; i++ )
  {
      yt[i]= y[i] + h*dym[i];
      dym[i]= dyt[i] + dym[i];
  }

  f( n, x+h, yt, dyt );

  for( i=1 ; i<=n ; i++ )
      yout[i]= y[i]+ h6 * (dy[i]+dyt[i]+2*dym[i]);
}

